// ============================================================================
//  $Id: TSoftwareTimerModule.cc,v 1.3 2002/12/08 13:27:53 iwai Exp $
//  $Name:  $
// ============================================================================
#include "TSoftwareTimerModule.hh"
#include "TDataSegment.hh"
#include "TDataElement.hh"

TSoftwareTimerModule::TSoftwareTimerModule( const Tstring& unit, Tint nchannel )
  : TSoftwareModule( nchannel ), 
    theIntervalTime( 0.0 ), theCurrentTime( 0.0 ), thePreviousTime( 0.0 )
{
  static const Tstring head = "TSoftwareTimerModule::TSoftwareTimerModule: ";
  if ( unit == Tsec || unit == Tmsec || unit == Tusec ) {
    theUnit = unit;
  } else {
    Tcerr << head << "invalid unit" << Tendl;
    Tcerr << head << "[" << Tmsec << "]" << " use" << Tendl;
    theUnit = Tmsec;
  }

  if ( nchannel > tNumberOfChannels ) {
    Tcerr << head << "invalid channels" << Tendl;
    Tcerr << head << "revised " << tNumberOfChannels << Tendl;
    theNumberOfChannels = tNumberOfChannels;
  }
  setCurrentTime();
  Update();
}

TSoftwareTimerModule::TSoftwareTimerModule( const TSoftwareTimerModule& right )
  : TSoftwareModule( right ), 
    theIntervalTime( right.theIntervalTime ),
    theCurrentTime( right.theCurrentTime ),
    thePreviousTime( right.thePreviousTime ),
    theUnit( right.theUnit )
{;}

TSoftwareTimerModule::~TSoftwareTimerModule()
{;}

Tvoid TSoftwareTimerModule::FillData( const TDataSegment& segment )
{
  Update();
  static const Tstring id[ tNumberOfChannels ] =
  {
    "Interval",
    "Current",
    "Previous",
    "Unit"
  };

  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    if ( i == tIntervalTime ) {
      TDataElement element( &theIntervalTime, tTypeDouble, id[ i ], 1 );
      ((TDataElementList*)(&segment)) -> push_back( element );
    } else if ( i == tCurrentTime ) {
      TDataElement element( &theCurrentTime, tTypeDouble, id[ i ], 1 );
      ((TDataElementList*)(&segment)) -> push_back( element );
    } else if ( i == tPreviousTime ) {
      TDataElement element( &thePreviousTime, tTypeDouble, id[ i ], 1 );
      ((TDataElementList*)(&segment)) -> push_back( element );
    } else if ( i == tUnit ) {
      TDataElement element( &theUnit, tTypeString, id[ i ], 1 );
      ((TDataElementList*)(&segment)) -> push_back( element );
    } else {
      theStatus = -EFAULT;
      TDataElement element( &theStatus, tTypeInt, itostr( i ), 1 );
      ((TDataElementList*)(&segment)) -> push_back( element );
    }
  }
  return;
}

Tvoid TSoftwareTimerModule::FillData( const TDataElement& element )
{
  Update();
  static const Tstring id[ tNumberOfChannels ] =
  {
    "Interval",
    "Current",
    "Previous",
    "Unit"
  };

  Tint ch = atoi( element.GetID().c_str() );

  if ( ch < 0 || ch >= theNumberOfChannels ) {
    Tcerr << "TSoftwareScalerModule::FillData: invalid ID" << Tendl;
    theStatus = -EFAULT;
    ((TDataElement*)(&element)) -> FillData( &theStatus, tTypeInt, 1 );
  } else if ( ch == tIntervalTime ) {
    ((TDataElement*)(&element)) -> FillData( &theIntervalTime, tTypeDouble, 1 );
    ((TDataElement*)(&element)) -> SetID( id[ ch ] );
  } else if ( ch == tCurrentTime ) {
    ((TDataElement*)(&element)) -> FillData( &theCurrentTime, tTypeDouble );
    ((TDataElement*)(&element)) -> SetID( id[ ch ] );
  } else if ( ch == tPreviousTime ) {
    ((TDataElement*)(&element)) -> FillData( &thePreviousTime, tTypeDouble );
    ((TDataElement*)(&element)) -> SetID( id[ ch ] );
  } else if ( ch == tUnit ) {
    ((TDataElement*)(&element)) -> FillData( &theUnit, tTypeString );
    ((TDataElement*)(&element)) -> SetID( id[ ch ] );
  } else {
    Tcerr << "TSoftwareScalerModule::FillData: invalid ID" << Tendl;
    theStatus = -EFAULT;
    ((TDataElement*)(&element)) -> FillData( &theStatus, tTypeInt, 1 );
  }

  return;
}

const TSoftwareTimerModule& TSoftwareTimerModule::operator=( const TSoftwareTimerModule& right )
{
  *( (TSoftwareModule*)this ) = *( (TSoftwareModule*)(&right) );
  theIntervalTime = right.theIntervalTime;
  theCurrentTime = right.theCurrentTime;
  thePreviousTime = right.thePreviousTime;
  theUnit = right.theUnit;
  return( *this );
}

Tbool TSoftwareTimerModule::operator==( const TSoftwareTimerModule& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *( (TSoftwareModule*)this ) == *( (TSoftwareModule*)(&right) ) );
  ret &= ( theUnit == right.theUnit );
  return( ret );
}

Tbool TSoftwareTimerModule::operator!=( const TSoftwareTimerModule& right ) const
{
  Tbool ret = Tfalse;
  ret |= ( *( (TSoftwareModule*)this ) != *( (TSoftwareModule*)(&right) ) );
  ret |= ( theUnit != right.theUnit );
  return( ret );
}

Tvoid TSoftwareTimerModule::setCurrentTime()
{
  thePreviousTime = theCurrentTime;

  struct timeval now;
  gettimeofday( &now, 0 );

  Tdouble tv_sec = (Tdouble)( now.tv_sec );
  Tdouble tv_usec = (Tdouble)( now.tv_usec );

  Tdouble usec = ( tv_sec * 1.0e+6 ) + tv_usec;

  if ( theUnit == Tsec )
    theCurrentTime = usec * 1.0e-6;
  else if ( theUnit == Tmsec )
    theCurrentTime = usec * 1.0e-3;
  else if ( theUnit == Tusec )
    theCurrentTime = usec;

  theIntervalTime = theCurrentTime - thePreviousTime;
  if ( theIntervalTime < 0 )
    theIntervalTime = (Tdouble)( -EFAULT );

  return;
}
