// ============================================================================
//  $Id: TTerminalUserInterface.cc,v 1.1.1.1 2003/01/04 08:07:14 iwai Exp $
//  $Name:  $
// ============================================================================
#include "TTerminalUserInterface.hh"
#include "TRunManager.hh"
#include "TCommand.hh"

static const Tchar TasciiCtrlA = '\001';
static const Tchar TasciiCtrlB = '\002';
static const Tchar TasciiCtrlD = '\004';
static const Tchar TasciiCtrlE = '\005';
static const Tchar TasciiCtrlF = '\006';
static const Tchar TasciiCtrlK = '\013';
static const Tchar TasciiCtrlL = '\014';
static const Tchar TasciiCtrlN = '\016';
static const Tchar TasciiCtrlP = '\020';
static const Tchar TasciiCtrlY = '\031';
static const Tchar TasciiCtrlZ = '\032';
static const Tchar TasciiTAB = '\011';
static const Tchar TasciiBS = '\010';
static const Tchar TasciiDEL = '\177';
static const Tchar TasciiESC = '\033';

TTerminalUserInterface::TTerminalUserInterface( const Tstring& prompt, const Tstring& history )
  : TUserInterface( history ), theCommandHistoryIndex( 0 ),
    theCursorPosition( 0 ), thePrompt( prompt ), theAcceptString(),
    theStringBuffer(), theCommandBuffer()
{
  tcgetattr( 0, &theTerminal );
  setupterm( 0, 1, (int*)0 );
}

TTerminalUserInterface::~TTerminalUserInterface()
{;}

Tvoid TTerminalUserInterface::initializeCommandLine()
{
  theCommandHistoryIndex = (Tint)theCommandHistory.size();
  theCursorPosition = 0;
  theAcceptString.erase();
  theCommandBuffer.erase();
  Tcout << thePrompt << Tflush;
  return;
}

const Tstring& TTerminalUserInterface::readLine()
{
  setTerminalInputMode();
  initializeCommandLine();

  Tchar cc;

  do {
    Tcin.get( cc );

    switch ( cc ) {

      case TasciiCtrlA:
	moveCursorTop();
	break;

      case TasciiCtrlB:
	backwardCursor();
	break;

      case TasciiCtrlD:
	deleteCharacter();
	break;

      case TasciiCtrlE:
	moveCursorEnd();
	break;

      case TasciiCtrlF:
	forwardCursor();
	break;

      case TasciiCtrlK:
	cutCharacter();
	break;

      case TasciiCtrlL:
	clearScreen();
	break;

      case TasciiCtrlN:
	nextCommand();
	break;

      case TasciiCtrlP:
	previousCommand();
	break;

      case TasciiCtrlY:
	pasteCharacter();
	break;

      case TasciiCtrlZ:
	suspendTerminal();
	break;

      case TasciiTAB:
	if ( isCursorTop() && theAcceptString.empty() ) {
	  Tcout << Tendl;
	  ExecuteCommand( "ls" );
	  Tcout << Tflush;
	  return( theAcceptString );
	} else if ( isCursorEnd() ) {
	  completeCommand();
	}
	break;

      case TasciiBS:
	backspaceCharacter();
	break;

      case TasciiDEL:
	deleteCharacter();
	break;

      default:
	break;
    }

    if ( cc == TasciiESC ) {
      Tcin.get( cc );
      if ( cc == '[' ) {
	Tcin.get( cc );
	switch ( cc ) {
	  case 'A':  // up
	    cc = 'P' - '@';
	    previousCommand();
	    break;
	  case 'B':  // down
	    cc = 'N' - '@';
	    nextCommand();
	    break;
	  case 'C':  // right
	    cc = 'F' - '@';
	    forwardCursor();
	    break;
	  case 'D':  // left
	    cc = 'B' - '@';
	    backwardCursor();
	    break;
	  default:
	    cc = 0;
	    break;
	}
      }
    }
    
    insertCharacter( cc );

  } while ( cc != Teol );

  
  if ( theAcceptString.size() > 0 ) {

    // ĤΥڡ
    while ( !( theAcceptString.empty() ) && ( theAcceptString.rfind( Tspace ) == theAcceptString.size() - 1 ) )
      theAcceptString.erase( theAcceptString.end() - 1 );

    // erase space character
    while ( !( theAcceptString.empty() ) && ( theAcceptString.find( Tspace ) == 0 ) ) {
      theAcceptString.erase( theAcceptString.begin() );
    }

//     static const Tstring format = "%Y %a %b %d %H:%M:%S";
//     static const Tsize_t max = 256;
//     Ttime_t time_now;
//     time( &time_now );
//     struct tm* tm_now = localtime( &time_now );
//     Tchar* s = new Tchar[ max ];
//     strftime( s, max, format.c_str(), tm_now );
//     Tstring timestr = s;
//     delete [] s;
//     if ( !( theAcceptString.empty() ) ) {
//       theHistoryFileStream << Twslash << Tspace;
//       theHistoryFileStream << theCommandHistory.size() << Ttab;
//       theHistoryFileStream << timestr << Tendl;
//       theHistoryFileStream << theAcceptString << Tendl;
//       theCommandHistory.push_back( theAcceptString );
//     }

  }
  Tcout << Tendl;

  resetTerminal();

  return( theAcceptString );
}

Tvoid TTerminalUserInterface::insertCharacter( Tchar cc )
{
  if ( ! isprint( cc ) )
    return;

  Tcout << cc;
  for ( Tint i = theCursorPosition; i < (Tint)theAcceptString.size(); i ++ )
    Tcout << theAcceptString[ i ];
  for ( Tint i = theCursorPosition; i < (Tint)theAcceptString.size(); i ++ )
    Tcout << TasciiBS;
  Tcout << Tflush;
    
  if ( isCursorEnd() )
    theAcceptString += cc;
  else
    theAcceptString.insert( theAcceptString.begin() + theCursorPosition, cc );

  theCursorPosition ++;
  return;
}

Tvoid TTerminalUserInterface::backspaceCharacter()
{
  if ( isCursorTop() )
    return;

  if ( isCursorEnd() ) {
    Tcout << TasciiBS << Tspace << TasciiBS;
  } else {
    Tcout << TasciiBS;
    for ( Tint i = theCursorPosition; i < (Tint)theAcceptString.size(); i ++ )
      Tcout << theAcceptString[ i ];
    Tcout << Tspace;
    for ( Tint i = theCursorPosition; i < (Tint)( theAcceptString.size() + 1 ); i ++ )
      Tcout << TasciiBS;
  }
  Tcout << Tflush;
  theAcceptString.erase( theCursorPosition - 1, 1 );
  theCursorPosition --;
  return;
}

Tvoid TTerminalUserInterface::deleteCharacter()
{
  forwardCursor();
  backspaceCharacter();
  return;
}

Tvoid TTerminalUserInterface::clearLine()
{
  moveCursorTop();
  clearAfterCursor();
  return;
}

Tvoid TTerminalUserInterface::clearScreen()
{
  //endwin();
  //refresh();
  //move( 0, 0 );
  //initializeCommandLine();
  return;
}

Tvoid TTerminalUserInterface::clearAfterCursor()
{
  if ( isCursorEnd() )
    return;
  for ( Tint i = theCursorPosition; i < (Tint)theAcceptString.size(); i ++ )
    Tcout << Tspace;
  for ( Tint i = (Tint)theAcceptString.size(); i > theCursorPosition; i -- )
    Tcout << TasciiBS;
  Tcout << Tflush;
  theAcceptString.erase( theCursorPosition, theAcceptString.size() - theCursorPosition );
  return;
}

Tvoid TTerminalUserInterface::forwardCursor()
{
  if ( isCursorEnd() )
    return;
  Tcout << theAcceptString[ (Tsize_t)( theCursorPosition ) ] << Tflush;
  theCursorPosition ++;
  return;
}

Tvoid TTerminalUserInterface::backwardCursor()
{
  if ( isCursorTop() )
    return;
  Tcout << TasciiBS << Tflush;
  theCursorPosition --;
  return;
}

Tvoid TTerminalUserInterface::moveCursorTop()
{
  while ( isCursorTop() == Tfalse )
    backwardCursor();
  return;
}

Tvoid TTerminalUserInterface::moveCursorEnd()
{
  while ( isCursorEnd() == Tfalse )
    forwardCursor();
  return;
}

Tvoid TTerminalUserInterface::cutCharacter()
{
  if ( !( theStringBuffer.empty() ) )
    theStringBuffer.erase();
  for ( Tint i = theCursorPosition; i < (Tint)theAcceptString.size(); i ++ )
    theStringBuffer += theAcceptString[ i ];
  Tint nback = (Tint)( theAcceptString.size() ) - theCursorPosition;
  moveCursorEnd();
  for ( Tint i = 0; i < nback; i ++ )
    backspaceCharacter();
  return;
}

Tvoid TTerminalUserInterface::pasteCharacter()
{
  if ( theStringBuffer.empty() )
    return;
  for ( int i = 0; i < (Tint)theStringBuffer.size(); i ++ )
    insertCharacter( theStringBuffer[ i ] );
  return;
}

Tvoid TTerminalUserInterface::nextCommand()
{
  if ( theCommandHistoryIndex > (Tint)theCommandHistory.size() - 1 )
    return;
  clearLine();
  Tstring com;
  if ( theCommandHistoryIndex == (Tint)theCommandHistory.size() - 1 )
    com = theCommandBuffer;
  else
    com = theCommandHistory[ theCommandHistoryIndex + 1 ];

  for ( int i = 0; i < (Tint)com.size(); i ++ )
    insertCharacter( com[ i ] );
  theCommandHistoryIndex ++;
  return;
}

Tvoid TTerminalUserInterface::previousCommand()
{
  if ( theCommandHistory.empty() || theCommandHistoryIndex == 0 )
    return;

  if ( theCommandHistoryIndex == (Tint)theCommandHistory.size() )
    theCommandBuffer = theAcceptString;

  clearLine();

  Tstring com = theCommandHistory[ theCommandHistoryIndex - 1 ];
  for ( int i = 0; i < (Tint)com.size(); i ++ )
    insertCharacter( com[ i ] );
  theCommandHistoryIndex --;
  return;
}

Tvoid TTerminalUserInterface::completeCommand()
{
  Tstring strbuf = theAcceptString;
  TstringList candidate;

  TCommand* com = 0;
  SetCommandIndex( 0 );
  while ( ( com = NextCommand() ) ) {
    Tstring name = com -> GetCommandName();
    if ( name.find( strbuf ) == 0 )
      candidate.push_back( name );
  }


  if ( !( candidate.empty() ) ) {
    if ( candidate.size() == 1 ) {
      clearLine();
      theAcceptString = candidate[ 0 ];
      theAcceptString += Tspace;
      theCursorPosition = theAcceptString.size();
      Tcout << theAcceptString;
    } else if ( candidate.size() >= 2 ) {

      Tcout << Tendl;
      for ( Tint i = 0; i < (Tint)candidate.size(); i ++ ) {
	Tcout << candidate[ i ];
	if ( i != (Tint)( candidate.size() - 1 ) )
	  Tcout << Twspace;
      }
      Tcout << Tendl;

      // Ƥθ䤬פƤʬõ
      Tsize_t pos = strbuf.size() - 1;
      Tbool coincidence = Ttrue;
      while ( coincidence == Ttrue ) {
	pos ++;
	for ( Tsize_t i = 0; i < ( candidate.size() - 1 ); i ++ )
	  coincidence &= ( ( candidate[ i ] )[ pos ] == ( candidate[ i + 1 ] )[ pos ] );
      }

      theAcceptString = candidate[ 0 ];
      theAcceptString.erase( pos, ( theAcceptString.size() - pos ) );
      theCursorPosition = theAcceptString.size();
      Tcout << thePrompt << theAcceptString;
    }
  }

  Tcout << Tflush;

  return;
}

Tvoid TTerminalUserInterface::suspendTerminal() const
{
  pid_t pid = getpid();
  if ( fork() == 0 )
    exit( kill( pid, SIGSTOP ) );
  else
    wait( 0 );
  return;
}

Tvoid TTerminalUserInterface::setTerminalInputMode()
{
  termios tiosbuf = theTerminal;
  tiosbuf.c_iflag |= IGNBRK;
  tiosbuf.c_iflag |= IGNPAR;
  tiosbuf.c_lflag &= ~ICANON;
  tiosbuf.c_lflag &= ~ECHO;
  tiosbuf.c_lflag &= ~ISIG;
  tiosbuf.c_cc[ VMIN ] = 1;
  tiosbuf.c_cc[ VTIME ] = 0;
  tcsetattr( 0, TCSANOW, &tiosbuf );

  return;
}

Tvoid TTerminalUserInterface::resetTerminal()
{
  tcsetattr( 0, TCSANOW, &theTerminal );
  return;
}

Tbool TTerminalUserInterface::AcceptCommand()
{
  return( ( theAcceptString.c_str() != 0 ) ? Ttrue : Tfalse );
}

const Tstring& TTerminalUserInterface::GetInputCommand()
{
  return( readLine() );
}

Tvoid TTerminalUserInterface::NotFoundCommand( const Tstring& commandname ) const
{
  Tcerr << commandname << ": Command not found." << Tendl;
  return;
}
