/* ------------------------------------------------------------------------- */
/*
 *  lexical_cast.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *  
 *  Distributed under the Boost Software License, Version 1.0. (See
 *  accompanying file LICENSE_1_0.txt or copy at
 *  http://www.boost.org/LICENSE_1_0.txt)
 *
 *  Last-modified: Fri 11 Jul 2008 08:38:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_LEXICAL_CAST_H
#define CLX_LEXICAL_CAST_H

#include <sstream>
#include <string>
#include <iomanip>
#include <iostream>

namespace clx {
	/* --------------------------------------------------------------------- */
	//  bad_lexical_cast
	/* --------------------------------------------------------------------- */
	class bad_lexical_cast : public std::bad_cast {
	public:
		bad_lexical_cast() : source_(&typeid(void)), target_(&typeid(void)) {}
		explicit bad_lexical_cast(const std::type_info& source, const std::type_info& target) :
			source_(&source), target_(&target) {}
		virtual ~bad_lexical_cast() throw() {}
		
		const std::type_info& source_type() const { return *source_; }
		const std::type_info& target_type() const { return *target_; }
		
		virtual const char* what() const throw() {
			return "bad lexical cast";
		}
		
	private:
		const std::type_info* source_;
		const std::type_info* target_;
	};
	
	/* --------------------------------------------------------------------- */
	//  cast_stream
	/* --------------------------------------------------------------------- */
	template <
		class Type,
		class Source
	>
	class cast_stream {
	public:
		typedef Source src_type;
		typedef Type dest_type;
		typedef std::string string_type;
		typedef std::stringstream internal_stream;
		
		// constructor and destructor
		explicit cast_stream(std::ios::fmtflags base = std::ios::dec) : ss_() {
			ss_.setf(base, std::ios::basefield);
		}
		
		virtual ~cast_stream() {}
		
		// operator
		bool operator<<(const src_type& src) {
			ss_ << std::setiosflags(std::ios::fixed);
			return !(ss_ << src).fail();
		}
		
		template <class ValueT>
		bool operator>>(ValueT& dest) {
			return !(ss_ >> dest).fail();
		}
		
		// speciallized for string_type;
		bool operator>>(string_type& dest) {
			dest = ss_.str();
			return true;
		}
	
	private:
		internal_stream ss_;
	};
	
	/* --------------------------------------------------------------------- */
	/*!
	 *  lexical_cast
	 *
	 *  The lexical_cast class is a sub-set class of boost::lexical_cast.
	 *  Scale back the function of wide string support.
	 */
	/* --------------------------------------------------------------------- */
	template <class Type, class Source>
	Type lexical_cast(const Source& src, std::ios::fmtflags base = std::ios::dec) {
		cast_stream<Type, Source> ss(base);
		Type dest;
		
		if(!(ss << src && ss >> dest)) {
			throw(bad_lexical_cast(typeid(Source), typeid(Type)));
		}
		return dest;
	};
	
	/* --------------------------------------------------------------------- */
	//  specialized lexical_cast
	/* --------------------------------------------------------------------- */
	template <> char lexical_cast(const char& src, std::ios::fmtflags base) { return src; }
	template <> int lexical_cast(const int& src, std::ios::fmtflags base) { return src; }
	template <> long lexical_cast(const long& src, std::ios::fmtflags base) { return src; }
	template <> unsigned char lexical_cast(const unsigned char& src, std::ios::fmtflags base) { return src; }
	template <> unsigned int lexical_cast(const unsigned int& src, std::ios::fmtflags base) { return src; }
	template <> unsigned long lexical_cast(const unsigned long& src, std::ios::fmtflags base) { return src; }
	template <> double lexical_cast(const double& src, std::ios::fmtflags base) { return src; }
	template <> std::string lexical_cast(const std::string& src, std::ios::fmtflags base) { return src; }
}

#endif // CLX_LEXICAL_CAST_H
