/* ------------------------------------------------------------------------- */
/*
 *  sockbuf.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Sun 13 Jul 2008 15:13:03 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_SOCKBUF_H
#define CLX_SOCKBUF_H

#include <streambuf>
#include <vector>
#include <cstring>

namespace clx {
	/* --------------------------------------------------------------------- */
	//  basic_sockbuf
	/* --------------------------------------------------------------------- */
	template <
		class Socket,
		class CharT,
		class Traits = std::char_traits<CharT>
	>
	class basic_sockbuf : public std::basic_streambuf<CharT, Traits> {
	public:
		typedef Socket socket_type;
		typedef CharT char_type;
		typedef Traits traits;
		typedef typename std::vector<CharT> container;
		typedef typename std::basic_string<CharT, Traits> string_type;
		typedef typename container::size_type size_type;
		typedef typename Traits::int_type int_type;
		
		static const int npback = 8;
		
		// constructor and destructor
		explicit basic_sockbuf(size_type n) :
			super(), sock_(), get_buffer_(n) {
			this->reset();
		}
		
		explicit basic_sockbuf(socket_type* s, size_type n) :
			super(), sock_(s), get_buffer_(n) {
			this->reset();
		}
		
		virtual ~basic_sockbuf() {}
		
		void socket(socket_type* s) { sock_ = s; }
		
	protected:
		virtual int_type overflow(int_type c) {
			if (!traits::eq_int_type(c, traits::eof())) {
				char tmp = static_cast<char>(c);
				if (this->xsputn(&tmp, 1) == 0) return traits::eof();
			}
			return traits::not_eof(c);
		}
		
		virtual std::streamsize xsputn(const char_type* s, std::streamsize n) {
			std::streamsize transferred = 0;
			while (transferred < n) {
				int l = sock_->send(s, n - transferred);
				if (l <= 0) break;
				transferred += l;
				s += l;
			}
			return transferred;
		}
		
		virtual int_type underflow() {
			if (this->gptr() == this->egptr()) {
				// copy the previous read string to the putback field
				int n = static_cast<int>(this->gptr() - this->eback());
				if (n > npback) n = npback;
				std::memcpy(&get_buffer_[npback-n], this->gptr() - n, n);
				
				int l = sock_->recv(&get_buffer_[npback], static_cast<int>(get_buffer_.size() - npback));
				if (l <= 0) return traits::eof();
				this->setg(&get_buffer_[npback-n], &get_buffer_[npback], &get_buffer_[npback + l]);
				return traits::to_int_type(*this->gptr());
			}
			else return traits::eof();
		}
		
		virtual int sync() { return this->overflow(traits::eof()); }
		
	private:
		typedef std::basic_streambuf<CharT, Traits> super;
		
		socket_type* sock_;
		container get_buffer_;
		
		void reset() {
			this->setg(&get_buffer_[0], &get_buffer_[npback], &get_buffer_[npback]);
		}
	};
};

#endif // CLX_SOCKBUF_H
