/* ------------------------------------------------------------------------- */
/*
 *  ini.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Fri 27 Feb 2008 04:13:16 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_INI_H
#define CLX_INI_H

#include <iostream>
#include <string>
#include <vector>
#include <map>
#include <fstream>
#include <stdexcept>
#include "lexical_cast.h"
#include "salgorithm.h"
#include "syntax_error.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  ini_container
	/* --------------------------------------------------------------------- */
	template <
		class Type,
		class CharT,
		class Traits = std::char_traits<CharT>
	>
	class ini_container : public std::map<std::basic_string<CharT, Traits>, Type> {
	public:
		typedef CharT char_type;
		typedef std::basic_string<CharT, Traits> string_type;
		typedef std::map<string_type, Type> container;
		typedef typename container::key_type key_type;
		typedef typename container::mapped_type mapped_type;
		typedef typename container::value_type value_type;
		typedef typename container::iterator iterator;
		typedef typename container::const_iterator const_iterator;
		
		ini_container() : container() {}
		
		std::pair<iterator, bool> insert(const string_type& s) {
			std::vector<string_type> v;
			split_if(s, v, is_any_of("="));
			if (v.empty()) return std::pair<iterator, bool>(this->end(), false);
			string_type key = v.at(0);
			string_type val;
			if (v.size() >= 2) val = v.at(1);
			return container::insert(value_type(key, lexical_cast<mapped_type>(val)));
		}
		
		std::pair<iterator, bool> insert(const char_type* s) {
			string_type tmp(s);
			return this->insert(tmp);
		}
	};
	
	/* --------------------------------------------------------------------- */
	//  basic_ini
	/* --------------------------------------------------------------------- */
	template <
		class Type,
		class CharT = char,
		class Traits = std::char_traits<CharT>
	>
	class basic_ini : public std::map<std::basic_string<CharT, Traits>,
		ini_container<Type, CharT, Traits> > {
	public:
		typedef CharT char_type;
		typedef std::basic_string<CharT, Traits> string_type;
		typedef std::map<string_type, ini_container<Type, CharT, Traits> > container;
		typedef std::basic_istream<CharT, Traits> istream;
		typedef typename container::key_type key_type;
		typedef typename container::mapped_type mapped_type;
		typedef typename container::value_type value_type;
		typedef typename container::iterator iterator;
		typedef typename container::const_iterator const_iterator;
		
		basic_ini() : container() {}
		
		explicit basic_ini(istream& sin) : container() {
			this->read(sin);
		}
		
		explicit basic_ini(const string_type& path) : container() {
			this->read(path);
		}
		
		explicit basic_ini(const char_type* path) : container() {
			this->read(path);
		}
		
		void read(istream& sin) {
			string_type cur;
			size_t n = 0;
			
			string_type tmp;
			while (std::getline(sin, tmp)) {
				chomp(tmp);
				n++;
				
				if (tmp.empty() || tmp.at(0) == ';') continue;
				else if (tmp.at(0) == '[') {
					size_t last = tmp.find(']');
					if (last == string_type::npos) throw syntax_error(n, "expected ']' token");
					cur = tmp.substr(1, last - 1);
					this->insert(cur);
				}
				else {
					if (cur.empty()) throw syntax_error(n, "expected section name");
					std::pair<typename mapped_type::iterator, bool> ret;
					ret = (*this)[cur].insert(tmp);
					if (!ret.second) throw syntax_error(n, "unknown error");
				}
			}
		}
		
		void read(const string_type& path) {
			this->read(path.c_str());
		}
		
		void read(const char_type* path) {
			std::basic_ifstream<CharT, Traits> fs(path);
			if (fs.fail()) {
				std::basic_stringstream<CharT, Traits> msg;
				msg << path << ": no such file or directory";
				throw std::runtime_error(msg.str());
			}
			this->read(fs);
		}
		
		void insert(const string_type& section) {
			container::insert(value_type(s, mapped_type()));
		}
		
		void insert(const char_type* section) {
			string_type tmp(s);
			this->insert(tmp);
		}
		
		bool exist(const string_type& section) {
			if (this->find(section) == this->end()) return false;
			return true;
		}
		
		bool exist(const char_type* section) {
			string_type tmp(section);
			return this->exist(tmp);
		}
		
		bool exist(const string_type& section, const string_type& entry) {
			iterator p = this->find(section);
			if (p == this->end()) return false;
			typename mapped_type::iterator q = p->second.find(entry);
			if (q == p->second.end()) return false;
			return true;
		}
		
		bool exist(const char_type* section, const char_type* entry) {
			string_type tmp1(section);
			string_type tmp2(entry);
			return this->exist(tmp1, tmp2);
		}
	};
	
	typedef basic_ini<std::string> ini;
}

#endif // CLX_INI_H
