/* ------------------------------------------------------------------------- */
/*
 *  date_time.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Thu 14 Jun 2007 14:38:02 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_DATE_TIME_H
#define CLX_DATE_TIME_H

#include <ctime>
#include <stdexcept>
#include "time_duration.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  date_time
	/* --------------------------------------------------------------------- */
	class date_time {
	public:
		typedef time_value value_type;
		typedef struct tm tm_type;
		
		// constructor and destructor
		date_time() {
			std::time_t t = std::time(NULL);
			this->assign(t);
		}
		
		explicit date_time(const tm_type& t) { this->assign(t); }
		explicit date_time(const std::time_t& t) { this->assign(t); }
		
		explicit date_time(value_type& y, value_type& m, value_type& d,
			value_type& h, value_type& mm, value_type& s) {
			this->assign(y, m, d, h, mm, s);
		}
		
		date_time& operator=(const date_time& t) { return this->assign(t); }
		date_time& operator=(const tm_type& t) { return this->assign(t); }
		date_time& operator=(const std::time_t& t) { return this->assign(t); }
		
		date_time& operator+=(value_type t) { return this->after(t); }
		date_time& operator+=(const date_duration& t) { return this->after(t); }
		date_time& operator-=(value_type t) { return this->before(t); }
		date_time& operator-=(const date_duration& t) { return this->before(t); }
		
		double operator-(const date_time& t) const { return this->difference(t); }
		double operator-(const std::time_t& t) const { return this->difference(t); }
		double operator-(const tm_type& t) const { return this->difference(t); }
		
		// binary operator
		friend date_time operator+(date_time x, value_type y) { return x += y; }
		friend date_time operator+(date_time x, const date_duration& y) { return x += y; }
		friend date_time operator-(date_time x, value_type y) { return x -= y; }
		friend date_time operator-(date_time x, const date_duration& y) { return x -= y; }
		friend date_time operator+(value_type y, date_time x) { return x += y; }
		friend date_time operator+(const date_duration& y, date_time x) { return x += y; }
		friend date_time operator-(value_type y, date_time x) { return x -= y; }
		friend date_time operator-(const date_duration& y, date_time x) { return x -= y; }
		
		// comparison operator
		friend bool operator<(const date_time& x, const date_time& y) {
			if (std::difftime(y.c_time(), x.c_time()) > 0.01) return true;
			return false;
		}
		
		friend bool operator>(const date_time& x, const date_time& y) {
			if (std::difftime(x.c_time(), y.c_time()) > 0.01) return true;
			return false;
		}
		
		friend bool operator<=(const date_time& x, const date_time& y) {
			return !(x > y);
		}
		
		friend bool operator>=(const date_time& x, const date_time& y) {
			return !(x < y);
		}
		
		friend bool operator==(const date_time& x, const date_time& y) {
			if (!(x < y) && !(x > y)) return true;
			return false;
		}
		
		friend bool operator!=(const date_time& x, const date_time& y) {
			return !(x == y);
		}
		
		// accessor
		value_type year() const { return date_.tm_year + 1900; }
		value_type mon() const { return date_.tm_mon + 1; }
		value_type day() const { return date_.tm_mday; }
		value_type hour() const { return date_.tm_hour; }
		value_type min() const { return date_.tm_min; }
		value_type sec() const { return date_.tm_sec; }
		value_type totaldays() const { return date_.tm_yday; }
		
		// get information as the traditional std::tm type
		const tm_type& reference() const { return date_; }
		const tm_type* data() const { return &date_; }
		
		std::time_t c_time() const {
			tm_type tmp = date_;
			return std::mktime(&tmp);
		}
		
		date_time& assign(const date_time& t) {
			return this->assign(t.date_);
		}
		
		date_time& assign(const tm_type& t) {
			date_ = t;
			return *this;
		}
		
		date_time& assign(const std::time_t& t) {
			struct tm tval;
			::localtime_r(&t, &tval);
			return this->assign(tval);
		}
		
		date_time& assign(value_type y, value_type m, value_type d,
			value_type h, value_type mm, value_type s) {
			date_.tm_year = y - 1900;
			date_.tm_mon  = m - 1;
			date_.tm_mday = d;
			date_.tm_hour = h;
			date_.tm_min  = mm;
			date_.tm_sec  = s;
			return *this;
		}
		
		date_time& after(value_type t) {
			return this->assign(this->c_time() + t);
		}
		
		date_time& after(const date_duration& t) {
			date_.tm_year += t.years();
			date_.tm_mon  += t.months();
			date_.tm_mday += t.days();
			
			return this->assign(this->c_time());
		}
		
		date_time& before(value_type t) {
			std::time_t tmp = this->c_time() - t;
			if (tmp < 0) throw std::range_error("negative value");
			return this->assign(tmp);
		}
		
		date_time& before(const date_duration& t) {
			date_.tm_year -= t.years();
			date_.tm_mon  -= t.months();
			date_.tm_mday -= t.days();
			
			return this->assign(this->c_time());
		}
		
		double difference(const date_time& t) const {
			return std::difftime(this->c_time(), t.c_time());
		}
		
		double difference(const std::time_t& t) const {
			return std::difftime(this->c_time(), t);
		}
		
		double difference(const std::tm& t) const {
			std::time_t tmp = std::mktime(const_cast<std::tm*>(&t));
			return std::difftime(this->c_time(), tmp);
		}
		
	private:
		tm_type date_;
	};
}

#endif // CLX_DATE_TIME_H
