/* ------------------------------------------------------------------------- */
/*
 *  smtp.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Thu 23 Oct 2008 07:56:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_SMTP_H
#define CLX_SMTP_H

#include <iostream>
#include <string>
#include <stdexcept>
#include "tcp.h"
#include "lexical_cast.h"
#include "strip.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  smtp_error
	/* --------------------------------------------------------------------- */
	class smtp_error : public std::runtime_error {
	public:
		typedef char char_type;
		typedef std::string string_type;
		
		explicit smtp_error(int code, const string_type& what_arg) :
			std::runtime_error(what_arg), code_(code) {}
		
		virtual ~smtp_error() throw() {}
		
		int code() const { return code_; }
		
	private:
		int code_;
	};
	
	/* --------------------------------------------------------------------- */
	//  basic_smtp
	/* --------------------------------------------------------------------- */
	template <class Socket>
	class basic_smtp : public Socket {
	public:
		typedef Socket socket_type;
		typedef char char_type;
		typedef std::basic_string<char_type> string_type;
		typedef std::pair<int, string_type> response_type;
		typedef std::vector<response_type> response_array;
		
		basic_smtp() : socket_type(), id_(), host_(), port_(0), res_(), sbuf_() {}
		
		explicit basic_smtp(const string_type& id, const socket_type& cp) :
			socket_type(cp), id_(id), host_(), port_(0), res_(), sbuf_() {
			host_ = cp.address().ipaddr();
			port_ = cp.address().port();
			this->start();
		}
		
		explicit basic_smtp(const string_type& id, const string_type& host, int port = 25) :
			socket_type(), id_(id), host_(host), port_(port), res_(), sbuf_() {
			this->start();
		}
		
		explicit basic_smtp(const char_type* id, const char_type* host, int port = 25) :
			socket_type(), id_(id), host_(host), port_(port), res_(), sbuf_() {
			this->start();
		}
		
		virtual ~basic_smtp() { this->finish(); }
		
		basic_smtp& start() {
			if (!this->is_open()) this->connect(host_, port_);
			sbuf_.socket(*this);
			std::basic_iostream<char_type> ss(&sbuf_);
			
			this->xresponse(ss);
			string_type buf;
			buf = "EHLO " + id_ + " " + host_ + "\r\n";
			ss << buf;
			this->xresponse(ss);
			
			return *this;
		}
		
		basic_smtp& start(const string_type& id, const string_type& host, int port = 25) {
			id_ = id;
			host_ = host;
			port_ = port;
			return this->start();
		}
		
		basic_smtp& start(const char_type* id, const char_type* host, int port = 25) {
			string_type tmp_id(id);
			string_type tmp_host(host);
			return this->start(tmp_id, tmp_host, port);
		}
		
		void finish() {
			if (this->is_open()) {
				sbuf_.socket(*this);
				std::basic_iostream<char_type> ss(&sbuf_);
				ss << "QUIT\r\n";
				this->xresponse(ss);
				this->close();
			}
		}
		
		basic_smtp& mail(const string_type& from, const string_type& to, const string_type& message) {
			sbuf_.socket(*this);
			std::basic_iostream<char_type> ss(&sbuf_);
			string_type buf;
			
			// A mail transaction has 4 steps.
			buf = "MAIL FROM:<" + from + ">\r\n";
			ss << buf;
			this->xresponse(ss);
			
			buf = "RCPT TO:<" + to + ">\r\n";
			ss << buf;
			this->xresponse(ss);
			
			ss << "DATA\r\n";
			this->xresponse(ss);
			
			ss << message;
			this->xresponse(ss);
			
			return *this;
		}
		
		basic_smtp& mail(const char_type* from, const char_type* to, const char_type* message) {
			string_type tmp_from(from);
			string_type tmp_to(to);
			string_type tmp_msg(message);
			return this->mail(tmp_from, tmp_to, tmp_msg);
		}
		
		// get response messages from the server
		response_array& responses() { return res_; }
		const response_array& responses() const { return res_; }
		const response_type& response(size_t index) { return res_.at(index); }
		
	private:
		string_type id_;
		string_type host_;
		int port_;
		response_array res_;
		
		basic_sockbuf<socket_type, char_type> sbuf_;
		
		template <class InputStream>
		void xresponse(InputStream& sin) {
			std::string buf;
			while (std::getline(sin, buf)) {
				chomp(buf);
				response_type tmp;
				tmp.first = lexical_cast<int>(buf.substr(0, 3));
				tmp.second = buf.substr(4);
				res_.push_back(tmp);
				if (tmp.first >= 400) throw smtp_error(tmp.first, tmp.second);
				if (buf[3] == ' ') break;
			}
		}
	};
	
	typedef basic_smtp<tcp::socket> smtp;
}

#endif // CLX_SMTP_H
