/* ------------------------------------------------------------------------- */
/*
 *  hexdump.h
 *
 *  Copyright (c) 2004 - 2009, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Tue 03 Feb 2009 11:13:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_HEXDUMP_H
#define CLX_HEXDUMP_H

#include <iterator>
#include <sstream>
#include <limits>

namespace clx {
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class InIter, class OutIter>
	OutIter hexdump(InIter first, InIter last, OutIter dest, const std::basic_string<Ch, Tr>& prefix) {
		static const int shift = ((sizeof(unsigned long) - sizeof(Ch)) * 8);
		static const unsigned long mask = std::numeric_limits<unsigned long>::max() >> shift;
		
		std::basic_stringstream<Ch, Tr> ss;
		for (; first != last; ++first) {
			if (!prefix.empty()) ss << prefix;
			ss << std::hex << (static_cast<int>(*first) & mask);
			*dest++ = ss.str();
			ss.str(std::basic_string<Ch, Tr>());
		}
		return dest;
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class CharT, class InIter, class OutIter>
	OutIter hexdump(InIter first, InIter last, OutIter dest, const CharT* prefix) {
		std::basic_string<CharT> tmp(prefix);
		return hexdump(first, last, dest, tmp);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class OutIter>
	OutIter hexdump(std::basic_istream<Ch, Tr>& src, OutIter dest,
		const std::basic_string<Ch, Tr>& prefix) {
		std::istream_iterator<Ch> input(src);
		std::istream_iterator<Ch> last;
		return hexdump(input, last, dest, prefix);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class OutIter>
	OutIter hexdump(std::basic_istream<Ch, Tr>& src, OutIter dest, const Ch* prefix) {
		std::basic_string<Ch, Tr> tmp(prefix);
		return hexdump(src, dest, tmp);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class OutIter>
	OutIter hexdump(std::basic_istream<Ch, Tr>& src, OutIter dest) {
		std::basic_string<Ch, Tr> tmp;
		return hexdump(src, dest, tmp);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class OutIter>
	OutIter hexdump(const std::basic_string<Ch, Tr>& src, OutIter dest,
		const std::basic_string<Ch, Tr>& prefix) {
		return hexdump(src.begin(), src.end(), dest, prefix);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class Ch, class Tr, class OutIter>
	OutIter hexdump(const std::basic_string<Ch, Tr>& src, OutIter dest) {
		std::basic_string<Ch, Tr> tmp;
		return hexdump(src, dest, tmp);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class CharT, class OutIter>
	OutIter hexdump(const CharT* src, OutIter dest, const CharT* prefix) {
		std::basic_string<CharT> tmp_src(src);
		std::basic_string<CharT> tmp_prefix(prefix);
		return hexdump(tmp_src, dest, tmp_prefix);
	}
	
	/* --------------------------------------------------------------------- */
	//  hexdump
	/* --------------------------------------------------------------------- */
	template <class CharT, class OutIter>
	OutIter hexdump(const CharT* src, OutIter dest) {
		std::basic_string<CharT> tmp_src(src);
		std::basic_string<CharT> tmp_prefix;
		return hexdump(tmp_src, dest, tmp_prefix);
	}
}

#endif // CLX_HEXDUMP_H
