/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.http;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.http.HttpServletRequest;

import org.codecluster.C2Constants;
import org.codecluster.http.C2RequestInfo;

/**
 * リダイレクトURLを生成するルールの実装クラスです。<br>
 * C2RequestInfo#isSecure() を元にリダイレクト先を決定します。<br>
 */
public class AutoRedirectRuleImpl implements RedirectRule, C2Constants {
	private static final Logger logger = Logger.getLogger(AutoRedirectRuleImpl.class.getName());

	/**
	 * C2RequestInfo#isSecure() を元にリダイレクト先を決定します。<br>
	 * <br>
	 * @see org.codecluster.http.RedirectRule#redirect(java.lang.String, javax.servlet.http.HttpServletRequest)
	 */
	public String redirect(String location, HttpServletRequest request) {

		// Scheme の調整。実際にクライアントとのSSL通信状態から 決める
		C2RequestInfo c2req = new C2RequestInfo(request);
		String scheme;
		if (c2req.isSecure()) {
			scheme = "https";
		} else {
			scheme = "http";
		}
		
		// リダイレクト先を再構築
		String newLocation = location;
		try {
			// 相対アドレスの調整
			URI uri = (new URI(request.getRequestURL().toString())).resolve(location);
			// URL 再構築
			int port = request.getServerPort();
			if (port == 80 || port == 443) {
				// デフォルトポート番号
				port = -1;
			}
			URL url = new URL(scheme, request.getServerName(), port, uri.toURL().getFile());
			newLocation = url.toString();
			if (logger.isLoggable(Level.FINE)) {
	    		logger.fine("sendRedirect: " + location + " -> " + newLocation);
			}
		} catch (MalformedURLException e) {
			// もとのままとする
			if (logger.isLoggable(Level.FINE)) {
	    		logger.fine(e.toString());
	    	}
		} catch (URISyntaxException e) {
			// もとのままとする
			if (logger.isLoggable(Level.FINE)) {
	    		logger.fine(e.toString());
	    	}
		}
		
		return newLocation;
	}
}
