/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.http;

import javax.servlet.http.HttpServletRequest;

import org.codecluster.C2Constants;
import org.codecluster.util.C2Properties;
import org.codecluster.util.C2PropertiesManager;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * HttpServletRequest 内のリクエストヘッダを解析してクライアントの追加情報を取得し格納するクラスです。<br>
 * <br>
 * C2config.xml にて指定したプロパティに応じて情報を取得します。<br>
 * <br>
 * remote-address-header: リモートアドレスが格納されいているリクエストヘッダ名で、
 * 空にすれば取得しません。<br>
 * secure-header: リモートからのリクエストがSSL通信によるものであったか否かを
 * 判定するためのリクエストヘッダ名で、空にすれば取得しません。<br>
 * secure-true-keyword: SSL通信であると判断するキーワード。
 * secure-header で指定したヘッダの内容がこの値と一致する場合にSSL通信とみなします。
 * 空にすれば secure-header があれば内容にかかわらずSSL通信とみなします。<br>
 * <br>
 * C2config.xml プロパティファイル中に記述されたプロパティ "cache" で指定した時間（秒）は
 * プロパティの再読み込みを行いません。"-1" を指定すると完全に再読み込みは行わなくなります。<br>
 * ただし、reload フラグを true で、コンストラクタを呼び出した場合にはキャッシュ時間にかかわらず
 * 強制的に再読み込みを行います。<br>
 * <br>
 * ※注意※<br>
 * <br>
 * リクエストヘッダ名はクライアントから類推できない名称にするか、リバースプロキシなどで、
 * 必ず正規の値に書きかえるか、一度削除することを推奨します。<br>
 * クライアントからのヘッダを信用しないように対策してください。
 * さもなければリモートアドレスなどを詐称できることになります。<br>
 * 
 */
public class C2RequestInfo implements java.io.Serializable, C2Constants {
	private static final long serialVersionUID = 1L;
	private static final Logger logger = Logger.getLogger(C2RequestInfo.class.getName());

	/**
	 * リクエスト解析に利用するプロパティ
	 */
	private static C2Properties prop = null;

	/**
	 * プロパティから読み取った値
	 */
	private static String propRemoteAddressHeader = "";
	private static String propSecureHeader = "";
	private static String propSecureTrueKeyword = "";

	// 読み取るべき HttpServletRequest
	private HttpServletRequest request = null;
	
	/**
	 * リクエストヘッダを解析した結果
	 */
	private String remoteAddr = null;
	private boolean initSecure = false; // secure を解析したかどうか
	private boolean secure = false;

	/**
	 * リクエストが SSL 通信であるかを確認します。
	 * @return SSL通信の場合は true を返す
	 */
	public boolean isSecure() {
		if (this.initSecure == false) {
			// HTTP ヘッダを見て SSL 状態を判断する
			String secureFlag;
			if (propSecureHeader.length() > 0 
				&& (secureFlag = request.getHeader(propSecureHeader)) != null) {
	
				// 設定ファイルの比較文字列が空か、比較文字列とヘッダ中の文字列が一致する場合は secureとみなす
				if (propSecureTrueKeyword.length() == 0
						|| propSecureTrueKeyword.equalsIgnoreCase(secureFlag)) {
					this.secure = true;
					if (logger.isLoggable(Level.FINE)) {
						logger.fine("secure: true by " + propSecureHeader + "=" + secureFlag);
					}
				} else {
					if (logger.isLoggable(Level.FINE)) {
						logger.fine("secure: false by " + propSecureHeader + "=" + secureFlag);
					}
				}
			} else {
				this.secure = request.isSecure();
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("secure: " + this.secure + " by request.isSecure()");
				}
			}
			this.initSecure = true;
		}
		return this.secure;
	}
	/**
	 * リクエストが SSL 通信であるかを設定します。
	 * @param secure SSL通信の場合は true を設定
	 */
	protected void setSecure(boolean secure) {
		this.secure = secure;
	}

	/**
	 * クライアントのリモートアドレスを返します。<br>
	 * <br>
	 * 設定ファイルで定義したヘッダ情報をもとにクライアントのリモートアドレスを返します。<br>
	 * @return リモートアドレス
	 */
	public String getRemoteAddr() {
		if (this.remoteAddr == null) {
			// 指定 http ヘッダを見てリモートアドレスを取得する
			String r;
			if (propRemoteAddressHeader.length() > 0
					&& (r = request.getHeader(propRemoteAddressHeader)) != null
					&& (this.remoteAddr = r.trim()).length() > 0) {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("remote address: " + this.remoteAddr + " by " + propRemoteAddressHeader);
				}
			} else {
				this.remoteAddr = request.getRemoteAddr();
				if (this.remoteAddr == null) {
					this.remoteAddr = "";
				}
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("remote address: " + this.remoteAddr + " by request.getRemoteAddr()");
				}
			}
		}

		return this.remoteAddr;
	}
	/**
	 * クライアントのリモートアドレスを設定します。<br>
	 * @param remoteAddr リモートアドレス
	 */
	protected void setRemoteAddr(String remoteAddr) {
		this.remoteAddr = remoteAddr;
	}

	/**
	 * 指定された HttpRequest より追加のクライアント情報を取得します。<br>
	 * reload フラグに true を指定するとプロパティファイルを強制的に再読み込みします。
	 * 指定していない場合にはプロパティファイルはキャッシュ時間内は再読み込みを行いません。
	 * @param request リクエスト
	 * @param reload true でプロパティファイルを強制再読み込みします
	 */
	public C2RequestInfo(HttpServletRequest request, boolean reload) {
		this.request = request;
		loadXML(reload);
	}

	/**
	 * 指定された HttpRequest より追加のクライアント情報を取得します。<br>
	 * プロパティファイルはキャッシュ時間内は再読み込みを行いません。
	 * @param request リクエスト
	 */
	public C2RequestInfo(HttpServletRequest request) {
		this.request = request;
		loadXML(false);
	}
	
	/**
	 * プロパティ XML の読み込みをキャッシュ時間を考慮して行います。<br>
	 * 実装は C2PropertiesManager に依存します。
	 * @param reload true でキャッシュを使用せず強制再読み込みします
	 */
	protected void loadXML(boolean reload) {
		prop = C2PropertiesManager.getC2PropertiesFromXML(DEFAULT_CONFIG_XML, reload);

		propRemoteAddressHeader = prop.getProperty(CONF_REMOTE_ADDRESS_HEADER, "").trim();
		propSecureHeader = prop.getProperty(CONF_SECURE_HEADER, "").trim();
		propSecureTrueKeyword = prop.getProperty(CONF_SECURE_TRUE_KEYWORD, "").trim();

		if (logger.isLoggable(Level.FINE)) {
			logger.fine("get properites:"
					+ "\nremote-address-header: " + propRemoteAddressHeader
					+ "\nsecure-header: " + propSecureHeader
					+ "\nsecure-true-keyword: " + propSecureTrueKeyword);
		}
	}
}
