/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.session;

import java.security.MessageDigest;
import java.util.HashMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

/**
 * 仮想セッションを生成管理するクラスです。<br>
 * <br>
 * HttpServletRequest に関連付けられた HttpSession 中の仮想セッションを返却し、
 * まだ作られていない場合には新規に仮想セッションIDを作成して HttpSession に関連付けます。<br>
 * 自動で新規に作成しないことも選択できます。<br>
 * <br>
 * 仮想セッションIDはデフォルトではリクエストパラメータ(vsid)から取得します。<br>
 * 取得するパラメータ名の規定値は DEFAULT_VIRTUAL_SESSION_KEY で定義した値を用います。
 * または getSession() 時に指定することもできます。<br>
 * <br>
 * C2Session が取得できた場合には同時に HttpServletRequest#getAttribute() にて取得可能な以下の情報が格納されます。
 * <pre>
 *  - String vsid = (String)request.getAttribute("vsidString");
 *  - HashMap map = (HashMap)request.getAttribute("vsidMap");
 *  - C2Session s = (C2Session)request.getAttribute("vsidBean");
 *  ※キーの "vsid" の部分は getSession() で virtualSessionKey を指定していればその値が使われます。
 * </pre>
 * <br>
 * {@link #generateVirtualSessionId() generateVirtualSessionId()} をオーバーライドすることで仮想セッションIDの生成ロジックを変更できます<br>
 * 
 */
public class C2SessionManager {
	
	// インスタンス化不可
	private C2SessionManager() {}
	
	/**
	 * 仮想セッションIDを取得するリクエストパラメータ名。規定値({@value})
	 */
	public static final String DEFAULT_VIRTUAL_SESSION_KEY = "vsid";

	/**
	 * HttpServletRequest に関連付けられた現在の仮想セッションを返します。<br>
	 * セッションが関連付けられていない場合は、新しく作成します。<br>
	 * 
	 * @param request
	 * @return HttpServletRequest に関連付けられた現在の仮想セッション
	 */
	public static C2Session getSession(HttpServletRequest request) {
		return getSession(request, DEFAULT_VIRTUAL_SESSION_KEY);
	}

	/**
	 * HttpServletRequest に関連付けられた現在の仮想セッションを返します。<br>
	 * セッションが関連付けられていない場合は、新しく作成します。<br>
	 * 
	 * @param request
	 * @param virtualSessionKey
	 * @return HttpServletRequest に関連付けられた現在の仮想セッション
	 */
	public static C2Session getSession(HttpServletRequest request, String virtualSessionKey) {
		C2Session s = null;
		try {
			s = getSession(request, virtualSessionKey, true);
		} catch (NoSessionException e) {
			// 出ないはず
			e.printStackTrace();
		} catch (ExpiresSessionException e) {
			// 出ないはず
			e.printStackTrace();
		}
		return s;
	}

	/**
	 * HttpServletRequest に関連付けられた現在の仮想セッションを返します。<br>
	 * <br>
	 * 現在のセッションがなく、create が true の場合は、新しいセッションを返します。
	 * create が false で、しかも要求に有効な HttpSession がない場合、
	 * このメソッドは例外 NoSessionException をスローします。<br>
	 * create が false で、仮想セッションの有効期限が切れていた場合、
	 * このメソッドは例外 ExpiresSessionException をスローします。<br>
	 * 
	 * @param request
	 * @param create
	 * @return HttpServletRequest に関連付けられた現在の仮想セッション
	 * @throws NoSessionException 有効な HttpSession か、仮想セッションがない
	 * @throws ExpiresSessionException 仮想セッションの有効期限が切れている
	 */
	public static C2Session getSession(HttpServletRequest request, boolean create)
			throws NoSessionException, ExpiresSessionException {
		return getSession(request, DEFAULT_VIRTUAL_SESSION_KEY, create);
	}

	/**
	 * HttpServletRequest に関連付けられた現在の仮想セッションを返します。<br>
	 * <br>
	 * 現在のセッションがなく、create が true の場合は、新しいセッションを返します。
	 * create が false で、しかも要求に有効な HttpSession がない場合、
	 * このメソッドは例外 NoSessionException をスローします。<br>
	 * create が false で、仮想セッションの有効期限が切れていた場合、
	 * このメソッドは例外 ExpiresSessionException をスローします。<br>
	 * 
	 * @param request
	 * @param virtualSessionKey
	 * @param create
	 *            新規セッションを必要に応じて作成する場合は true。現在のセッションがないときに null を返す場合は false
	 * @return HttpServletRequest に関連付けられた現在の仮想セッション
	 * @throws NoSessionException 有効な HttpSession か、仮想セッションがない
	 * @throws ExpiresSessionException 仮想セッションの有効期限が切れている
	 */
	public static C2Session getSession(HttpServletRequest request,
			String virtualSessionKey, boolean create)
			throws NoSessionException, ExpiresSessionException {
		
		// HttpSession の取得
		HttpSession session = request.getSession(create);
		if (session == null) {
			throw new NoSessionException(NoSessionException.NO_HTTP_SESSION);
		}

		// 仮想セッションIDの取得 or 新規作成
		String vsid = (String) request.getParameter(virtualSessionKey);
		if (vsid == null) {
			if (create == false) {
				throw new NoSessionException(NoSessionException.NO_VIRTUAL_SESSION, "not found virtual session key = '" + virtualSessionKey + "'");
			}
			vsid = generateVirtualSessionId();
		}

		C2Session s = null;
		try {
			// 仮想セッションの作成
			s = new C2Session(request.getSession(), virtualSessionKey, vsid, create);
		}
		catch (ExpiresSessionException e) {
			// 既存の仮想セッションが有効期限切れだった場合
			if (create == true) {
				// create が  true のときの有効期限切れは新しいセッションを自動作成する
				vsid = generateVirtualSessionId();
				// ここで例外が出た場合にはキャッチせずにそのままとする
				s = new C2Session(request.getSession(), virtualSessionKey, vsid, create);
			} else {
				// create が false のときは有効期限切れを投げる
				throw e;
			}
		}
		
		// JSPなどで使いやすいようにリクエストセッションに仮想セッションIDなどを保存
		//   virtualSessionKey が "vsid" の場合、
		//    String vsid = (String)request.getAttribute("vsidString");
		//    HashMap map = (HashMap)request.getAttribute("vsidMap");
		//    C2Session s = (C2Session)request.getAttribute("vsidBean");
		// Struts を使う場合には ActionForm に適切に設定するほうが便利
		HashMap<String, String> map = new HashMap<String, String>();
		map.put(virtualSessionKey, s.getId());
		request.setAttribute(virtualSessionKey + "Map", map);
		request.setAttribute(virtualSessionKey + "Bean", s);
		request.setAttribute(virtualSessionKey + "String", s.getId());

		return s;
	}

	/**
	 * 仮想セッションIDを生成し返却します。<br>
	 * <br>
	 * 仮想セッションIDの生成ロジックを変更する場合にはこのメソッドをオーバーライドします。
	 * 
	 * @return 新規作成された仮想セッションID
	 */
	public static String generateVirtualSessionId() {
		String result = "";
		String time = String.valueOf(System.currentTimeMillis());
		String hash = String.valueOf(Thread.currentThread().hashCode());

		try {
			byte[] digest = getStringDigest(time + hash);
			result = byteToHexString(digest);
		}
		catch( Exception exception ) {
			result = time + hash;
		}
		return result;
	}
	
	/**
	 * byte配列を16進数の文字列で返却します。
	 * @param digest
	 */
	private static String byteToHexString( byte[] digest ) {
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < digest.length; i++) {
			// ダイジェスト値の1バイトを16進数2桁に変換で表示する
			builder.append(String.format("%02x", digest[i]));
		}
		return builder.toString();
	}

	/**
	 * 文字列からダイジェストを生成します。
	 * @param data
	 * @return
	 * @throws Exception
	 */
	private static byte[] getStringDigest(String data) throws Exception {
		MessageDigest md = MessageDigest.getInstance("MD5");
		byte[] dat = data.getBytes();
		md.update(dat);
		return md.digest();
	}
}
