/*
 * Copyright 2009 Project CodeCluster
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KI ND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codecluster.util;

import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.InvalidPropertiesFormatException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * プロパティファイル、XMLファイルを class path から読み込むメソッド、パス指定するメソッドを追加した java.util.Properties クラスです。
 * 
 */
public class C2Properties extends java.util.Properties {
	private static final long serialVersionUID = 1L;
	private static final Logger logger = Logger.getLogger(C2Properties.class.getName());

	private boolean cached = false;

	/**
	 * このプロパティ値が C2PropertiesManager のキャッシュ値なのか否かを返します。
	 * @return true はキャッシュが返却されたことを示します。
	 */
	public boolean isCached() {
		return cached;
	}

	/**
	 * このプロパティ値が C2PropertiesManager のキャッシュ値なのか否かを設定します。<br>
	 * このメソッドは C2PropertiesManager によって設定されることを想定しています。
	 * @param cached true はキャッシュを返却することを示します。
	 */
	public void setCached(boolean cached) {
		this.cached = cached;
	}

	/**
	 * 指定した XML ファイルを class path からさがして読み込みます。<br>
	 * ファイルが存在しない場合にはなにもせずに終了します。<br>
	 * <br>
	 * XML ファイルのフォーマットは java.util.Properties に従ってください。<br>
	 * @param xmlfilename
	 * @throws InvalidPropertiesFormatException
	 * @throws IOException
	 */
	public void loadFromXML(String xmlfilename)
		throws InvalidPropertiesFormatException, IOException {
		
		InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(xmlfilename);
		if (is != null) {
			try {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("load XML file = " + Thread.currentThread().getContextClassLoader().getResource(xmlfilename).toString());
				}
				loadFromXML(is);
			}
			finally {
				try {
					is.close();
				}
				catch(IOException e) {
				}
			}
		} else {
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("not found: " + xmlfilename);
			}
		}
	}

	/**
	 * 指定したプロパティファイルを class path からさがして読み込みます。<br>
	 * ファイルが存在しない場合にはなにもせずに終了します。<br>
	 * @param filename
	 * @throws IllegalArgumentException
	 * @throws IOException
	 */
	public void load(String filename)
		throws IllegalArgumentException, IOException {
		
		InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(filename);
		if (is != null) {
			try {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("load properties file = " + Thread.currentThread().getContextClassLoader().getResource(filename).toString());
				}
				load(is);
			}
			finally {
				try {
					is.close();
				}
				catch(IOException e) {
				}
			}
		} else {
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("not found: " + filename);
			}
		}
	}

	/**
	 * 指定したプロパティファイルを読み込みます。<br>
	 * ファイルが存在しない場合には FileNotFoundException が発生します。<br>
	 * @param file
	 * @throws IllegalArgumentException
	 * @throws IOException
	 * @throws FileNotFoundException
	 */
	public void loadFromFile(String file)
		throws IllegalArgumentException, IOException {
		
		InputStream is = new FileInputStream(file);
		try {
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("load properties file = " + file);
			}
			load(is);
		}
		finally {
			try {
				is.close();
			}
			catch(IOException e) {
			}
		}
	}
}
