package org.codecluster.util;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * <p>時間制限機能を付加したHashMapクラスです。</p>
 * @author OSCA
 *
 */
@SuppressWarnings("unchecked")
public class TimeLimitedHashMap extends HashMap {

	private static final long serialVersionUID = 1L;

	private long limitTime;
	private Map<Object,Date> limitTimeMap;

	/**
	 * <p>デフォルトの初期容量 (16) とデフォルトの負荷係数 (0.75) 、値保持制限時間 (long.MAX_VALUE) で空の TimeLimitedHashMap を作成します。</p>
	 */
	public TimeLimitedHashMap() {
		super();
		this.limitTime    = Long.MAX_VALUE;
		this.limitTimeMap = new HashMap<Object,Date>();
	}

	/**
	 * <p>指定された初期容量とデフォルトの負荷係数 (0.75) 、値保持制限時間 (long.MAX_VALUE) で空の TimeLimitedHashMap を作成します。</p>
	 * @param initialCapacity 初期容量
	 */
	public TimeLimitedHashMap( int initialCapacity ) {
		super(initialCapacity);
		this.limitTime    = Long.MAX_VALUE;
		this.limitTimeMap = new HashMap<Object,Date>();
	}

	/**
	 * <p>指定された初期容量と負荷係数、値保持制限時間 (long.MAX_VALUE) で空の TimeLimitedHashMap を作成します。</p>
	 * @param initialCapacity 初期容量
	 * @param loadFactory 負荷係数
	 */
	public TimeLimitedHashMap( int initialCapacity, float loadFactory ) {
		super(initialCapacity, loadFactory);
		this.limitTime    = Long.MAX_VALUE;
		this.limitTimeMap = new HashMap<Object,Date>();
	}

	/**
	 * <p>指定された Map と同じマッピングで新規 TimeLimitedHashMap を作成します。</p>
	 * @param m マッピングがこのマップに配置されるマップ
	 * @exception NullPointerException 指定されたマップが null の場合
	 */
	public TimeLimitedHashMap( Map<?, ?> m ) {
		super(m);
		this.limitTime    = Long.MAX_VALUE;
		this.limitTimeMap = new HashMap<Object,Date>();
	}

	/**
	 * <p>
	 * 	すべてのマッピングをマップから削除します。
	 * </p>
	 */
	@Override
	public void clear() {
		super.clear();
		this.limitTimeMap.clear();
	}

	/**
	 * <p>
	 * 	マップが指定のキーのマッピングを保持する場合に true を返します。
	 * 	ただし、次の瞬間に時間制限を向かえたオブジェクトが無効になる可能性があることを注意してください。
	 * </p>
	 * @param key マップにあるかどうかが判定されるキー
	 * @return マップが指定のキーのマッピングを保持する場合は true
	 */
	@Override
	public boolean containsKey( Object key ) {
		if( !super.containsKey(key) ) return false ;
		if( this.exceededLimitTime(key) ) return false ;
		return true;
	}

	/**
	 * <p>
	 * 	マップが 1 つまたは複数のキーと指定された値をマッピングしている場合に true を返します。
	 * 	ただし、次の瞬間に時間制限を向かえたオブジェクトが無効になる可能性があることを注意してください。
	 * </p>
	 * @param value マップにあるかどうかを判定される値
	 * @return マップが 1 つまたは複数のキーと指定された値をマッピングしている場合は true
	 */
	@Override
	public boolean containsValue( Object value ) {
		if( !super.containsValue(value) ) return false ;
		for( Iterator iterator = super.entrySet().iterator(); iterator.hasNext(); ) {
			Map.Entry entry = (Map.Entry)iterator.next();
			Object k = entry.getKey();
			Object v = entry.getValue();

			if( value == v && !this.exceededLimitTime(k) ) return true ;
		}
		return false;
	}

	/**
	 * <p>マップに格納されているマッピングのコレクションビューを返します。</p>
	 * @return マップ内に保持されているマッピングのコレクションビュー
	 */
	@Override
	public Set entrySet() {
		this.clearTimeLimitedValues();
		return super.entrySet();
	}

	/**
	 * <p>
	 * 	マップがキーと値のマッピングを保持しない場合に true を返します。
	 * 	時間切れを迎えたオブジェクトも空として扱われます。
	 * </p>
	 * @return マップがキーと値のマッピングを保持しない場合は true
	 */
	@Override
	public boolean isEmpty() {
		this.clearTimeLimitedValues();
		return super.isEmpty();
	}

	/**
	 * <p>
	 * 	マップに格納されているキーのセットビューを返します。
	 * 	時間切れを迎えたオブジェクトのキーは含みません。
	 * </p>
	 * @return マップに含まれているキーのセットビュー
	 */
	@Override
	public Set keySet() {
		this.clearTimeLimitedValues();
		return super.keySet();
	}

	/**
	 * <p>
	 * 	指定の値と指定されたキーをこのマップに関連付けます。
	 * </p>
	 * @param key 指定される値が関連付けられるキー
	 * @param value 指定されるキーに関連付けられる値
	 * @return 指定されたキーに関連した値。または、キーのマッピングがなかった場合は null。戻り値 null は、HashMap が以前に null と指定されたキーを関連付けていたことを示す場合もある
	 */
	@Override
	public Object put( Object key, Object value ) {
		Object object = super.put(key, value);
		this.limitTimeMap.put(key, new Date());
		return object;
	}

	/**
	 * <p>
	 * 	この識別情報ハッシュマップで指定されたキーにマップされている値を返します。
	 * 	値が時間切れを迎えている場合はnullを返します。
	 * </p>
	 * @param key 関連付けられた値が返されるキー
	 * @return マップが、指定されたキーにマッピングしている値。このキーに対するマッピングがマップにない場合は null
	 */
	@Override
	public Object get( Object key ) {
		Object object = super.get(key);
		if( object == null ) return null ;
		if( this.exceededLimitTime(key) ) return null ;
		return object;
	}

	/**
	 * <p>
	 * 	キーに対するマッピングがあれば、そのキーをマップから削除します。
	 * </p>
	 * @param key マッピングがマップから削除されるキー
	 * @return 指定されたキーに関連した値。または、キーのマッピングがなかった場合は null。戻り値 null は、マップが以前に null と指定されたキーを関連付けていたことを示す場合もある
	 */
	@Override
	public Object remove( Object key ) {
		Object object = super.remove(key);
		if( object == null ) return null ;
		if( this.exceededLimitTime(key) ) return null;
		this.limitTimeMap.remove(key);
		return object;
	}

	/**
	 * <p>
	 * 	マップ内のキー値マッピングの数を返します。
	 * 	ただし、次の瞬間に時間制限を向かえたオブジェクトが無効になる可能性があることを注意してください。
	 * </p>
	 * @return マップ内のキー値マッピングの数を返します。
	 */
	@Override
	public int size() {
		this.clearTimeLimitedValues();
		return super.size();
	}

	/**
	 * <p>
	 * 	オブジェクトを保有する制限時間を設定します。
	 * </p>
	 * @param time オブジェクトを保有する制限時間
	 */
	public void setLimitTime( long time ) {
		this.limitTime = time;
	}

	/**
	 * <p>
	 * 	マップに格納されている値のコレクションビューを返します。
	 * </p>
	 * @return マップ内に保持されている値のコレクションビュー
	 */
	@Override
	public Collection values() {
		this.clearTimeLimitedValues();
		return super.values();
	}

	/**
	 * <p>
	 * MEMO : iteratorでループしながらremoveは処理としてまずいので、以下の実装になっている。
	 * iteratorでループしながらremoveすると、iteratorコレクションの中身を壊してしまうため。
	 * </p>
	 */
	private void clearTimeLimitedValues() {
		Object[] keys = super.keySet().toArray();

		for( Object key : keys ) {
			if( this.exceededLimitTime(key) ) {
				this.remove(key);
			}
		}
	}

	/**
	 * 
	 * @param key
	 * @return
	 */
	private boolean exceededLimitTime( Object key ) {
		// 現在時刻を取得
		long now = System.currentTimeMillis();

		// キャッシュした時間を取得
		Date putTime  = this.limitTimeMap.get(key);
		long putTimeL = putTime.getTime();

		if( (now-putTimeL) < (this.limitTime) ) {
			return false;
		}
		return true;
	}

}
