package jp.sf.codememo;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.jface.text.TextViewerUndoManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.part.ViewPart;

/**
 * 
 * @author bskuroneko
 */
public class CodeMemoView extends ViewPart implements ISelectionListener, IMenuListener {

    private TextViewer textViewer;
    private IPartListener2 partListener;
    private volatile int computeCount;
    private List actionHandlers;
    
    public void createPartControl(Composite parent) {
        createTextViewer(parent);
        createListener();
        createContextMenu();
        fillActionBars(getViewSite().getActionBars());
    }

    private void createTextViewer(Composite parent) {
        textViewer = new TextViewer(parent, SWT.H_SCROLL | SWT.V_SCROLL);
        textViewer.setEditable(false);
        
        IUndoManager undoManager = new TextViewerUndoManager(100);
        textViewer.setUndoManager(undoManager);
        undoManager.connect(textViewer);
    }

    private void createListener() {
        partListener = new IPartListener2(){
            public void partVisible(IWorkbenchPartReference partRef) {
                if (isMyPart(partRef)) {
                    IWorkbenchPart activePart= partRef.getPage().getActivePart();
                    if (activePart != null) {
                        selectionChanged(activePart, partRef.getPage().getSelection());
                    }
                    startListeningForSelectionChanges();
                }
            }
            
            public void partHidden(IWorkbenchPartReference partRef) {
                if (isMyPart(partRef)) {
                    stopListeningForSelectionChanges();
                }
            }

            public void partInputChanged(IWorkbenchPartReference partRef) {
                if (!isMyPart(partRef)) {
                    computeAndSetInput(partRef.getPart(false));
                }
            }

            public void partActivated(IWorkbenchPartReference partRef) {
            }
            public void partBroughtToTop(IWorkbenchPartReference partRef) {
            }
            public void partClosed(IWorkbenchPartReference partRef) {
            }
            public void partDeactivated(IWorkbenchPartReference partRef) {
            }
            public void partOpened(IWorkbenchPartReference partRef) {
            }
            
            private boolean isMyPart(IWorkbenchPartReference partRef) {
                return partRef.getId().equals(getSite().getId());
            }
        };
        getSite().getWorkbenchWindow().getPartService().addPartListener(partListener);
    }

    private void startListeningForSelectionChanges() {
        getSite().getPage().addPostSelectionListener(this);
    }

    private void stopListeningForSelectionChanges() {
        getSite().getPage().removePostSelectionListener(this);
    }

    private void createContextMenu() {
        MenuManager menuManager= new MenuManager("#PopupMenu"); //$NON-NLS-1$
        menuManager.setRemoveAllWhenShown(true);
        menuManager.addMenuListener(this);
        Menu contextMenu= menuManager.createContextMenu(getControl());
        getControl().setMenu(contextMenu);
        getSite().registerContextMenu(menuManager, getViewSite().getSelectionProvider());
    }

    protected void fillActionBars(IActionBars actionBars) {
        for (Iterator it = getGlobalActionHandlers().iterator(); it.hasNext();) {
            IAction actionHandler = (IAction) it.next();
            actionBars.setGlobalActionHandler(actionHandler.getId(), actionHandler);
        }
    }

    private Control getControl() {
        return textViewer.getControl();
    }

    public void setFocus() {
        getControl().setFocus();
    }

    public void selectionChanged(IWorkbenchPart part, ISelection selection) {
        if (part.equals(this)) {
            return;
        }
        computeAndSetInput(part);
    }

    /*
     * @see IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
     */
    public void menuAboutToShow(IMenuManager menu) {
        Set groups = new HashSet();
        for (Iterator it = getGlobalActionHandlers().iterator(); it.hasNext();) {
            TextViewerAction action = (TextViewerAction) it.next();
            if (!groups.contains(action.getGroupName())) {
                menu.add(new Separator(action.getGroupName()));
            }
            menu.appendToGroup(action.getGroupName(), action);
        }
    }

    private Collection getGlobalActionHandlers() {
        if (actionHandlers != null) {
            return actionHandlers;
        }
        actionHandlers = new ArrayList();
        actionHandlers.add(TextViewerAction.createUndoAction(textViewer, "undo"));
        actionHandlers.add(TextViewerAction.createRedoAction(textViewer, "undo"));
        actionHandlers.add(TextViewerAction.createCutAction(textViewer, "edit"));
        actionHandlers.add(TextViewerAction.createCopyAction(textViewer, "edit"));
        actionHandlers.add(TextViewerAction.createPasteAction(textViewer, "edit"));
        actionHandlers.add(TextViewerAction.createSelectAllAction(textViewer, "select"));
        return actionHandlers;
    }
    
    final public void dispose() {
        computeCount++;
        getSite().getWorkbenchWindow().getPartService().removePartListener(partListener);
        textViewer = null;
    }

    private void computeAndSetInput(final IWorkbenchPart part) {
        final int currentCount= ++computeCount;

        ISelectionProvider provider= part.getSite().getSelectionProvider();
        if (provider == null) {
            return;
        }

        final ISelection selection= provider.getSelection();
        if (selection == null || selection.isEmpty()) {
            return;
        }

        Thread thread= new Thread("CodeMemo view input computer") { //$NON-NLS-1$
            public void run() {
                if (currentCount != computeCount) {
                    return;
                }
                
                final Object element = Activator.getDefault().findSupportedElement(part, selection);
                if (isIgnoringNewElement(element)) {
                    return;
                }
                
                Shell shell= getSite().getShell();
                if (shell.isDisposed()) {
                    return;
                }

                Display display= shell.getDisplay();
                if (display.isDisposed()) {
                    return;
                }

                display.asyncExec(new Runnable() {
                    /*
                     * @see java.lang.Runnable#run()
                     */
                    public void run() {

                        if (computeCount != currentCount || getViewSite().getShell().isDisposed()) {
                            return;
                        }

                        Activator.getDefault().changeElement(element);
                        updateView();
                    }
                });
            }
        };

        thread.setDaemon(true);
        thread.setPriority(Thread.MIN_PRIORITY);
        thread.start();
    }

    private boolean isIgnoringNewElement(Object element) {
        if (element == null) {
            return true;
        }
        return Activator.getDefault().getCurrentElement() != null && Activator.getDefault().getCurrentElement().equals(element);
    }
    
    private void updateView() {
        String label = Activator.getDefault().getCurrentElementLabel();
        setContentDescription(label);
        setTitleToolTip(label);
        
        textViewer.setEditable(true);
        textViewer.setDocument(Activator.getDefault().getCurrentMemo());
    }
}
