package jp.sf.codememo;

import org.eclipse.core.commands.operations.IOperationHistoryListener;
import org.eclipse.core.commands.operations.OperationHistoryEvent;
import org.eclipse.core.commands.operations.OperationHistoryFactory;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextInputListener;
import org.eclipse.jface.text.ITextListener;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.TextEvent;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;

/**
 * 
 * @author bskuroneko
 */
public class TextViewerAction extends Action {

    private TextViewer textViewer;
    private int operation;
    private String groupName;

    public TextViewerAction(TextViewer textViewer, int operation, ActionFactory actionFactory, String groupName) {
        this.textViewer = textViewer;
        this.operation = operation;
        
        initialize(actionFactory);
        
        addChangedListeners();
        refreshEnabled();
        this.groupName = groupName;
    }
    
    protected void initialize(ActionFactory actionFactory) {
        IWorkbenchAction action = actionFactory.create(PlatformUI.getWorkbench().getActiveWorkbenchWindow());
        setId(action.getId());
        setDescription(action.getActionDefinitionId());
        setText(action.getText());
        setToolTipText(action.getToolTipText());
        setImageDescriptor(action.getImageDescriptor());
        setDisabledImageDescriptor(action.getDisabledImageDescriptor());
        action.dispose();
    }
    
    protected void addChangedListeners() {
        textViewer.addSelectionChangedListener(new ISelectionChangedListener(){
            public void selectionChanged(SelectionChangedEvent event) {
                refreshEnabled();
            }
        });
        
        textViewer.addTextInputListener(new ITextInputListener(){
            public void inputDocumentAboutToBeChanged(IDocument oldInput, IDocument newInput) {
                refreshEnabled();
            }
            
            public void inputDocumentChanged(IDocument oldInput, IDocument newInput) {
                refreshEnabled();
            }
        });
        
        textViewer.addTextListener(new ITextListener(){
            public void textChanged(TextEvent event) {
                refreshEnabled();
            }
        });
        
        OperationHistoryFactory.getOperationHistory().addOperationHistoryListener(new IOperationHistoryListener(){
            public void historyNotification(OperationHistoryEvent event) {
                refreshEnabled();
            }
        });
    }

    protected void refreshEnabled() {
        setEnabled(textViewer.canDoOperation(operation));
    }
    
    public void run() {
        textViewer.doOperation(operation);
    }
    
    public String getGroupName() {
        return groupName;
    }

    public static TextViewerAction createCutAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.CUT, ActionFactory.CUT, groupName);
    }
    
    public static TextViewerAction createCopyAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.COPY, ActionFactory.COPY, groupName);
    }
    
    public static TextViewerAction createPasteAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.PASTE, ActionFactory.PASTE, groupName);
    }
    
    public static TextViewerAction createDeleteAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.DELETE, ActionFactory.DELETE, groupName);
    }
    
    public static TextViewerAction createUndoAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.UNDO, ActionFactory.UNDO, groupName);
    }
    
    public static TextViewerAction createRedoAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.REDO, ActionFactory.REDO, groupName);
    }
    
    public static TextViewerAction createSelectAllAction(TextViewer textViewer, String groupName) {
        return new TextViewerAction(textViewer, ITextOperationTarget.SELECT_ALL, ActionFactory.SELECT_ALL, groupName);
    }
}
