/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2007 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software, you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY, without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program, if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 3600 Bridge Parkway #102, Redwood City, CA 94065, USA      *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.model;

import java.net.*;
import java.sql.*;
import java.util.logging.*;
import org.compiere.util.*;

/**
 *	Session Model.
 *	Maintained in AMenu.
 *	
 *  @author Jorg Janke
 *  @version $Id: MSession.java,v 1.1 2008/07/29 16:04:18 jrmt Exp $
 */
public class MSession extends X_AD_Session
{
	/**
	 * 	Get existing or create local session
	 *	@param ctx context
	 *	@param createNew create if not found
	 *	@return session session
	 */
	public static MSession get (Ctx ctx, boolean createNew)
	{
		int AD_Session_ID = ctx.getContextAsInt("#AD_Session_ID");
		MSession session = null;
		if (AD_Session_ID > 0)
			session = (MSession)s_sessions.get(new Integer(AD_Session_ID));
		if (session == null && createNew)
		{
			session = new MSession (ctx, null);	//	local session
			session.save();
			AD_Session_ID = session.getAD_Session_ID();
			ctx.setContext("#AD_Session_ID", AD_Session_ID);
			s_sessions.put (new Integer(AD_Session_ID), session);
		}	
		return session;
	}	//	get
	
	/**
	 * 	Get existing or create remote session
	 *	@param ctx context
	 *	@param Remote_Addr remote address
	 *	@param Remote_Host remote host
	 *	@param WebSession web session
	 *	@return session
	 */
	public static MSession get (Ctx ctx, String Remote_Addr, String Remote_Host, String WebSession)
	{
		int AD_Session_ID = ctx.getContextAsInt("#AD_Session_ID");
		MSession session = null;
		if (AD_Session_ID > 0)
			session = (MSession)s_sessions.get(new Integer(AD_Session_ID));
		if (session == null)
		{
			session = new MSession (ctx, Remote_Addr, Remote_Host, WebSession, null);	//	remote session
			session.save();
			AD_Session_ID = session.getAD_Session_ID();
			ctx.setContext("#AD_Session_ID", AD_Session_ID);
			s_sessions.put(new Integer(AD_Session_ID), session);
		}	
		return session;
	}	//	get

	/**	Sessions					*/
	private static CCache<Integer, MSession> s_sessions = Ini.isClient() 
		? new CCache<Integer, MSession>("AD_Session_ID", 1, 0)		//	one client session 
		: new CCache<Integer, MSession>("AD_Session_ID", 30, 0);	//	no time-out	
	
	
	/**************************************************************************
	 * 	Standard Constructor
	 *	@param ctx context
	 *	@param AD_Session_ID id
	 *	@param trxName transaction
	 */
	public MSession (Ctx ctx, int AD_Session_ID, String trxName)
	{
		super(ctx, AD_Session_ID, trxName);
		if (AD_Session_ID == 0)
		{
			setProcessed (false);
			int AD_Role_ID = ctx.getAD_Role_ID();
			setAD_Role_ID(AD_Role_ID);
		}
	}	//	MSession

	/**
	 * 	Load Constructor
	 *	@param ctx context
	 *	@param rs result set
	 *	@param trxName transaction
	 */
	public MSession(Ctx ctx, ResultSet rs, String trxName)
	{
		super(ctx, rs, trxName);
	}	//	MSession

	/**
	 * 	New (remote) Constructor
	 *	@param ctx context
	 *	@param Remote_Addr remote address
	 *	@param Remote_Host remote host
	 *	@param WebSession web session
	 *	@param trxName transaction
	 */
	public MSession (Ctx ctx, String Remote_Addr, String Remote_Host, String WebSession, String trxName)
	{
		this (ctx, 0, trxName);
		if (Remote_Addr != null)
			setRemote_Addr(Remote_Addr);
		if (Remote_Host != null)
			setRemote_Host(Remote_Host);
		if (WebSession != null)
			setWebSession(WebSession);
	}	//	MSession

	/**
	 * 	New (local) Constructor
	 *	@param ctx context
	 *	@param trxName transaction
	 */
	public MSession (Ctx ctx, String trxName)
	{
		this (ctx, 0, trxName);
		try
		{
			InetAddress lh = InetAddress.getLocalHost();
			setRemote_Addr(lh.getHostAddress());
			setRemote_Host(lh.getHostName());
		}
		catch (UnknownHostException e)
		{
			log.log(Level.SEVERE, "No Local Host", e);
		}
	}	//	MSession

	/**	Web Store Session		*/
	private boolean		m_webStoreSession = false;
	
	/**
	 * 	Is it a Web Store Session
	 *	@return Returns true if Web Store Session.
	 */
	public boolean isWebStoreSession ()
	{
		return m_webStoreSession;
	}	//	isWebStoreSession
	
	/**
	 * 	Set Web Store Session
	 *	@param webStoreSession The webStoreSession to set.
	 */
	public void setWebStoreSession (boolean webStoreSession)
	{
		m_webStoreSession = webStoreSession;
	}	//	setWebStoreSession
	
	/**
	 * 	String Representation
	 *	@return info
	 */
	public String toString()
	{
		StringBuffer sb = new StringBuffer("MSession[")
			.append(getAD_Session_ID())
			.append(",AD_User_ID=").append(getCreatedBy())
			.append(",").append(getCreated())
			.append(",Remote=").append(getRemote_Addr());
		String s = getRemote_Host();
		if (s != null && s.length() > 0)
			sb.append(",").append(s);
		if (m_webStoreSession)
			sb.append(",WebStoreSession");
		sb.append("]");
		return sb.toString();
	}	//	toString

	/**
	 * 	Session Logout
	 */
	public void logout()
	{
		setProcessed(true);
		save();
		s_sessions.remove(new Integer(getAD_Session_ID()));
		log.info(TimeUtil.formatElapsed(getCreated(), getUpdated()));
	}	//	logout


	/**
	 * 	Create Change Log only if table is logged
	 * 	@param TrxName transaction name
	 *	@param AD_ChangeLog_ID 0 for new change log
	 *	@param AD_Table_ID table
	 *	@param AD_Column_ID column
	 *	@param keyInfo key value(s)
	 *	@param AD_Client_ID client
	 *	@param AD_Org_ID org
	 *	@param OldValue old
	 *	@param NewValue new
	 *	@return saved change log or null
	 */
	public MChangeLog changeLog (String TrxName, int AD_ChangeLog_ID,
		int AD_Table_ID, int AD_Column_ID, Object keyInfo,
		int AD_Client_ID, int AD_Org_ID,
		Object OldValue, Object NewValue, 
		String tableName, String type)
	{
		//	Null handling
		if (OldValue == null && NewValue == null)
			return null;
		//	Equal Value
		if (OldValue != null && NewValue != null && OldValue.equals(NewValue))
			return null;

		//	No Log
		if (MChangeLog.isNotLogged(AD_Table_ID, tableName, AD_Column_ID, type))
			return null;
		
		//	Role Logging
		MRole role = MRole.getDefault(getCtx(), false);
		//	Do we need to log
		if (m_webStoreSession						//	log if WebStore
			|| MChangeLog.isLogged(AD_Table_ID, type)		//	im/explicit log
			|| (role != null && role.isChangeLog()))//	Role Logging
			;
		else
			return null;
		//
		log.finest("AD_ChangeLog_ID=" + AD_ChangeLog_ID
				+ ", AD_Session_ID=" + getAD_Session_ID()
				+ ", AD_Table_ID=" + AD_Table_ID + ", AD_Column_ID=" + AD_Column_ID
				+ ": " + OldValue + " -> " + NewValue);
		boolean success = false;
		
		try
		{
			MChangeLog cl = new MChangeLog(getCtx(), 
				AD_ChangeLog_ID, TrxName, getAD_Session_ID(),
				AD_Table_ID, AD_Column_ID, keyInfo, AD_Client_ID, AD_Org_ID,
				OldValue, NewValue);
			if (cl.save())
				return cl;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "AD_ChangeLog_ID=" + AD_ChangeLog_ID
				+ ", AD_Session_ID=" + getAD_Session_ID()
				+ ", AD_Table_ID=" + AD_Table_ID + ", AD_Column_ID=" + AD_Column_ID, e);
			return null;
		}
		log.log(Level.SEVERE, "AD_ChangeLog_ID=" + AD_ChangeLog_ID
			+ ", AD_Session_ID=" + getAD_Session_ID()
			+ ", AD_Table_ID=" + AD_Table_ID + ", AD_Column_ID=" + AD_Column_ID);
		return null;
	}	//	changeLog

	/**
	 * 	Create Query Log
	 *	@param AD_Client_ID login client
	 *	@param AD_Org_ID login org
	 *	@param AD_Table_ID table
	 *	@param whereClause where
	 *	@param recordCount records
	 *	@param parameter parameter
	 *	@return Log
	 */
	public MQueryLog queryLog(int AD_Client_ID, int AD_Org_ID,
		int AD_Table_ID, String whereClause, int recordCount, String parameter)
	{
		MQueryLog qlog = null;
		try
		{
			qlog = new MQueryLog(getCtx(), getAD_Session_ID(),  
				AD_Client_ID, AD_Org_ID,
				AD_Table_ID, whereClause, recordCount, parameter);
			qlog.save();
		}
		catch (Exception e) 
		{
			log.log(Level.SEVERE, "AD_Session_ID=" + getAD_Session_ID()
				+ ", AD_Table_ID=" + AD_Table_ID + ", Where=" + whereClause
				, e);
		}
		return qlog;
	}	//	queryLog

	/**
	 * 	Create Query Log
	 *	@param AD_Client_ID login client
	 *	@param AD_Org_ID login org
	 *	@param AD_Table_ID table
	 *	@param whereClause where
	 *	@param recordCount records
	 *	@return Log
	 */
	public MQueryLog queryLog(int AD_Client_ID, int AD_Org_ID,
		int AD_Table_ID, String whereClause, int recordCount)
	{
		return queryLog(AD_Client_ID, AD_Org_ID, AD_Table_ID, 
			whereClause, recordCount, (String)null);
	}	//	queryLog
	
	/**
	 * 	Create Query Log
	 *	@param AD_Client_ID login client
	 *	@param AD_Org_ID login org
	 *	@param AD_Table_ID table
	 *	@param whereClause where
	 *	@param recordCount records
	 *	@param parameter parameter
	 *	@return Log
	 */
	public MQueryLog queryLog(int AD_Client_ID, int AD_Org_ID,
		int AD_Table_ID, String whereClause, int recordCount, Object parameter)
	{
		String para = null;
		if (parameter != null)
			para = parameter.toString();
		return queryLog(AD_Client_ID, AD_Org_ID, AD_Table_ID, 
			whereClause, recordCount, para);
	}	//	queryLog

	/**
	 * 	Create Query Log
	 *	@param AD_Client_ID login client
	 *	@param AD_Org_ID login org
	 *	@param AD_Table_ID table
	 *	@param whereClause where
	 *	@param recordCount records
	 *	@param parameters parameter
	 *	@return Log
	 */
	public MQueryLog queryLog(int AD_Client_ID, int AD_Org_ID,
		int AD_Table_ID, String whereClause, int recordCount, Object[] parameters)
	{
		String para = null;
		if (parameters != null && parameters.length > 0)
		{
			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < parameters.length; i++)
			{
				if (i > 0)
					sb.append(", ");
				if (parameters[i] == null)
					sb.append("NULL");
				else
					sb.append(parameters[i].toString());
			}
			para = sb.toString();
		}
		return queryLog(AD_Client_ID, AD_Org_ID, AD_Table_ID, 
			whereClause, recordCount, para);
	}	//	queryLog

	/**
	 * 	Create Window Log
	 *	@param AD_Client_ID client
	 *	@param AD_Org_ID org
	 *	@param AD_Window_ID window
	 *	@param AD_Form_ID form
	 *	@return Log
	 */
	public MWindowLog windowLog(int AD_Client_ID, int AD_Org_ID,
		int AD_Window_ID, int AD_Form_ID)
	{
		MWindowLog wlog = null;
		try
		{
			wlog = new MWindowLog(getCtx(), getAD_Session_ID(),  
				AD_Client_ID, AD_Org_ID,
				AD_Window_ID, AD_Form_ID);
			wlog.save();
		}
		catch (Exception e) 
		{
			log.log(Level.SEVERE, "AD_Session_ID=" + getAD_Session_ID()
				+ ", AD_Window_ID=" + AD_Window_ID + ", AD_Form_ID=" + AD_Form_ID
				, e);
		}
		return wlog;
	}	//	windowLog
	
}	//	MSession

