﻿module coneneko.camera;
private import coneneko.math;
private import std.math;
private import std.cstream;

/*
注視点が基準
ヘッド、左右の回転
ピッチ、上下の回転
バンク、upのzに対する回転
ズーム
これらからカメラ位置を計算できる
*/

///
interface _Camera
{
	void tick(float x, float y); ///
	Matrix getViewMatrix(); ///
	void useLeftDrag(bit a); ///
	void pushLeft(); ///
	void popLeft(); ///
	void pushRightAtCenter(); ///
	void pushRight(); ///
	void popRight(); ///
	void distance(float a); ///
}

///
class Camera : _Camera
{
	unittest // 中央付近以外の右ドラッグ
	{
		Camera camera = new Camera();
		float d, rx, ry, rup;
		Vector at;
		camera.tick(-1, -1);
		camera.revision(d, rx, ry, rup, at);
		assert(PI * 2 * 0 == rx);
		assert(PI * 2 * 0 == ry);
		camera.pushRight();
		camera.tick(0, 0);
		camera.revision(d, rx, ry, rup, at);
		assert(fabs(-PI * 2 * move - rx) < 0.00001);
		assert(fabs(-PI * 2 * move - ry) < 0.00001);
		camera.tick(1, 1);
		camera.popRight();
		camera.revision(d, rx, ry, rup, at);
		assert(fabs(-PI_2 - rx) < 0.00001);
		assert(fabs(-PI * 2 * 2 * move - ry) < 0.00001);
	}
	
	unittest // 両ドラッグ
	{
		Camera camera = new Camera();
		float distance = camera._distance;
		float d, rx, ry, rup;
		Vector at;
		camera.tick(-1, -1);
		camera.revision(d, rx, ry, rup, at);
		assert(distance == d);
		assert(0.0 == rup);
		camera.pushLeft();
		camera.pushRight();
		camera.tick(0, 0);
		camera.revision(d, rx, ry, rup, at);
		assert(fabs((distance - move * camera._dmove) - d)  < 0.00001);
		assert(fabs(-PI * move - rup) < 0.00001);
		camera.tick(1, 1);
		camera.popLeft();
		camera.popRight();
		camera.revision(d, rx, ry, rup, at);
		assert(fabs((distance - move * camera._dmove * 2) - d) < 0.00001);
		assert(fabs(-PI * move * 2 - rup) < 0.00001);
	}
	
	unittest // 中央付近の右ドラッグ
	{
		Camera camera = new Camera();
		float d, rx, ry, rup;
		Vector at;
		camera.tick(0, 0);
		camera.revision(d, rx, ry, rup, at);
		assert(fabs(0.0 - at.x) < 0.00001
			&& fabs(0.0 - at.y) < 0.00001);
		camera.pushRightAtCenter();
		camera.tick(0.5, 0.5);
		camera.revision(d, rx, ry, rup, at);
		assert(fabs(0.5 - at.x) < 0.00001
			&& fabs(0.5 - at.y) < 0.00001);
		camera.tick(1, 1);
		camera.popRight();
		camera.revision(d, rx, ry, rup, at);
		assert(fabs(1.0 - at.x) < 0.00001
			&& fabs(1.0 - at.y) < 0.00001);
	}
	
	/*
	unittest // 左ドラッグ
	{
		bit eq(float a, float b)
		{
			return fabs(a - b) < 0.00001 ? true : false;
		}
		
		Camera camera = new Camera();
		camera.useLeftDrag = true;
		float d, rx, ry, rup;
		Vector at;
		camera.tick(0, 0);
		camera.revision(d, rx, ry, rup, at);
		assert(eq(200.0, d));
		assert(eq(0.0, rx));
		assert(eq(0.0, ry));
		assert(eq(0.0, at.x));
		assert(eq(0.0, at.y));
		camera.pushLeft();
		
		assert(false);
	}
	*/
	
	///
	this()
	{
		revision = &defaultState;
		atPoint = Vector.create(0, 0, 0);
	}
	
	private void delegate(out float d,
		out float rx, out float ry, out float rup, out Vector at) revision;
	
	private float _distance = 500.0;
	private float _dmove = 200.0;
	void distance(float a) { _distance = a; _dmove = a * 0.5; }
	private float radianX = 0, radianY = 0, radianUp = 0;
	private Vector atPoint;
	void dmove(float a) { _dmove = a; } ///
	
	private float currentX = 0, currentY = 0;
	private float baseX = 0, baseY = 0; // ドラッグ開始マウス位置

	public void tick(float x, float y)
	{
		currentX = x;
		currentY = y;
	}
	
	public Matrix getViewMatrix()
	{
		float d, rx, ry, rup;
		Vector at;
		revision(d, rx, ry, rup, at);
		Matrix m = Matrix.translation(0, 0, d)
			* Matrix.rotationX(-rx)
			* Matrix.rotationY(ry)
			* Matrix.translation(at.x, at.y, at.z);
		Vector v = mul3(Vector.create(0, 0, 0), m);
		
		return Matrix.rotationZ(rup)
			* Matrix.lookAtRH(
				v.x, v.y, v.z,
				at.x, at.y, at.z,
				0, 1, 0
			);
	}
	
	private void defaultState(out float d,
		out float rx, out float ry, out float rup, out Vector at)
	{
		d = _distance;
		rx = radianX;
		ry = radianY;
		rup = radianUp;
		at = atPoint;
	}
	
	private void rightState(out float d,
		out float rx, out float ry, out float rup, out Vector at)
	{
		d = _distance;
		rx = radianX - (currentY - baseY) * PI * 2 * move;
		rx = clamp(rx, -PI_2 + 0.00001, PI_2 - 0.00001);
		ry = radianY - (currentX - baseX) * PI * 2 * move;
		rup = radianUp;
		at = atPoint;
	}
	
	private void bothState(out float d,
		out float rx, out float ry, out float rup, out Vector at)
	{
		d = _distance - (currentY - baseY) * move * _dmove;
		d = clamp(d, 0.01, 2000.0);
		rx = radianX;
		ry = radianY;
		rup = radianUp - (currentX - baseX) * PI * move;
		at = atPoint;
	}
	
	private void rightCenterState(out float d,
		out float rx, out float ry, out float rup, out Vector at)
	{
		d = _distance;
		rx = radianX;
		ry = radianY;
		rup = radianUp;

		float y = sin(-rx) * d;
		float z = -cos(-rx) * d;
		float x = cos(-PI_2 - ry) * z;
		z = sin(-PI_2 - ry) * z;
		Vector position = Vector.create(x, y, z);
		Vector up = Vector.create(0, 1, 0);
		assert(atPoint != position);
		Vector v = normalize(atPoint - position);

		at = atPoint
			+ up * (currentY - baseY)
			+ cross(v, up) * (currentX - baseX);
	}

	private void leftState(out float d,
		out float rx, out float ry, out float rup, out Vector at)
	{
		d = _distance;
		rx = radianX;
		ry = radianY;
		rup = radianUp;
		if (_useLeftDrag)
		{
			/*
			at = atPoint
				- Vector.create(1, 0, 0) * (currentX - baseX) * _dmove
				- Vector.create(0, 1, 0) * (currentY - baseY) * _dmove;
			*/
			float dx = (currentX - baseX) * _dmove * 0.1;
			float dy = (currentY - baseY) * _dmove * 0.1;
			
			Matrix m = Matrix.translation(0, 0, 1.0)
				* Matrix.rotationX(-radianX)
				* Matrix.rotationY(radianY);
			Vector cameraPosition = mul3(vector(0, 0, 0), m);
			
			Vector dirAt = vector(0, 0, 0) - cameraPosition; // camera -> at
			Vector dirUp = vector(0, 1, 0) - cameraPosition; // camera -> up
			assert(0.0 != length(dirAt));
			assert(0.0 != length(dirUp));
			dirAt = normalize(dirAt);
			dirUp = normalize(dirUp);
			
			Vector dirX = cross(dirAt, dirUp);
			Vector dirY = -cross(dirAt, dirX);
			assert(0.0 != length(dirX));
			assert(0.0 != length(dirY));
			at = atPoint - normalize(dirX) * dx - normalize(dirY) * dy;
		}
		else at = atPoint;
	}
	private bit _useLeftDrag = true;
	void useLeftDrag(bit a) { _useLeftDrag = a; }
	
	const float move = 0.25; // mouseの移動量の調整
	
	public void pushLeft()
	in
	{
		//assert(&defaultState == revision
		//	|| &rightState == revision
		//	|| &rightCenterState == revision);
		//derr.writeLine("pushLeft");
	}
	out
	{
		assert(&leftState == revision
			|| &bothState == revision);
	}
	body
	{
		float d, rx, ry, rup;
		Vector at;
		revision(d, rx, ry, rup, at);
		if (revision == &defaultState)
		{
			revision = &leftState;
		}
		else if (revision == &rightState)
		{
			radianX = rx;
			radianY = ry;
			revision = &bothState;
		}
		else if (revision == &rightCenterState)
		{
			atPoint = at;
			revision = &bothState;
		}
		baseX = currentX;
		baseY = currentY;
	}
	
	public void popLeft()
	in
	{
		//assert(&leftState == revision
		//	|| &bothState == revision);
	}
	out
	{
		assert(&defaultState == revision
			|| &rightState == revision);
	}
	body
	{
		float d, rx, ry, rup;
		Vector at;
		revision(d, rx, ry, rup, at);
		if (&leftState == revision)
		{
			atPoint = at;
			revision = &defaultState;
		}
		else if (&bothState == revision)
		{
			_distance = d;
			radianUp = rup;
			revision = &rightState;
		}
		baseX = currentX;
		baseY = currentY;
	}
	
	public void pushRightAtCenter()
	in
	{
		assert(&defaultState == revision
			|| &leftState == revision);
	}
	out
	{
		assert(&rightCenterState == revision
			|| &bothState == revision);
	}
	body
	{
		if (revision == &leftState)
		{
			pushRight();
			return;
		}
		revision = &rightCenterState;
		baseX = currentX;
		baseY = currentY;
	}

	public void pushRight()
	in
	{
		//assert(&defaultState == revision
		//	|| &leftState == revision);
		//derr.writeLine("pushRight");
	}
	out
	{
		assert(&rightState == revision
			|| &bothState == revision);
	}
	body
	{
		float d, rx, ry, rup;
		Vector at;
		revision(d, rx, ry, rup, at);
		if (revision == &defaultState)
		{
			revision = &rightState;
		}
		else if (revision == &leftState)
		{
			atPoint = at;
			revision = &bothState;
		}
		baseX = currentX;
		baseY = currentY;
	}
	
	public void popRight()
	in
	{
		assert(&rightState == revision
			|| &rightCenterState == revision
			|| &bothState == revision);
	}
	out
	{
		assert(&defaultState == revision
			|| &leftState == revision);
	}
	body
	{
		float d, rx, ry, rup;
		Vector at;
		revision(d, rx, ry, rup, at);
		if (revision == &rightState)
		{
			radianX = rx;
			radianY = ry;
			revision = &defaultState;
		}
		else if (revision == &bothState)
		{
			_distance = d;
			radianUp = rup;
			revision = &leftState;
		}
		else if (revision == &rightCenterState)
		{
			atPoint = at;
			revision = &defaultState;
		}
		baseX = currentX;
		baseY = currentY;
	}
}
