﻿module coneneko.exunits;
import coneneko.scenegraph, coneneko.units, coneneko.math, coneneko.nanami, coneneko.texture;

///
class MatrixIndexBuffer : AttributeBuffer
{
	/// float2
	this(float[] data)
	{
		super(data, 2);
	}
}

///
class WeightBuffer : AttributeBuffer
{
	/// float1
	this(float[] data)
	{
		super(data, 1);
	}
}

///
class PcntBuffer : UnitMerger
{
	private const uint P_SIZE = 3, C_SIZE = 4, N_SIZE = 3, T_SIZE = 2;
	
	///
	const uint VERTEX_SIZE = 12;
	
	///
	this(uint vertexLength, float[] buffer)
	{
		super(createPcnt(vertexLength, buffer));
	}
	
	///
	protected static Unit createPcnt(uint vertexLength, float[] buffer)
	{
		if (buffer.length != VERTEX_SIZE * vertexLength)
		{
			exception(__FILE__, __LINE__, null, "PcntBuffer size");
		}
		
		uint i = 0;
		PositionBuffer pb = new PositionBuffer(buffer[i..i + P_SIZE * vertexLength]);
		i += P_SIZE * vertexLength;
		ColorBuffer cb = new ColorBuffer(buffer[i..i + C_SIZE * vertexLength]);
		i += C_SIZE * vertexLength;
		NormalBuffer nb = new NormalBuffer(buffer[i..i + N_SIZE * vertexLength]);
		i += N_SIZE * vertexLength;
		TexCoordBuffer tb = new TexCoordBuffer(buffer[i..i + T_SIZE * vertexLength]);
		assert(buffer.length == i + T_SIZE * vertexLength);
		
		return new UnitMerger(tb, nb, cb, pb);
	}
	
	///
	protected this(Unit[] a ...)
	{
		super(a);
	}
}

///
class PcntmwBuffer : PcntBuffer
{
	private const uint MI_SIZE = 2, W_SIZE = 1;
	private final MatrixIndexBuffer miBuffer;
	private final WeightBuffer wBuffer;
	
	///
	const uint VERTEX_SIZE = 15;
	
	///
	this(uint vertexLength, float[] buffer)
	{
		glErrorCheck(__FILE__, __LINE__);
		
		if (buffer.length != VERTEX_SIZE * vertexLength)
		{
			exception(__FILE__, __LINE__, this, "PcntmwBuffer size");
		}
		
		uint i = PcntBuffer.VERTEX_SIZE * vertexLength;
		miBuffer = new MatrixIndexBuffer(buffer[i..i + MI_SIZE * vertexLength]);
		i += MI_SIZE * vertexLength;
		wBuffer = new WeightBuffer(buffer[i..i + W_SIZE * vertexLength]);
		assert(buffer.length == i + W_SIZE * vertexLength);
		
		super(
			new UnitMerger(wBuffer, miBuffer),
			createPcnt(vertexLength, buffer[0..PcntBuffer.VERTEX_SIZE * vertexLength])
		);
		glErrorCheck(__FILE__, __LINE__);
	}
	
	void matrixIndexLocation(uint a) ///
	{
		miBuffer.location = a;
	}
	
	void weightLocation(uint a) ///
	{
		wBuffer.location = a;
	}
}

/// uniform vec4 lightDirection; // vertex
/// uniform vec4 ambient; // vertex
/// uniform sampler2D texture0, texture1; // fragment
class ToonShaderBase : Shader
{
	private final Texture texture1;
	
	this(char[] vertexSource, char[] fragmentSource, Texture texture1)
	{
		this.texture1 = texture1;
		super(vertexSource, fragmentSource);
		setInt("texture0", 0);
		setInt("texture1", 1);
		lightDirection = vector(-1.0, -1.0, 0.0);
		ambient = vector(1.0, 1.0, 1.0);
	}
	
	void lightDirection(Vector a) ///
	{
		if (length(a) == 0.0) a = vector(0.0, -1.0, 0.0);
		setVector("lightDirection", normalize(a));
	}
	
	void ambient(Vector a) ///
	{
		setVector("ambient", clamp(a, 0.0, 1.0));
	}
	
	override void attach()
	{
		texture1.attach();
		super.attach();
	}
	
	override void detach()
	{
		super.detach();
		texture1.detach();
		glErrorCheck(__FILE__, __LINE__);
	}
	
	uint matrixIndexLocation() /// attribute vec2 matrixIndex; // vertexスキンメッシュ用
	{
		return getAttributeLocation("matrixIndex");
	}
	
	uint weightLocation() /// attribute float weight; // vertexスキンメッシュ用
	{
		return getAttributeLocation("weight");
	}
	
	void matrixArray(Matrix[] a) /// uniform mat4 matrixArray[34]; // vertexスキンメッシュ用
	{
		setMatrixArray("matrixArray", a);
	}

///
const char[] TOON_VERTEX = "
uniform vec4 lightDirection;
uniform vec4 ambient;
float getTexCoord1X()
{
	return clamp(dot(gl_NormalMatrix * gl_Normal, lightDirection.xyz), 0.1, 0.9);
}
void main()
{
	gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;
	gl_FrontColor = vec4(gl_Color.rgb * ambient.rgb, gl_Color.a);
	gl_TexCoord[0] = gl_MultiTexCoord0;
	gl_TexCoord[1].x = getTexCoord1X();
}
";

///
const char[] TOON_FRAGMENT = "
uniform sampler2D texture0, texture1;
void main()
{
	gl_FragColor = gl_Color
		* texture2D(texture0, gl_TexCoord[0].xy)
		* texture2D(texture1, gl_TexCoord[1].xy);
}
";

///
const char[] HSV_TOON_FRAGMENT = "
uniform sampler2D texture0, texture1;
vec3 toHsv(vec3 rgb)
{
	vec3 result;
	
	float max = max(rgb.r, max(rgb.g, rgb.b));
	float min = min(rgb.r, min(rgb.g, rgb.b));
	float delta = max - min;
	
	result.z = max;
	result.y = max != 0.0 ? delta / max : 0.0;
	
	result.x =
		rgb.r == max ? (rgb.g - rgb.b) / delta :
		rgb.g == max ? 2.0 + (rgb.b - rgb.r) / delta :
		4.0 + (rgb.r - rgb.g) / delta;
	result.x /= 6.0;
	if (result.x < 0.0) result.x += 1.0;
	
	return result;
}
vec3 toRgb(vec3 hsv)
{
	if (hsv.y == 0.0) return vec3(hsv.z, hsv.z, hsv.z);
	
	if (1.0 <= hsv.x) hsv.x -= 1.0;
	hsv.x *= 6.0;
	float i = floor(hsv.x);
	float f = hsv.x - i;
	float aa = hsv.z * (1.0 - hsv.y);
	float bb = hsv.z * (1.0 - (hsv.y * f));
	float cc = hsv.z * (1.0 - (hsv.y * (1.0 - f)));
	
	if (i < 1.0) return vec3(hsv.z, cc, aa);
	else if (i < 2.0) return vec3(bb, hsv.z, aa);
	else if (i < 3.0) return vec3(aa, hsv.z, cc);
	else if (i < 4.0) return vec3(aa, bb, hsv.z);
	else if (i < 5.0) return vec3(cc, aa, hsv.z);
	else return vec3(hsv.z, aa, bb);
}
void main()
{
	vec4 base = gl_Color * texture2D(texture0, gl_TexCoord[0].xy);
	vec3 baseHsv = toHsv(base.rgb);
	float v = toHsv(texture2D(texture1, gl_TexCoord[1].xy).rgb).z;
	
	baseHsv.x -= (1.0 - v) * 0.1;
	if (baseHsv.x < 0.0) baseHsv.x += 1.0;
	gl_FragColor.rgb = toRgb(baseHsv);
	gl_FragColor.a = base.a;
}
";

///
const char[] SKINNED_MESH_VERTEX = "
attribute vec2 matrixIndex;
attribute float weight;
uniform mat4 matrixArray[34]; // とりあえず固定
uniform vec4 lightDirection;
uniform vec4 ambient;
vec4 blend(vec4 p)
{
	return matrixArray[int(matrixIndex.x)] * p * weight
		+ matrixArray[int(matrixIndex.y)] * p * (1.0 - weight);
}
vec3 nblend(vec3 n)
{
	vec4 n2 = vec4(n, 0);
	vec4 result = matrixArray[int(matrixIndex.x)] * n2 * weight
		+ matrixArray[int(matrixIndex.y)] * n2 * (1.0 - weight);
	return vec3(result);
}
float getTexCoord1X()
{
	return clamp(
		dot(nblend(gl_NormalMatrix * gl_Normal), lightDirection.xyz), 0.1, 0.9
	);
}
void main()
{
	gl_Position = gl_ModelViewProjectionMatrix * blend(gl_Vertex);
	gl_FrontColor = vec4(gl_Color.rgb * ambient.rgb, gl_Color.a);
	gl_TexCoord[0] = gl_MultiTexCoord0;
	gl_TexCoord[1].x = getTexCoord1X();
}
";
}

///
class ToonShader : ToonShaderBase
{
	///
	this()
	{
		uint[] pixels = new uint[8];
		pixels[0] = 0xffe0e0e0;
		pixels[1..$] = uint.max;
		super(TOON_VERTEX, TOON_FRAGMENT, new Texture(8, 1, pixels, 1));
	}
}

///
class HsvToonShader : ToonShaderBase
{
	///
	this()
	{
		uint[] pixels = new uint[8];
		pixels[0] = 0xff000000;
		pixels[1..$] = uint.max;
		super(TOON_VERTEX, HSV_TOON_FRAGMENT, new Texture(8, 1, pixels, 1));
	}
}

///
class SkinnedMeshToonShader : ToonShaderBase
{
	///
	this()
	{
		uint[] pixels = new uint[8];
		pixels[0] = 0xffe0e0e0;
		pixels[1..$] = uint.max;
		super(SKINNED_MESH_VERTEX, TOON_FRAGMENT, new Texture(8, 1, pixels, 1));
	}
}

///
class SkinnedMeshHsvToonShader : ToonShaderBase
{
	///
	this()
	{
		uint[] pixels = new uint[8];
		pixels[0] = 0xff000000;
		pixels[1..$] = uint.max;
		super(SKINNED_MESH_VERTEX, HSV_TOON_FRAGMENT, new Texture(8, 1, pixels, 1));
	}
}

///
class Subset : Unit
{
	private final PcntBuffer pcntOrPcntmw;
	private final Texture texture;
	private bool _visible = false;
	
	///
	this(PcntBuffer pcntOrPcntmw, Texture texture)
	{
		this.pcntOrPcntmw = pcntOrPcntmw;
		this.texture = texture;
	}
	
	void visible(bool a) ///
	{
		_visible = a;
	}
	
	bool visible() ///
	{
		return _visible;
	}
	
	void attach()
	{
		texture.attach();
		pcntOrPcntmw.attach();
	}
	
	void detach()
	{
		pcntOrPcntmw.detach();
		texture.detach();
	}
	
	void matrixIndexLocation(uint a) ///
	{
		PcntmwBuffer b = cast(PcntmwBuffer)pcntOrPcntmw;
		if (!b) exception(__FILE__, __LINE__, this);
		b.matrixIndexLocation = a;
	}
	
	void weightLocation(uint a) ///
	{
		PcntmwBuffer b = cast(PcntmwBuffer)pcntOrPcntmw;
		if (!b) exception(__FILE__, __LINE__, this);
		b.weightLocation = a;
	}
}

///
class NullSubset : Subset
{
	this()
	{
		super(null, null);
	}
	
	override void attach() {}
	override void detach() {}
	override void matrixIndexLocation(uint a) {}
	override void weightLocation(uint a) {}
}

///
class Nanami : Unit
{
	const int FPS = 60; ///
	final Subset[][] subset; ///
	final Matrix[][][] matrixArray; ///
	final char[] fileName; ///
	
	///
	this(char[] nanamiFileName)
	{
		this.fileName = nanamiFileName;
		NanamiFile file = new NanamiFile(nanamiFileName);
		for (int y = 0; y < 99; y++)
		{
			if (2 <= y && (!file.hasVertexBuffer(0, y) || !file.hasTexture(0, y))) break;
			
			Subset[] subsetList;
			for (int x = 0; x < 99; x++)
			{
				if (!file.hasVertexBuffer(x, y) || !file.hasTexture(x, y)) break;
				
				uint vertexSize, vertexLength;
				float[] vertexBuffer = file.expandVertexBuffer(x, y, vertexSize, vertexLength);
				if (vertexSize == 0)
				{
					subsetList ~= new NullSubset();
				}
				else
				{
					uint textureWidth, textureHeight;
					uint[] pixels = file.expandTexture(x, y, textureWidth, textureHeight);
					subsetList ~= new Subset(
						vertexSize == PcntBuffer.VERTEX_SIZE
						? new PcntBuffer(vertexLength, vertexBuffer)
						: new PcntmwBuffer(vertexLength, vertexBuffer),
						new Texture(textureWidth, textureHeight, pixels, 0)
					);
				}
				glErrorCheck(__FILE__, __LINE__);
			}
			subset ~= subsetList;
		}
		
		foreach (inout Subset a; subset[0]) a.visible = true;
		
		if (!file.hasSkeletonRelations) return;
		
		SkeletonRelations relations = new SkeletonRelations(file.expandSkeletonRelations());
		
		MotionMatrix[] mmList;
		for (int i = 0; i < 99; i++)
		{
			if (!file.hasMotionMatrix(i)) break;
			mmList ~= new MotionMatrix(file.expandMotionMatrix(i));
		}
		
		matrixArray.length = mmList.length;
		for (int i = 0; i < matrixArray.length; i++)
		{
			matrixArray[i].length = mmList[i].frameLength;
			for (int t = 0; t < matrixArray[i].length; t++)
			{
				matrixArray[i][t] = relations.getMatrixArray(mmList[i], t);
			}
		}
	}
	
	void attach()
	{
		foreach (Subset[] a; subset)
		{
			foreach (Subset b; a)
			{
				if (!b.visible) continue;
				b.attach();
				b.detach();
			}
		}
		glErrorCheck(__FILE__, __LINE__);
	}
	
	void detach() {}
	
	void matrixIndexLocation(uint l) ///
	{
		foreach (Subset[] a; subset)
		{
			foreach (Subset b; a) b.matrixIndexLocation = l;
		}
	}
	
	void weightLocation(uint l) ///
	{
		foreach (Subset[] a; subset)
		{
			foreach (Subset b; a) b.weightLocation = l;
		}
	}
	
	bool hasMotion() ///
	{
		return matrixArray ? true : false;
	}
}

/// ファイルから読み込んで適当な大きさに調整
class ImageTexture : Texture
{
	///
	char[] fileName;
	
	/// 本来のサイズ
	uint imageWidth, imageHeight;
	
	///
	this(char[] imageFileName, uint index)
	{
		fileName = imageFileName;
		Rgba rgba = new ImageLoader(imageFileName);
		imageWidth = rgba.width;
		imageHeight = rgba.height;
		if (!TextureSizeCorrector.isCorrectSize(rgba))
		{
			rgba = new TextureSizeScalingCorrector(rgba);
		}
		super(rgba, index);
	}
}

/// フォントから作成して適当な大きさに調整
class TextTexture : Texture
{
	///
	char[] text;
	
	/// 本来のサイズ
	uint imageWidth, imageHeight;
	
	///
	this(Font font, char[] text, Vector color)
	{
		this.text = text;
		Rgba rgba = new TextLoader(
			font, text,
			toUbyte(color.x), toUbyte(color.y), toUbyte(color.z), toUbyte(color.w)
		);
		imageWidth = rgba.width;
		imageHeight = rgba.height;
		if (!TextureSizeCorrector.isCorrectSize(rgba))
		{
			rgba = new TextureSizeScalingCorrector(rgba);
		}
		super(rgba, 0);
	}
	
	private static ubyte toUbyte(float a)
	{
		return cast(ubyte)(a * ubyte.max);
	}
}

///
class Text : UnitMerger, VirtualSizeUnit
{
	///
	char[] text;
	
	///
	int x, y, width, height;
	
	///
	this(
		int x, int y, char[] text,
		Vector color = vector(1.0, 1.0, 1.0, 1.0),
		Font font = new DefaultFont()
	)
	{
		this.text = text;
		TextTexture texture = new TextTexture(font, text, color);
		this.x = x;
		this.y = y;
		width = texture.imageWidth;
		height = texture.imageHeight;
		super(
			new AlphaBlend(),
			texture,
			new Quadrangle(x, y, width, height)
		);
	}
}

///
class Image : Quadrangle, VirtualSizeUnit
{
	final char[] fileName; ///
	private Unit abTexture;
	
	///
	this(char[] imageFileName, int x, int y)
	{
		ImageTexture it = new ImageTexture(imageFileName, 0);
		this(it, x, y, it.imageWidth, it.imageHeight);
	}
	
	///
	this(char[] imageFileName, int x, int y, int width, int height)
	{
		this(new ImageTexture(imageFileName, 0), x, y, width, height);
	}
	
	private this(ImageTexture it, int x, int y, int width, int height)
	{
		this.fileName = it.fileName;
		abTexture = new UnitMerger(new AlphaBlend(), it);
		super(x, y, width, height);
	}
	
	override void attach()
	{
		abTexture.attach();
		super.attach();
		abTexture.detach();
	}
}

///
interface ButtonEventFactory
{
	Event create(int x, int y, int width, int height, void delegate() onClick); ///
	VirtualMousePoint virtualMousePoint(); ///
}

///
class Button : VirtualSizeUnit
{
	private final Text text;
	private final Quadrangle quad;
	private final ButtonEventFactory factory;
	
	///
	this(Text text, ButtonEventFactory factory)
	{
		this(text, factory, text.width, text.height);
	}
	
	///
	this(Text text, ButtonEventFactory factory, int width, int height)
	{
		this.text = text;
		this.quad = new Quadrangle(text.x, text.y, width, height, backColor);
		this.factory = factory;
	}
	
	Event clickEvent() ///
	{
		return factory.create(quad.x, quad.y, quad.width, quad.height, &onClick);
	}
	
	void onClick() {} ///
	
	void attach()
	{
		VirtualMousePoint mp = factory.virtualMousePoint;
		quad.color = mp.isInner(quad.x, quad.y, quad.width, quad.height) ? floatColor : backColor;
		quad.attach();
		quad.detach();
		text.attach();
		text.detach();
	}
	
	void detach() {}
	
	protected Vector backColor() ///
	{
		return vector(0.0, 0.0, 0.0, 1.0);
	}
	
	protected Vector floatColor() ///
	{
		return vector(0.4, 0.4, 0.4, 1.0);
	}
	
	int x() ///
	{
		return quad.x;
	}
	
	int y() ///
	{
		return quad.y;
	}
	
	int width() ///
	{
		return quad.width;
	}
	
	int height() ///
	{
		return quad.height;
	}
}

///
class CheckButton : Button
{
	bool value = false; ///
	
	///
	this(Text text, ButtonEventFactory factory)
	{
		super(text, factory);
	}
	
	///
	this(Text text, ButtonEventFactory factory, int width, int height)
	{
		super(text, factory, width, height);
	}
	
	override void onClick()
	{
		value = !value;
	}
	
	override Vector backColor()
	{
		return value ? vector(0.6, 0.6, 0.6, 1.0) : super.backColor;
	}
	
	override Vector floatColor()
	{
		return value ? vector(0.8, 0.8, 0.8, 1.0) : super.floatColor;
	}
}

///
class ButtonFocus : LineQuadrangle
{
	private final Button[] buttons;
	private final VirtualMousePoint mousePoint;
	
	///
	this(Button[] buttons, VirtualMousePoint mousePoint, Vector color)
	{
		this.buttons = buttons;
		this.mousePoint = mousePoint;
		super(0, 0, 0, 0, color);
	}
	
	override void attach()
	{
		super.attach();
		foreach (Button a; buttons)
		{
			if (!mousePoint.isInner(a.x, a.y, a.width, a.height)) continue;
			x = (x + a.x) / 2;
			y = (y + a.y) / 2;
			width = (width + a.width) / 2;
			height = (height + a.height) / 2;
			return;
		}
		x = (x + mousePoint.x) / 2;
		y = (y + mousePoint.y) / 2;
		width = width / 2;
		height = height / 2;
	}
}

///
class MultilineText : UnitMerger, VirtualSizeUnit
{
	char[][] texts; ///
	
	///
	this(
		int x, int y, char[][] texts, uint heightMargin = 0,
		Vector color = vector(1.0, 1.0, 1.0, 1.0),
		Font font = new DefaultFont()
	)
	{
		this.texts = texts;
		int y2 = y;
		foreach (char[] a; texts)
		{
			Text text = new Text(x, y2, a, color, font);
			y2 += text.height + heightMargin;
			this ~= text;
		}
	}
}

///
class ImageEffect : Quadrangle
{
	final ImageTexture texture; ///
	uint currentIndex; ///
	bool visible = true; ///
	
	///
	this(ImageTexture texture, int x, int y)
	{
		this.texture = texture;
		super(x, y, texture.imageHeight, texture.imageHeight);
		tcWidth = 1.0 / length;
	}
	
	uint length() ///
	{
		return texture.width / texture.height;
	}
	
	override void attach()
	{
		if (!visible || length <= currentIndex) return;
		tcX = cast(float)currentIndex / length;
		texture.attach();
		super.attach();
		texture.detach();
	}
}
