﻿module coneneko.nanami;
private import
	coneneko.scenegraph,
	coneneko.math,
	std.zip,
	std.stream,
	std.file,
	std.string;

///
class NanamiFile : ZipArchive
{
	///
	this(char[] fileName)
	{
		super(read(fileName));
	}
	
	bool hasSkeletonRelations() ///
	{
		return "skeleton_relations" in directory ? true : false;
	}
	
	ubyte[] expandSkeletonRelations() ///
	{
		return expand(directory["skeleton_relations"]);
	}
	
	bool hasVertexBuffer(uint x, uint y) ///
	{
		return format("vertex_buffer[%d][%d]", y, x) in directory ? true : false;
	}
	
	bool hasTexture(uint x, uint y) ///
	{
		return format("texture[%d][%d]", y, x) in directory ? true : false;
	}
	
	float[] expandVertexBuffer(uint x, uint y, out uint vertexSize, out uint vertexLength) ///
	{
		MemoryStream ms = new MemoryStream(
			expand(directory[format("vertex_buffer[%d][%d]", y, x)])
		);
		ms.read(vertexSize);
		if (vertexSize == 0) return null;
		ms.read(vertexLength);
		float[] result = new float[vertexSize * vertexLength];
		ms.readExact(result.ptr, float.sizeof * result.length);
		return result;
	}
	
	uint[] expandTexture(uint indexX, uint indexY, out uint width, out uint height) ///
	{
		MemoryStream ms = new MemoryStream(
			expand(directory[format("texture[%d][%d]", indexY, indexX)])
		);
		ms.read(width);
		ms.read(height);
		uint[] result = new uint[width * height];
		ms.readExact(result.ptr, uint.sizeof * result.length);
		return result;
	}
	
	bool hasMotionMatrix(uint index) ///
	{
		return format("motion_matrix[%d]", index) in directory ? true : false;
	}
	
	void[] expandMotionMatrix(uint index) ///
	{
		return expand(directory[format("motion_matrix[%d]", index)]);
	}
	
	///
	this()
	{
		super();
	}
	
	///
	this(void[] data)
	{
		super(data);
	}
	
	void add(char[] name, void[] data) ///
	{
		ArchiveMember am = new ArchiveMember();
		am.name = name;
		am.expandedData = cast(ubyte[])data;
		am.compressionMethod = 8;
		addMember(am);
	}
	
	unittest
	{
		float[] testData = new float[2 * 3];
		for (int i = 0; i < testData.length; i++) testData[i] = cast(float)i;
		
		NanamiFile encoder = new NanamiFile();
		encoder.addVertexBuffer(0, 1, 2, 3, testData);
		NanamiFile decoder = new NanamiFile(encoder.build());
		assert(decoder.hasVertexBuffer(0, 1));
		uint w, h;
		assert(testData == decoder.expandVertexBuffer(0, 1, w, h));
		assert(2 == w);
		assert(3 == h);
	}
	
	void addVertexBuffer(uint x, uint y, uint vertexSize, uint vertexLength, float[] buffer) ///
	{
		if (vertexSize * vertexLength != buffer.length) exception(__FILE__, __LINE__, this);
		MemoryStream ms = new MemoryStream();
		ms.write(vertexSize);
		ms.write(vertexLength);
		ms.writeExact(buffer.ptr, float.sizeof * buffer.length);
		add(format("vertex_buffer[%d][%d]", y, x), ms.data);
	}
	
	unittest
	{
		uint[] testData = new uint[3 * 4];
		for (int i = 0; i < testData.length; i++) testData[i] = i;
		
		NanamiFile encoder = new NanamiFile();
		encoder.addTexture(1, 2, 3, 4, testData);
		NanamiFile decoder = new NanamiFile(encoder.build());
		assert(decoder.hasTexture(1, 2));
		uint w, h;
		assert(testData == decoder.expandTexture(1, 2, w, h));
		assert(3 == w);
		assert(4 == h);
	}
	
	void addTexture(uint indexX, uint indexY, uint width, uint height, uint[] pixels) ///
	{
		if (width * height != pixels.length) exception(__FILE__, __LINE__, this);
		MemoryStream ms = new MemoryStream();
		ms.write(width);
		ms.write(height);
		ms.writeExact(pixels.ptr, uint.sizeof * pixels.length);
		add(format("texture[%d][%d]", indexY, indexX), ms.data);
	}
}

///
class MotionMatrix
{
	private Matrix[][] list;
	
	///
	this(uint boneLength, uint frameLength)
	{
		list.length = frameLength;
		foreach (inout Matrix[] a; list) a.length = 3 * boneLength;
	}
	
	void setLrt(uint boneIndex, uint frameIndex,
		Matrix lookAt, Matrix rotation, Matrix translation) ///
	{
		list[frameIndex][3 * boneIndex + 0] = lookAt;
		list[frameIndex][3 * boneIndex + 1] = rotation;
		list[frameIndex][3 * boneIndex + 2] = translation;
	}
	
	///
	Matrix getLookAt(uint t, uint boneIndex) { return list[t][3 * boneIndex + 0]; }
	
	///
	Matrix getRotation(uint t, uint boneIndex) { return list[t][3 * boneIndex + 1]; }
	
	///
	Matrix getTranslation(uint t, uint boneIndex) { return list[t][3 * boneIndex + 2]; }
	
	uint boneLength() ///
	{
		return list[0].length / 3;
	}
	
	uint frameLength() ///
	{
		return list.length;
	}
	
	void[] toData() ///
	out (result)
	{
		assert(result.length == uint.sizeof + float.sizeof * 16 * 3 * boneLength * frameLength);
	}
	body
	{
		MemoryStream result = new MemoryStream();
		result.write(frameLength);
		for (int fi = 0; fi < frameLength; fi++)
		{
			for (int bi = 0; bi < boneLength; bi++)
			{
				result.writeExact(&list[fi][3 * bi + 0], Matrix.sizeof);
				result.writeExact(&list[fi][3 * bi + 1], Matrix.sizeof);
				result.writeExact(&list[fi][3 * bi + 2], Matrix.sizeof);
			}
		}
		return result.data();
	}
	
	///
	this(void[] data, bit logging = false)
	{
		if (logging)
		{
			log = new MemoryStream();
			log.writeLine("MotionMatrix.this");
		}
		
		MemoryStream ms = new MemoryStream(cast(ubyte[])data);
		uint frameLength;
		ms.read(frameLength);
		list.length = frameLength;
		
		uint boneLength = (data.length - uint.sizeof) / Matrix.sizeof / frameLength / 3;
		if (logging)
		{
			log.writefln("frameLength=%d boneLength=%d", frameLength, boneLength);
		}
		
		for (int f = 0; f < frameLength; f++)
		{
			list[f].length = 3 * boneLength;
			for (int b = 0; b < boneLength; b++)
			{
				Matrix l, r, t;
				ms.readExact(&l, Matrix.sizeof);
				ms.readExact(&r, Matrix.sizeof);
				ms.readExact(&t, Matrix.sizeof);
				setLrt(b, f, l, r, t);
			}
		}
	}
	
	MemoryStream log; ///
}

///
class SkeletonRelations
{
	private final ubyte[] pairs;
	
	/// (first, second) (first, second) ...
	this(ubyte[] pairs)
	{
		this.pairs = pairs;
	}
	
	Matrix[] getMatrixArray(MotionMatrix motionMatrix, uint t) ///
	{
		Matrix[] result = new Matrix[motionMatrix.boneLength];
		for (int i = 0; i < result.length; i++) result[i] = getBoneMatrix(motionMatrix, t, i);
		return result;
	}
	
	private Matrix getBoneMatrix(MotionMatrix motionMatrix, uint t, uint boneIndex)
	{
		Matrix result = motionMatrix.getLookAt(t, boneIndex);
		uint currentIndex = boneIndex;
		while (true)
		{
			result *= motionMatrix.getRotation(t, currentIndex);
			result *= motionMatrix.getTranslation(t, currentIndex);
			if (isRoot(currentIndex)) break;
			currentIndex = getParent(currentIndex);
		}
		return result;
	}
	
	private bool isRoot(ubyte a)
	{
		return a == 0;
	}
	
	private ubyte getParent(ubyte a)
	{
		for (int i = 0; i < pairs.length; i += 2)
		{
			ubyte first = pairs[i];
			ubyte second = pairs[i + 1];
			if (second == a) return first;
		}
		exception(__FILE__, __LINE__, this);
	}
}
