﻿module coneneko.scenegraph;
private import
	std.stream,
	std.string,
	std.c.string,
	std.math,
	std.date;

///
interface RenderTarget
{
	void draw(Node a); ///
	void flip(); ///
}

///
interface Unit
{
	void attach(); ///
	void detach(); ///
}

///
class NullUnit : Unit
{
	void attach() {}
	void detach() {}
}

///
class UnitMerger : Unit
{
	unittest
	{
		MemoryStream trace = new MemoryStream();
		Unit unit = new UnitMerger(
			new TestUnit("1", trace),
			new TestUnit("2", trace)
		);
		unit.attach();
		unit.detach();
		assert("attach(1) attach(2) detach(2) detach(1) " == trace.toString());
	}
	
	unittest
	{
		MemoryStream trace = new MemoryStream();
		Unit[] testUnits; // TestUnit[]だとエラー
		testUnits ~= new TestUnit("1", trace);
		testUnits ~= new TestUnit("2", trace);
		Unit unit = new UnitMerger(testUnits);
		unit.attach();
		unit.detach();
		assert("attach(1) attach(2) detach(2) detach(1) " == trace.toString());
	}
	
	unittest
	{
		MemoryStream trace = new MemoryStream();
		UnitMerger unit = new UnitMerger();
		unit ~= new TestUnit("1", trace);
		unit ~= new TestUnit("2", trace);
		unit.attach();
		unit.detach();
		assert("attach(1) attach(2) detach(2) detach(1) " == trace.toString());
	}
	
	private Unit[] units;
	
	/// Unit派生配列は無理
	this(Unit[] a ...)
	{
		units = a.dup;
		foreach (inout Unit u; units)
		{
			if (!u) u = new NullUnit();
		}
	}
	
	void attach()
	{
		for (int i = 0; i < units.length; i++) units[i].attach();
	}
	
	void detach()
	{
		for (int i = units.length - 1; i >= 0; i--) units[i].detach();
	}
	
	void opCatAssign(Unit a) /// ~=
	{
		units ~= a;
	}
}

///
class Node
{
	unittest
	{
		MemoryStream trace = new MemoryStream();
		Node root = new Node();
		Unit testUnit = new TestUnit("r", trace);
		root.unit = testUnit;
		assert(testUnit == root.unit);
		
		Node node1 = new Node(new TestUnit("1", trace));
		Node node2 = new Node(new TestUnit("2", trace));
		Node node3 = new Node(new TestUnit("3", trace));
		
		root ~= node1;
		node1 ~= node2;
		node1 ~= node3;
		
		root();
		
		assert(
			"attach(r) attach(1) attach(2) detach(2) attach(3) detach(3) detach(1) detach(r) "
			== trace.toString()
		);
	}
	
	invariant
	{
		assert(_unit);
	}
	
	private Unit _unit;
	protected Node[] children;
	bool visible = true; ///
	
	///
	this(Unit a = null)
	{
		_unit = a ? a : new NullUnit();
	}
	
	void opCall() /// ()
	{
		if (!visible) return;
		_unit.attach();
		foreach (Node a; children) a();
		_unit.detach();
	}
	
	Unit unit() ///
	{
		return _unit;
	}
	
	void unit(Unit a) ///
	{
		_unit = a ? a : new NullUnit();
	}
	
	void opCatAssign(Node a) /// ~=
	{
		children ~= a;
	}
}

///
class ConenekoException : Exception
{
	///
	this(char[] file, long line, Object obj, char[] msg)
	{
		super(
			format("%s(%d): %s %s", file, line, obj ? obj.classinfo.name : "", msg)
		);
	}
}

/// throw new ConenekoException(__FILE__, __LINE__, this, "message");
void exception(char[] file, long line, Object obj = null, char[] msg = null)
{
	throw new ConenekoException(file, line, obj, msg);
}

private class TestUnit : Unit
{
	unittest
	{
		MemoryStream result = new MemoryStream();
		TestUnit tu = new TestUnit("1", result);
		tu.attach();
		tu.detach();
		assert("attach(1) detach(1) " == result.toString());
	}
	
	private final char[] name;
	private final MemoryStream trace;
	
	this(char[] name, MemoryStream trace)
	{
		this.name = name;
		this.trace = trace;
	}
	
	void attach()
	{
		trace.writef("attach(%s) ", name);
	}
	
	void detach()
	{
		trace.writef("detach(%s) ", name);
	}
}

/// graphvizのdotを作れる
class DotNode : Node // TODO NodeのtoStringに実装して削除
{
	unittest
	{
		Node root = new DotNode("0");
		Node node1 = new DotNode("1");
		Node node2 = new DotNode("2");
		Node node3 = new DotNode("3");
		
		root ~= node1;
		node1 ~= node2;
		node1 ~= node3;
		
		MemoryStream dot = new MemoryStream();
		dot.writeString("digraph DotNode\n");
		dot.writeString("{\n");
		dot.writeString("\t\"0\" -> \"1\";\n");
		dot.writeString("\t\"1\" -> \"2\";\n");
		dot.writeString("\t\"1\" -> \"3\";\n");
		dot.writeString("}\n");
		
		assert(dot.toString() == root.toString());
	}
	
	private final char[] name;
	
	///
	this(char[] name, Unit a = null)
	{
		super(a);
		this.name = name;
	}
	
	override char[] toString() /// dot
	{
		MemoryStream result = new MemoryStream();
		result.writeString("digraph DotNode\n");
		result.writeString("{\n");
		result.writeString(toString2());
		result.writeString("}\n");
		return result.toString();
	}
	
	private char[] toString2()
	{
		MemoryStream result = new MemoryStream();
		foreach (Node a; children)
		{
			DotNode dn = cast(DotNode)a;
			if (!dn) exception(__FILE__, __LINE__, this);
			result.writef("\t\"%s\" -> \"%s\";\n", name, dn.name);
		}
		foreach (Node a; children)
		{
			DotNode dn = cast(DotNode)a;
			if (!dn) exception(__FILE__, __LINE__, this);
			result.writeString(dn.toString2());
		}
		return result.toString();
	}
}

///
interface Window : RenderTarget
{
	bool alive(); ///
	uint width(); ///
	uint height(); ///
	
	void opCatAssign(Event e); /// ~=
	void clearEvents(); ///
	Event popReachedEvent(); ///
	
	MousePoint mousePoint(); ///
}

///
interface MousePoint
{
	int x(); ///
	int y(); ///
}

///
class Event {}

/**
基本的に、Event登録、killTime記述、switch(waiter.wait())の順に使います。
killTimeで画面を更新する場合は、引数RenderTargetのdraw, flipを使ってください。
waitが返す値は、killTimeのreturn値か、flip内で発生したイベントのidです。
*/
class Waiter : RenderTarget
{
	invariant
	{
		assert(wnd);
	}
	
	private final Window wnd;
	private uint[Event] map;
	
	/// event待ちの間、実行される。
	uint delegate(RenderTarget a) killTime;
	
	///
	this(Window wnd)
	{
		this.wnd = wnd;
	}
	
	void addEvent(uint result, Event e) ///
	{
		if (!e) exception(__FILE__, __LINE__, this, "event is null");
		map[e] = result;
	}
	
	void clearEvents() ///
	{
		map = null;
	}
	
	uint wait() /// Eventが発生するまで待つ。addEventで登録した値を返す。
	{
		if (!killTime) return 0;
		try
		{
			foreach (Event a; map.keys) wnd ~= a;
			return killTime(this);
		}
		catch (Event e)
		{
			return map[e];
		}
		finally
		{
			wnd.clearEvents();
		}
	}
	
	void draw(Node a)
	{
		wnd.draw(a);
	}
	
	void flip()
	{
		wnd.flip();
		if (!wnd.alive) throw new WindowDeadException();
		
		while (true)
		{
			Event e = wnd.popReachedEvent();
			if (!e) return;
			if (e in map) throw e;
		}
	}
}

/// Windowが閉じていたら投げる
class WindowDeadException : Exception
{
	this()
	{
		super("closed");
	}
}

/// 32bit
interface Rgba
{
	uint width(); ///
	uint height(); ///
	uint[] pixels(); ///
}

/// 2のn乗に修正
class TextureSizeCorrector : Rgba
{
	private final Rgba base;
	private uint _width, _height;
	
	///
	this(Rgba base)
	{
		this.base = base;
		_width = correctLength(base.width);
		_height = correctLength(base.height);
	}
	
	uint width()
	{
		return _width;
	}
	
	uint height()
	{
		return _height;
	}
	
	uint[] pixels()
	{
		uint[] result = new uint[_width * _height];
		for (uint y = 0; y < base.height; y++)
		{
			memcpy(&result[_width * y], &base.pixels[base.width * y], base.width * uint.sizeof);
		}
		return result;
	}
	
	unittest
	{
		assert(1 == correctLength(1));
		assert(2 == correctLength(2));
		assert(4 == correctLength(3));
		assert(4 == correctLength(4));
		assert(8 == correctLength(5));
		assert(8 == correctLength(6));
	}
	
	static uint correctLength(uint a) ///
	{
		if (a == 0) exception(__FILE__, __LINE__, null, "TextureSizeCorrector.correctLength size is 0");
		return cast(uint)pow(2.0, ceil(log2(cast(uint)a)));
	}
	
	static bool isCorrectSize(Rgba rgba) ///
	{
		return rgba.width == correctLength(rgba.width)
			&& rgba.height == correctLength(rgba.height);
	}
}

/// 拡大して修正
class TextureSizeScalingCorrector : TextureSizeCorrector
{
	///
	this(Rgba base)
	{
		super(base);
	}
	
	override uint[] pixels()
	{
		uint[] result = new uint[width * height];
		float bwPerW = cast(float)base.width / width;
		float bhPerH = cast(float)base.height / height;
		for (int y = 0; y < height; y++)
		{
			int srcY = cast(int)(cast(float)y * bhPerH);
			int width_y = width * y;
			int baseWidth_srcY = base.width * srcY;
			for (int x = 0; x < width; x++)
			{
				int srcX = cast(int)(cast(float)x * bwPerW);
				result[width_y + x] = base.pixels[baseWidth_srcY + srcX];
			}
		}
		return result;
	}
}

///
class WindowDecorator : Window
{
	private final Window base;
	
	///
	this(Window base)
	{
		this.base = base;
	}
	
	void draw(Node a)
	{
		base.draw(a);
	}
	
	void flip()
	{
		base.flip();
	}
	
	bool alive()
	{
		return base.alive;
	}
	
	uint width()
	{
		return base.width;
	}
	
	uint height()
	{
		return base.height;
	}
	
	void opCatAssign(Event e)
	{
		base ~= e;
	}
	
	void clearEvents()
	{	
		base.clearEvents();
	}
	
	Event popReachedEvent()
	{
		return base.popReachedEvent();
	}
	
	MousePoint mousePoint()
	{
		return base.mousePoint;
	}
}

/// srcを任意の大きさに TODO TextureSizeScalingCorrectorと重複しているので修正しておく
class RgbaScaler : Rgba
{
	private uint _width, _height;
	private uint[] _pixels;
	
	///
	this(Rgba src, uint width, uint height)
	{
		_width = width;
		_height = height;
		
		_pixels = new uint[width * height];
		float swPerW = cast(float)src.width / width;
		float shPerH = cast(float)src.height / height;
		for (int y = 0; y < height; y++)
		{
			int srcY = cast(int)(cast(float)y * shPerH);
			int width_y = _width * y;
			int baseWidth_srcY = src.width * srcY;
			for (int x = 0; x < width; x++)
			{
				int srcX = cast(int)(cast(float)x * swPerW);
				_pixels[width_y + x] = src.pixels[baseWidth_srcY + srcX];
			}
		}
	}
	
	uint width()
	{
		return _width;
	}
	
	uint height()
	{
		return _height;
	}
	
	uint[] pixels()
	{
		return _pixels;
	}
}

///
class UnitDecorator : Unit
{
	protected Unit base; ///
	
	///
	this(Unit base = new NullUnit())
	{
		this.base = base;
	}
	
	void attach()
	{
		base.attach();
	}
	
	void detach()
	{
		base.detach();
	}
}

/// 一定時間経過で無効になる
class TermUnit : UnitDecorator
{
	private long millisecond, start;
	private bool disable = false;
	
	///
	this(Unit base, long millisecond)
	{
		super(base);
		this.millisecond = millisecond;
		start = getUTCtime();
	}
	
	override void attach()
	{
		if (start + millisecond < getUTCtime()) disable = true;
		if (!disable) super.attach();
	}
	
	override void detach()
	{
		if (!disable) super.detach();
	}
}
