﻿module coneneko.sceneview;
public import coneneko.sdlwindow;
import std.string, std.date, std.math, std.utf, std.random, std.file, std.conv, std.gc;
import coneneko.pngfile, coneneko.math, coneneko.scenegraph, coneneko.units,
	coneneko.exunits, coneneko.texture;
import opengl;

///
struct ModelInfo
{
	bool enable; ///
	char[] file; /// ファイル名
	Vector point; ///
	float angle = 0; /// 正面から反時計回り[0, 360]
	Vector ambient = { 1.0, 1.0, 1.0, 0.0 }; /// 環境光、それぞれ[0, 1]
	byte fmi; /// first motion index -1で無効
	byte smi; /// second motion index -1で無効
	bool[] vis; /// visibleItems
	MorphInfo[] mps; /// morphs
}

///
struct MorphInfo
{
	bool enable; ///
	ubyte frameRate = 10; /// [f/s] 一秒間辺りの描画回数、1以上
	ubyte frequency; /// [0.1*s/morph発動] morph頻度、10で一秒間に一度くらい
	bool syncTe; ///
}

///
struct EffectInfo
{
	bool enable; ///
	char[] file; /// ファイル名
	short x; ///
	short y; ///
	short width; /// 0ならscalingなし
	short height; /// 0ならscalingなし
	ubyte t; /// [s/10] t * 0.1秒
	ubyte frameRate = 10; /// [f/s] 一秒間辺りの描画回数、1以上
	bool loop; ///
}

///
struct SoundEffectInfo
{
	bool enable; ///
	char[] file; /// ファイル名
	ubyte t; /// [s/10] t * 0.1秒
	bool loop; ///
}

private long max(long a, long b) { return a > b ? a : b; }
private uint min(uint a, uint b) { return a < b ? a : b; }
static assert(1000 == TicksPerSecond);

abstract class Texts : UnitDecorator
{
	protected SceneView view;
	this(SceneView view) { this.view = view; }
	
	override void attach()
	{
		MultilineText mt = cast(MultilineText)base;
		char[][] baseTexts = mt ? mt.texts : null;
		char[][] anyTexts = getViewAnyText().splitlines();
		if (baseTexts != anyTexts) base = createUnit(anyTexts);
		super.attach();
	}
	
	char[] getViewAnyText();
	Unit createUnit(char[][] texts);
}

class Te : Texts
{
	const int X = 76 + 48, Y = 362 + 48;
	this(SceneView view) { super(view); }
	char[] getViewAnyText() { return view.te; }
	Unit createUnit(char[][] texts) { return new MultilineText(X, Y, texts, 12); }
}

class Trace : Texts
{
	this(SceneView view) { super(view); }
	char[] getViewAnyText() { return view.trace; }
	Unit createUnit(char[][] texts) { return new MultilineText(0, 0, texts, 0, vector(1, 0, 0, 1)); }
}

class Bg : UnitDecorator
{
	private SceneView view;
	this(SceneView view) { this.view = view; }
	
	override void attach()
	{
		if (view.bg == "")
		{
			base = new NullUnit();
		}
		else
		{
			char[] fullPath = std.path.join(view.currentDir, view.bg);
			Image image = cast(Image)base;
			char[] imageFileName = image ? image.fileName : null;
			if (fullPath != imageFileName)
			{
				try
				{
					base = new Image(fullPath, 0, 0, 800, 600);
				}
				catch (Exception e)
				{
					view.bg = "";
					throw e;
				}
			}
		}
		super.attach();
	}
}

class Bgm : Unit
{
	private SceneView view;
	this(SceneView view) { this.view = view; }
	
	void attach()
	{
		SdlSoundMixer sm = SdlSoundMixer.getInstance();
		if (view.bgm == "")
		{
			sm.stopMusic();
			return;
		}
		char[] fullPath = std.path.join(view.currentDir, view.bgm);
		if (fullPath != sm.musicFileName)
		{
			try
			{
				sm.playMusic(fullPath);
			}
			catch (Exception e)
			{
				view.bgm = "";
				throw e;
			}
		}
	}
	
	void detach() {}
}

abstract class ArrayNode : Node
{
	protected SceneView view;
	this(SceneView view) { this.view = view; }
	private Unit[] units;
	
	override void opCall()
	{
		for (int i = viewAnyArrayLength; i < units.length; i++) delete units[i];
		units.length = viewAnyArrayLength;
		for (int i = 0; i < units.length; i++) if (!units[i]) units[i] = createUnit(i);
		foreach (Unit a; units) { a.attach(); a.detach(); }
		super.opCall();
	}
	
	uint viewAnyArrayLength();
	Unit createUnit(uint index);
}

class EsNode : ArrayNode
{
	this(SceneView view) { super(view); }
	uint viewAnyArrayLength() { return view.es.length; }
	Unit createUnit(uint index) { return new Effect(view, index); }
}

class Effect : UnitDecorator
{
	private SceneView view;
	private uint index;
	
	this(SceneView view, uint index)
	{
		this.view = view;
		this.index = index;
	}
	
	override void attach()
	{
		EffectInfo info = view.es[index];
		if (!info.enable || info.file == "") return;
		char[] fullPath = std.path.join(view.currentDir, info.file);
		char[] textureFileName = imageEffect ? imageEffect.texture.fileName : "";
		if (textureFileName != fullPath)
		{
			try
			{
				base = new ImageEffect(new ImageTexture(fullPath, 0), info.x, info.y);
			}
			catch (Exception e)
			{
				view.es[index].file = "";
				throw e;
			}
		}
		
		long begin = view.timeStamp + info.t * 100;
		long current = getUTCtime();
		int imageIndex = (current - begin) / (1000 / info.frameRate);
		if (imageIndex < 0) return;
		if (!info.loop && imageEffect.length <= imageIndex) return;
		while (imageEffect.length <= imageIndex) imageIndex -= imageEffect.length;
		imageEffect.currentIndex = imageIndex;
		imageEffect.x = info.x;
		imageEffect.y = info.y;
		imageEffect.width = info.width != 0 ? info.width : imageEffect.texture.imageHeight;
		imageEffect.height = info.height != 0 ? info.height : imageEffect.texture.imageHeight;
		
		super.attach();
	}
	
	private ImageEffect imageEffect()
	{
		return cast(ImageEffect)base;
	}
}

class SesNode : ArrayNode
{
	this(SceneView view) { super(view); }
	uint viewAnyArrayLength() { return view.ses.length; }
	Unit createUnit(uint index) { return new SoundEffect(view, index); }
}

class SoundEffect : Unit
{
	private SceneView view;
	private uint index;
	
	// timeStampが修正された瞬間に全ての状態はAに遷移する、ただしC->Cの場合は条件によっては遷移しない
	// timeStamp修正後、t * 100秒経過した瞬間にA->*の遷移が発生する
	enum State { STOPPING, PLAYING, LOOPING }; // P->L L->P P->Pの遷移は発生しない
	private State state = State.STOPPING;
	private long stockedTimeStamp = 0;
	private bool overT = false; // timeStamp修正後false, +tを超えたらtrue
	private SdlSoundEffect se;
	
	this(SceneView view, uint index)
	{
		this.view = view;
		this.index = index;
	}
	
	~this()
	{
		delete se;
	}
	
	void attach()
	{
		if (stockedTimeStamp != view.timeStamp)
		{
			stockedTimeStamp = view.timeStamp;
			overT = false;
			
			// ループ継続
			if (state == State.LOOPING && info.enable && info.t == 0 && fullPath == se.fileName)
			{
				overT = true;
				return;
			}
			
			stop();
		}
		
		if (overT) return;
		if (view.timeStamp + info.t * 100 <= getUTCtime())
		{
			overT = true;
			
			if (!info.enable || info.file == "") return;
			info.loop ? playLoop() : play();
		}
	}
	
	private SoundEffectInfo info() { return view.ses[index]; }
	private char[] fullPath() { return std.path.join(view.currentDir, info.file); }
	
	private void play()
	in
	{
		assert(!se);
		assert(State.STOPPING == state);
	}
	body
	{
		se = new SdlSoundEffect(fullPath, false);
		state = State.PLAYING;
	}
	
	private void playLoop()
	in
	{
		assert(State.STOPPING == state || State.LOOPING == state);
	}
	body
	{
		if (state == State.STOPPING) se = new SdlSoundEffect(fullPath, true);
		state = State.LOOPING;
	}
	
	private void stop()
	{
		if (se && se.playing) se.stop();
		se = null;
		state = State.STOPPING;
	}
	
	void detach() {}
}

class MsNode : Node
{
	private SceneView view;
	private Unit stateUnit;
	private LoadMatrix lookAt;
	private ToonShaderBase toonShader, smShader;
	private Model[] models;
	
	this(SceneView view)
	{
		this.view = view;
		lookAt = new LoadMatrix(Matrix.identity);
		//toonShader = new ToonShader();
		//smShader = new SkinnedMeshToonShader();
		toonShader = new HsvToonShader();
		smShader = new SkinnedMeshHsvToonShader();
		stateUnit = new UnitMerger(
			new Enable(GL_CULL_FACE),
			new CullFace(GL_FRONT),
			new Enable(GL_DEPTH_TEST),
			new MatrixMode(GL_PROJECTION),
			new LoadMatrix(Matrix.perspectiveFov),
			new MatrixMode(GL_MODELVIEW),
			lookAt
		);
	}
	
	override void opCall()
	{
		updateLookAt();
		updateLightDirection();
		
		for (int i = view.ms.length; i < models.length; i++) delete models[i];
		models.length = view.ms.length;
		foreach (int i, inout Model a; models)
		{
			if (!a) a = new Model(view, toonShader, smShader, i);
		}
		
		stateUnit.attach();
		try
		{
			foreach (Model a; models) { a.attach(); a.detach(); }
			super.opCall();
		}
		catch (Exception e)
		{
			throw e;
		}
		finally
		{
			stateUnit.detach();
		}
	}
	
	private void updateLookAt()
	{
		Vector cp = view.cameraPoint;
		Vector ca = view.cameraAt;
		if (cp == ca) ca.z -= 1.0;
		lookAt.matrix = Matrix.lookAt(cp.x, cp.y, cp.z,  ca.x, ca.y, ca.z);
	}
	
	private void updateLightDirection()
	{
		toonShader.lightDirection = view.lightDirection;
		smShader.lightDirection = view.lightDirection;
	}
}

class Model : UnitDecorator
{
	private SceneView view;
	private ToonShaderBase toonShader, smShader;
	private uint index;
	private long[] morphBeginTimes; // 待機状態=-1, 開始時間=timeStamp
	private RandomFrequency randomFrequency;
	
	this(SceneView view, ToonShaderBase toonShader, ToonShaderBase smShader, uint index)
	{
		this.view = view;
		this.toonShader = toonShader;
		this.smShader = smShader;
		this.index = index;
		randomFrequency = new RandomFrequency();
	}
	
	private ModelInfo info() { return view.ms[index]; }
	private Nanami nanami() { return cast(Nanami)base; }
	
	private ToonShaderBase currentShader()
	{
		if (nanami.hasMotion) return smShader;
		else return toonShader;
	}
	
	override void attach()
	{
		if (!info.enable || info.file == "") return;
		
		char[] fullPath = std.path.join(view.currentDir, info.file);
		char[] nanamiFileName = nanami ? nanami.fileName : "";
		if (fullPath != nanamiFileName)
		{
			try
			{
				base = new Nanami(fullPath);
			}
			catch (Exception e)
			{
				view.ms[index].file = "";
				throw e;
			}
			initMorph();
		}
		
		currentShader.ambient = info.ambient;
		
		setVisibleItems();
		morphing();
		if (nanami.hasMotion) setMotion();
		
		MultMatrix rtMatrix = createRotationAndTranslation();
		rtMatrix.attach();
		currentShader.attach();
		super.attach();
		currentShader.detach();
		rtMatrix.detach();
	}
	
	private MultMatrix createRotationAndTranslation()
	{
		return new MultMatrix(
			Matrix.rotationY(toRadian(info.angle))
			* Matrix.translation(info.point.x, info.point.y, info.point.z)
		);
	}
	
	private void setMotion()
	{
		nanami.matrixIndexLocation = smShader.matrixIndexLocation;
		nanami.weightLocation = smShader.weightLocation;
		
		Matrix[] identityArray = new Matrix[nanami.matrixArray[0][0].length];
		identityArray[] = Matrix.identity;
		smShader.matrixArray = identityArray;
		
		if (nanami.matrixArray.length <= info.fmi) info.fmi = -1;
		if (nanami.matrixArray.length <= info.smi) info.smi = -1;
		int firstLength = 0, secondLength = 0;
		if (info.fmi != -1) firstLength = nanami.matrixArray[info.fmi].length;
		if (info.smi != -1) secondLength = nanami.matrixArray[info.smi].length;
		
		const int RANGE = 1000 / Nanami.FPS;
		long passage = getUTCtime() - view.timeStamp;
		bool motionIsFirstRange = passage < RANGE * firstLength;
		bool motionIsSecondRange = !motionIsFirstRange && 1 <= secondLength;
		if (motionIsFirstRange)
		{
			smShader.matrixArray = nanami.matrixArray[info.fmi][passage / RANGE];
		}
		else if (motionIsSecondRange)
		{
			long p2 = passage - RANGE * firstLength;
			p2 /= RANGE;
			p2 %= secondLength;
			smShader.matrixArray = nanami.matrixArray[info.smi][p2];
		}
	}
	
	private void setVisibleItems()
	{
		if (nanami.subset.length >= 1)
		{
			foreach (Subset a; nanami.subset[1]) a.visible = false;
		}
		for (int i = 0; i < info.vis.length; i++)
		{
			nanami.subset[1][i].visible = info.vis[i];
		}
	}
	
	private void initMorph()
	{
		morphBeginTimes.length = nanami.subset.length >= 2 ? nanami.subset.length - 2 : 0;
		morphBeginTimes[] = -1;
	}
	
	private void morphing()
	{
		if (nanami.subset.length >= 2)
		{
			for (int i = 2; i < nanami.subset.length; i++)
			{
				foreach (Subset a; nanami.subset[i]) a.visible = false;
			}
		}
		for (int i = 0; i < info.mps.length; i++)
		{
			MorphInfo mpInfo = info.mps[i];
			Subset[] visibleList = nanami.subset[2 + i];
			
			if (!mpInfo.enable)
			{
				morphBeginTimes[i] = -1;
				continue;
			}
			
			bool waiting() { return morphBeginTimes[i] == -1; }
			long visibleTime = 1000 / mpInfo.frameRate;
			long allVisibleTime = visibleTime * visibleList.length;
			long currentTime = getUTCtime();
			
			if (waiting && mpInfo.syncTe && view.te != "")
			{
				int charLength = toUTF32(view.te).length
					- count(view.te, "\r")
					- count(view.te, "\n");
				long serifEndTime = view.timeStamp + allVisibleTime * charLength;
				if (currentTime < serifEndTime) morphBeginTimes[i] = currentTime;
			}
			else if (waiting && !mpInfo.syncTe) // frequency
			{
				if (randomFrequency.random(mpInfo.frequency * 0.1))
				{
					morphBeginTimes[i] = currentTime;
				}
			}
			
			if (!waiting && morphBeginTimes[i] + allVisibleTime <= currentTime)
			{
				morphBeginTimes[i] = -1;
			}
			
			long si = waiting ? 0 : (currentTime - morphBeginTimes[i]) / visibleTime;
			visibleList[si].visible = true;
		}
	}
}

///
class RandomFrequency
{
	static assert(1000 == TicksPerSecond);
	private long preTime;
	
	///
	this(long time = getUTCtime())
	{
		preTime = time;
	}
	
	bool random(float a, long time = getUTCtime()) ///
	{
		float a2 = a * (cast(float)(time - preTime) / 1000);
		bool result = a2 >= 1.0 ? true : rand() < (uint.max * a2);
		preTime = time;
		return result;
	}
}

/**
単位は[cm]ぐらい
配列は.lengthの後代入、ただしvis, mpsの.lengthはModelInfo.fileに依存するので変更不要
*/
class SceneView : WindowDecorator
{
	char[] te; /// text テキストエリアの、'\n'で改行、3行まで
	char[] trace; /// エラー表示などに使う
	char[] bg; /// background ファイル名
	char[] bgm; /// backgroundMusic ファイル名
	EffectInfo[] es; /// effects
	SoundEffectInfo[] ses; /// soundEffects
	Vector cameraPoint; /// [cm]
	Vector cameraAt; ///
	Vector lightDirection; ///
	ModelInfo[] ms; /// models
	
	Node rootNode; ///
	long timeStamp; /// getUTCtime();
	Node textNode; ///
	
	char[] baseDir; /// args[0]のdir
	char[] resourceDir; /// baseDirからの相対dir
	
	char[] currentDir() /// join(baseDir, resourceDir);
	{
		return std.path.join(baseDir, resourceDir);
	}
	
	///
	this(Window base)
	{
		super(base);
		
		rootNode = new Node(
			new UnitMerger(
				new ClearColor(1, 1, 1, 1),
				new Clear()
			)
		);
		Unit trace = new Trace(this);
		rootNode ~= new Node(trace);
		rootNode ~= new Node(new Bg(this));
		rootNode ~= new Node(trace);
		rootNode ~= new MsNode(this);
		
		textNode = new Node(new BlendQuadrangle(76, 362, 647, 200, vector(0, 0, 0, 0.5)));
		textNode ~= new Node(new Te(this));
		
		rootNode ~= textNode;
		rootNode ~= new EsNode(this);
		rootNode ~= new SesNode(this);
		rootNode ~= new Node(new Bgm(this));
		
		initialize();
	}
	
	void initialize() ///
	{
		te = te.init;
		trace = trace.init;
		bg = bg.init;
		bgm = bgm.init;
		es.length = 0;
		ses.length = 0;
		cameraPoint = vector(0, 80, 300);
		cameraAt = vector(0, 40, 0);
		lightDirection = vector(-1, -1, 0);
		ms.length = 0;
	}
}

///
class LoadEvent : Event {}

///
class AdvView : SceneView
{
	private bool skipping = false;
	private Node choicesNode;
	// TODO ボタン replay, load, 文字消し, skip
	private Button saveButton, loadButton; // TODO Buttonは汎用性がないので独自のものに置き換え
	private ButtonEventFactory factory; //
	private Node messageNode;
	private Location location;
	Location loadingTarget; /// 通常はnull load中は設定されて、そのままメインwhile終わりに達する場合はエラー
	
	///
	this(Window base)
	{
		choicesNode = new Node();
		super(base);
		textNode ~= choicesNode;
		version (DWT_WINDOW) factory = new DwtButtonEventFactory(cast(DwtWindow)base);
		else factory = new SdlButtonEventFactory(cast(SdlWindow)base);
		saveButton = new Button(new Text(0, 500, "save"), factory);
		loadButton = new Button(new Text(0, 550, "load"), factory);
		textNode ~= new Node(
			new UnitMerger(
				saveButton,
				loadButton
			)
		);
		messageNode = new Node();
		textNode ~= messageNode;
	}
	
	override void initialize() ///
	{
		super.initialize();
		
		location = new Location();
		choicesNode.unit = null;
	}
	
	void wc() /// waitClick
	{
		if (loadingTarget)
		{
			if (loadingTarget == location)
			{
				loadingTarget = null;
				message("loadしました", vector(0, 0, 1, 0.5));
			}
			else
			{
				location++;
				return;
			}
		}
		
		const int LEFT_CTRL = 306, RIGHT_CTRL = 305;
		
		timeStamp = getUTCtime();
		
		Waiter waiter = new Waiter(this);
		waiter.addEvent(1, new SdlClickEvent(100, 0, 700, 600)); // TODO ボタンと重複しないように
		waiter.addEvent(2, new SdlKeyDownEvent(' '));
		waiter.addEvent(2, new SdlRightClickEvent(0, 0, 800, 600));
		waiter.addEvent(3, new SdlKeyDownEvent(LEFT_CTRL));
		waiter.addEvent(3, new SdlKeyDownEvent(RIGHT_CTRL));
		waiter.addEvent(4, new SdlKeyUpEvent(LEFT_CTRL));
		waiter.addEvent(4, new SdlKeyUpEvent(RIGHT_CTRL));
		waiter.addEvent(5, saveButton.clickEvent);
		waiter.addEvent(6, loadButton.clickEvent);
		waiter.killTime = delegate uint(RenderTarget rt)
		{
			while (true)
			{
				rt.draw(rootNode);
				rt.flip();
				if (skipping) return 0;
			}
			return 0;
		};
		while (true)
		{
			uint id = waiter.wait();
			if (id == 1)
			{
				if (!textNode.visible) textNode.visible = true;
				else break;
			}
			else if (id == 2) textNode.visible = !textNode.visible;
			else if (id == 3) skipping = true;
			else if (id == 4) skipping = false;
			else if (id == 5) save();
			else if (id == 6) load();
			
			if (skipping) break;
		}
		location++;
	}
	
	ubyte waitChoice(char[][] choices ...) /// 選択肢の順に1, 2, 3...を返す
	{
		if (loadingTarget)
		{
			if (loadingTarget == location)
			{
				loadingTarget = null;
				message("loadしました", vector(0, 0, 1, 0.5));
			}
			else
			{
				ubyte result = Location.nextChoice(loadingTarget, location);
				location.addChoice(result);
				return result;
			}
		}
		
		if (choices.length == 0) throw new Error("waitChoice");
		
		Waiter waiter = new Waiter(this);
		UnitMerger merger = new UnitMerger();
		for (int i = 0; i < choices.length; i++)
		{
			Button button = new Button(new Text(306, 200 + 50 * i, choices[i]), factory);
			waiter.addEvent(1 + i, button.clickEvent);
			merger ~= button;
		}
		waiter.addEvent(ubyte.max + 5, saveButton.clickEvent);
		waiter.addEvent(ubyte.max + 6, loadButton.clickEvent);
		choicesNode.unit = merger;
		waiter.killTime = delegate uint(RenderTarget rt)
		{
			while (true)
			{
				rt.draw(rootNode);
				rt.flip();
			}
			return 0;
		};
		ubyte result;
		while (true)
		{
			uint id = waiter.wait();
			if (id <= ubyte.max)
			{
				result = id;
				break;
			}
			else if (id == ubyte.max + 5) save();
			else if (id == ubyte.max + 6) load();
		}
		choicesNode.unit = null;
		location.addChoice(result);
		return result;
	}
	
	void message(char[] a, Vector bgcolor) ///
	{
		messageNode.unit = new TermUnit(
			new UnitMerger(
				new Quadrangle(0, 250, 800, 40, bgcolor),
				new Text(350, 250, a)
			),
			1000
		);
	}
	
	private void save()
	{
		if (!exists("save")) mkdir("save");
		isdir("save");
		
		/*
		d_time t = UTCtoLocalTime(getUTCtime()); // まだ実装されてない?
		char[] fileName = format(
			`save\_%02d%02d%02d_%02d%02d%02d`,
			YearFromTime(t) % 100,
			MonthFromTime(t) + 1,
			DateFromTime(t),
			HourFromTime(t),
			MinFromTime(t),
			SecFromTime(t)
		);
		*/
		char[] fileName = format(`save\_%x`, getUTCtime());
		
		write(fileName, location.toString());
		Rgba rgba = new RgbaScaler(new WindowRgba(this), 133, 100);
		writePng(fileName ~ ".png", rgba.pixels, rgba.width, rgba.height);
		
		message("saveしました", vector(1, 0, 0, 0.5));
	}
	
	private void load()
	{
		char[][] fileList = listdir("save"); // sort済み
		int currentIndex = fileList.length / 2 / 10 * 10;
		
		Node loadNode = new Node(new Clear());
		Button previousButton = new Button(new Text(0, 550, "previous"), factory);
		Button nextButton = new Button(new Text(700, 550, "next"), factory);
		loadNode ~= new Node(previousButton);
		loadNode ~= new Node(nextButton);
		Node[] fileButtonNodes;
		for (int i = 0; i < 10; i++) fileButtonNodes ~= new Node();
		foreach (Node a; fileButtonNodes) loadNode ~= a;
		Node info = new Node();
		loadNode ~= info;
		
		Waiter waiter = new Waiter(this);
		
		void reloadButtons()
		{
			waiter.clearEvents();
			waiter.addEvent(1, new SdlRightClickEvent(0, 0, 800, 600));
			waiter.addEvent(2, previousButton.clickEvent);
			waiter.addEvent(3, nextButton.clickEvent);
			
			info.unit = new Text(
				400, 550, format("%d / %d", currentIndex, fileList.length / 2)
			);
			for (int i = 0; i < 10; i++)
			{
				fileButtonNodes[i].unit = null;
				int fileIndex = currentIndex * 2 + i * 2;
				if (fileList.length <= fileIndex) continue;
				
				int x = i < 5 ? 0 : 400;
				int y = i * 100 % 500;
				Button button = new Button(new Text(x + 150, y + 50, fileList[fileIndex]), factory);
				Image image = new Image(`save\` ~ fileList[fileIndex + 1], x, y);
				fileButtonNodes[i].unit = new UnitMerger(button, image);
				waiter.addEvent(4 + i, button.clickEvent);
			}
		}
		
		reloadButtons();
		
		waiter.killTime = delegate uint(RenderTarget rt)
		{
			while (true)
			{
				rt.draw(loadNode);
				rt.flip();
			}
			return 0;
		};
		char[] fileName;
		while (true)
		{
			uint id = waiter.wait();
			if (id == 1) return;
			else if (id == 2 && 10 <= currentIndex)
			{
				currentIndex -= 10;
				reloadButtons();
			}
			else if (id == 3)
			{
				currentIndex += 10;
				reloadButtons();
			}
			else if (id >= 4)
			{
				fileName = `save\` ~ fileList[currentIndex * 2 + (id - 4) * 2];
				break;
			}
		}
		
		loadingTarget = new Location(cast(char[])read(fileName));
		throw new LoadEvent();
	}
}

class Location
{
	unittest
	{
		Location l = new Location();
		assert("_0" == l.toString());
		l.fromString("_11");
		assert(11 == l.clickLength);
		l.addChoice(2);
		assert("2_0" == l.toString());
		l++;
		assert("2_1" == l.toString());
		l.addChoice(3);
		assert("23_0" == l.toString());
		l++;
		assert(new Location("23_1") == l);
		assert(4 == nextChoice(new Location("2345_0"), l));
	}
	
	char[] choicesString;
	uint clickLength = 0;
	
	this() {}
	this(char[] a) { fromString(a); }
	
	char[] toString()
	{
		return choicesString ~ "_" ~ std.string.toString(clickLength);
	}
	
	void fromString(char[] a)
	{
		uint _ = find(a, "_");
		if (-1 == _) throw new Error("fromString");
		choicesString = a[0.._];
		clickLength = toUint(a[_ + 1..$]);
	}
	
	void addChoice(uint choices)
	{
		choicesString ~= std.string.toString(choices);
		clickLength = 0;
	}
	
	void opPostInc()
	{
		clickLength++;
	}
	
	bool opEquals(Location a)
	{
		return toString() == a.toString();
	}
	
	static uint nextChoice(Location target, Location current)
	{
		char[] c0 = target.choicesString;
		char[] c1 = current.choicesString;
		if (c0.length <= c1.length) throw new Error("nextChoice");
		if (0 != c0.find(c1)) throw new Error("nextChoice");
		return toUint("" ~ c0[c1.length]);
	}
}
