﻿module coneneko.sdlwindow;
public import
	coneneko.scenegraph, coneneko.units, coneneko.math,
	coneneko.nanami, coneneko.camera, coneneko.exunits, coneneko.texture;
import std.string, std.c.string, std.file;
import sdl, sdl_ttf, sdl_image, sdl_mixer;

/// できるだけautoで
class SdlWindow : Window
{
	private bool isQuit = false;
	
	///
	this(uint width = 640, uint height = 480)
	{
		if(SDL_Init(SDL_INIT_VIDEO) == -1) sdlException(__FILE__, __LINE__, this);
		
		SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
		SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
		SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
		if (!SDL_SetVideoMode(width, height, 24, SDL_OPENGL)) sdlException(__FILE__, __LINE__, this);
	}
	
	~this()
	{
		Font.close();
		SdlJoystickEvent.close();
		SdlSoundMixer.close();
		SDL_Quit();
	}
	
	void draw(Node a)
	{
		a();
	}
	
	void flip()
	{
		SDL_GL_SwapBuffers();
		doEvents();
	}
	
	bool alive()
	{
		return isQuit ? false : true;
	}
	
	uint width()
	{
		return SDL_GetVideoSurface().w;
	}
	
	uint height()
	{
		return SDL_GetVideoSurface().h;
	}
	
	private void doEvents()
	{
		SDL_Delay(1);
		SDL_Event event;
		while (SDL_PollEvent(&event))
		{
			foreach (SdlEvent e; events)
			{
				if (e.isReached(event)) reachedEvents ~= e;
			}
			
			if (event.type == SDL_QUIT)
			{
				isQuit = true;
				return;
			}
		}
	}
	
	
	private SdlEvent[] events;
	private Event[] reachedEvents;
	
	void opCatAssign(Event e)
	{
		SdlEvent se = cast(SdlEvent)e;
		if (!se) sdlException(__FILE__, __LINE__, this);
		events ~= se;
	}
	
	void clearEvents()
	{
		events = null;
	}
	
	Event popReachedEvent()
	{
		if (reachedEvents.length == 0) return null;
		Event result = reachedEvents[0];
		reachedEvents = reachedEvents[1..$];
		return result;
	}
	
	
	MousePoint mousePoint()
	{
		return new class MousePoint
		{
			int rx, ry;
			
			void update()
			{
				SDL_GetMouseState(&rx, &ry);
			}
			
			int x()
			{
				update();
				return rx;
			}
			
			int y()
			{
				update();
				return ry;
			}
		};
	}
}

///
abstract class SdlEvent : Event
{
	bool isReached(SDL_Event e); ///
}

///
class SdlClickEventBase : SdlEvent
{
	private final int x, y, width, height;
	private final ubyte button;
	private final void delegate() onClick;
	
	///
	this(int x, int y, int width, int height, ubyte button, void delegate() onClick)
	{
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
		this.button = button;
		this.onClick = onClick;
	}
	
	bool isReached(SDL_Event e)
	{
		bool result = e.button.type == SDL_MOUSEBUTTONDOWN
			&& e.button.button == button
			&& isInner(e.button.x, e.button.y);
		if (result && onClick) onClick();
		return result;
	}
	
	private bool isInner(int mouseX, int mouseY)
	{
		return x <= mouseX && mouseX <= x + width
			&& y <= mouseY && mouseY <= y + height;
	}
}

///
class SdlClickEvent : SdlClickEventBase, VirtualSizeUnit
{
	///
	this(int x, int y, int width, int height, void delegate() onClick = null)
	{
		int rx, ry, rwidth, rheight;
		toReal(x, y, width, height, rx, ry, rwidth, rheight);
		super(rx, ry, rwidth, rheight, SDL_BUTTON_LEFT, onClick);
	}
	
	///
	static void toReal(
		int x, int y, int width, int height,
		out int rx, out int ry, out int rwidth, out int rheight
	)
	{
		int ww = SDL_GetVideoSurface().w;
		int wh = SDL_GetVideoSurface().h;
		rx = x * ww / virtualWidth;
		ry = y * wh / virtualHeight;
		rwidth = width * ww / virtualWidth;
		rheight = height * wh / virtualHeight;
	}
	
	void attach() {}
	void detach() {}
}

///
class SdlRightClickEvent : SdlClickEventBase, VirtualSizeUnit
{
	///
	this(int x, int y, int width, int height, void delegate() onClick = null)
	{
		int rx, ry, rwidth, rheight;
		SdlClickEvent.toReal(x, y, width, height, rx, ry, rwidth, rheight);
		super(rx, ry, rwidth, rheight, SDL_BUTTON_RIGHT, onClick);
	}
	
	void attach() {}
	void detach() {}
}

///
class SdlKeyEvent : SdlEvent
{
	static assert(97 == 'a');
	
	private	final int sym;
	private final int keyType; // SDL_KEYDOWN or SDL_KEYUP
	
	///
	this(int sym, int keyType)
	{
		this.sym = sym;
		this.keyType = keyType;
	}
	
	bool isReached(SDL_Event e)
	{
		return e.key.type == keyType &&	e.key.keysym.sym ==	sym;
	}
}

/// 97 == 'a' CapsLockの影響を受けない
class SdlKeyDownEvent : SdlKeyEvent
{
	///
	this(int sym)
	{
		super(sym, SDL_KEYDOWN);
	}
}

///
class SdlKeyUpEvent : SdlKeyEvent
{
	///
	this(int sym)
	{
		super(sym, SDL_KEYUP);
	}
}

///
class SdlButtonEventFactory : ButtonEventFactory
{
	private final SdlWindow wnd;
	
	///
	this(SdlWindow wnd)
	{
		this.wnd = wnd;
	}
	
	Event create(int x, int y, int width, int height, void delegate() onClick)
	{
		return new SdlClickEvent(x, y, width, height, onClick);
	}
	
	VirtualMousePoint virtualMousePoint()
	{
		return new VirtualMousePoint(wnd);
	}
}

///
class SdlCamera : SdlEvent, Unit
{
	private Camera camera;
	private LoadMatrix lm;
	
	///
	this()
	{
		camera = new Camera();
		camera.dmove = 2400;
		lm = new LoadMatrix(Matrix.identity);
	}
	
	bool isReached(SDL_Event e)
	{
		bool down = e.type == SDL_MOUSEBUTTONDOWN;
		bool up = e.type == SDL_MOUSEBUTTONUP;
		bool left = e.button.button == SDL_BUTTON_LEFT;
		bool right = e.button.button == SDL_BUTTON_RIGHT;
		if (left && down) camera.pushLeft();
		else if (left && up) camera.popLeft();
		else if (right && down) camera.pushRight();
		else if (right && up) camera.popRight();
		return false;
	}
	
	void attach()
	{
		int mx, my;
		SDL_GetMouseState(&mx, &my);
		camera.tick(
			cast(float)mx / SDL_GetVideoSurface().w * 2.0 - 1.0,
			cast(float)my / SDL_GetVideoSurface().h * -2.0 + 1.0
		);
		
		lm.matrix = camera.getViewMatrix();
		lm.attach();
	}
	
	void detach()
	{
		lm.detach();
	}
}

///
class SdlSoundEffect
{
	private int channel;
	final char[] fileName; ///
	
	///
	this(char[] fileName, bool loop)
	{
		this.fileName = fileName;
		channel = SdlSoundMixer.getInstance().playSoundEffect(fileName, loop);
	}
	
	///
	~this()
	{
		stop();
	}
	
	void stop() ///
	{
		Mix_HaltChannel(channel);
	}
	
	bool playing() ///
	{
		return Mix_Playing(channel) ? true : false;
	}
}

///
class SdlSoundMixer
{
	private static SdlSoundMixer _instance;
	
	static SdlSoundMixer getInstance()
	{
		if (!_instance) _instance = new SdlSoundMixer();
		return _instance;
	}
	
	static void close() /// サウンドリソース開放
	{
		delete _instance;
	}
	
	
	private Mix_Chunk*[char[]] chunkMap;
	private Mix_Music* music = null;
	char[] musicFileName; ///
	
	private this()
	{
		Mix_OpenAudio(44100, AUDIO_S16, 2, 4096);
	}
	
	~this()
	{
		foreach (Mix_Chunk* a; chunkMap.values) Mix_FreeChunk(a);
		Mix_CloseAudio();
		stopMusic();
	}
	
	int playSoundEffect(char[] fileName, bool loop) ///
	{
		if (!(fileName in chunkMap))
		{
			chunkMap[fileName] = Mix_LoadWAV_RW(
				SDL_RWFromFile(toMBSz(fileName), "rb"),
				1
			);
			if (!chunkMap[fileName])
			{
				chunkMap.remove(fileName);
				exception(__FILE__, __LINE__, this, "can not load se " ~ fileName);
			}
		}
		return Mix_PlayChannelTimed(-1, chunkMap[fileName], loop ? -1 : 0, -1);
	}
	
	void playMusic(char[] fileName) ///
	{
		stopMusic();
		music = Mix_LoadMUS(toMBSz(fileName));
		if (!music) exception(__FILE__, __LINE__, this, "can not load bgm " ~ fileName);
		musicFileName = fileName;
		Mix_PlayMusic(music, -1);
	}
	
	void stopMusic() ///
	{
		if (!music) return;
		Mix_HaltMusic();
		Mix_FreeMusic(music);
		music = null;
		musicFileName = "";
	}
}

///
abstract class SdlJoystickEvent : SdlEvent
{
	private static SDL_Joystick* joystick;
	
	///
	this()
	{
		if (!SDL_WasInit(SDL_INIT_JOYSTICK))
		{
			SDL_InitSubSystem(SDL_INIT_JOYSTICK);
			if (1 <= SDL_NumJoysticks()) joystick = SDL_JoystickOpen(0);
		}
	}
	
	static void close()
	{
		if (!SDL_WasInit(SDL_INIT_JOYSTICK)) return;
		
		if (joystick) SDL_JoystickClose(joystick);
		SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
	}
}

///
class SdlJoystickRightEvent : SdlJoystickEvent
{
	bool isReached(SDL_Event e)
	{
		if (e.type != SDL_JOYAXISMOTION) return false;
		with (e.jaxis) return axis == 0 && short.max / 2 < value;
	}
}

///
class SdlJoystickLeftEvent : SdlJoystickEvent
{
	bool isReached(SDL_Event e)
	{
		if (e.type != SDL_JOYAXISMOTION) return false;
		with (e.jaxis) return axis == 0 && value < short.min / 2;
	}
}

///
class SdlJoystickUpEvent : SdlJoystickEvent
{
	bool isReached(SDL_Event e)
	{
		if (e.type != SDL_JOYAXISMOTION) return false;
		with (e.jaxis) return axis == 1 && value < short.min / 2;
	}
}

///
class SdlJoystickDownEvent : SdlJoystickEvent
{
	bool isReached(SDL_Event e)
	{
		if (e.type != SDL_JOYAXISMOTION) return false;
		with (e.jaxis) return axis == 1 && short.max / 2 < value;
	}
}

///
class SdlJoystickButtonEvent : SdlJoystickEvent
{
	private final uint buttonId;
	
	///
	this(uint buttonId)
	{
		this.buttonId = buttonId;
	}
	
	bool isReached(SDL_Event e)
	{
		if (e.type != SDL_JOYBUTTONDOWN) return false;
		return e.jbutton.button == buttonId;
	}
}
