﻿// とりあえずSDL依存、image, fontを扱う
module coneneko.texture;
import coneneko.scenegraph;
import std.string, std.file, std.c.string;
version (SDL_WINDOW)
{
	import sdl, sdl_ttf, sdl_image;
}

/// SDL_GetError付き、throw new ConenekoException(__FILE__, __LINE__, this, "message");
void sdlException(char[] file, long line, Object obj = null, char[] msg = null)
{
	exception(file, line, obj, msg ~ " " ~ std.string.toString(SDL_GetError()));
}

template SdlRgba32()
{
	static uint[] createRgba32Pixels(SDL_Surface* surface)
	{
		SDL_PixelFormat format = rgba32PixelFormat;
		surface = SDL_ConvertSurface(surface, &format, SDL_SWSURFACE);
		uint[] result = new uint[surface.w * surface.h];
		memcpy(result.ptr, surface.pixels, uint.sizeof * result.length);
		return result;
	}
	
	static SDL_PixelFormat rgba32PixelFormat()
	{
		SDL_PixelFormat result;
		with (result)
		{
			palette = null;
			BitsPerPixel = 32;
			BytesPerPixel = 4;
			Rmask = 0xff;
			Gmask = 0xff00;
			Bmask = 0xff0000;
			Amask = 0xff000000;
			Rshift = 0;
			Gshift = 8;
			Bshift = 16;
			Ashift = 24;
			Rloss = Gloss = Bloss = Aloss = 0;
			colorkey = 0;
			alpha = 255;
		}
		return result;
	}
	
	static uint[] readImage(char[] fileName, out uint width, out uint height)
	{
		SDL_Surface* image = IMG_Load(toMBSz(fileName));
		if (!image) sdlException(__FILE__, __LINE__);
		width = image.w;
		height = image.h;
		uint[] result = createRgba32Pixels(image);
		SDL_FreeSurface(image);
		return result;
	}
	
	static void computeSize(Font font, char[] text, out int width, out int height)
	{
		if (TTF_SizeUTF8(cast(TTF_Font*)font.handle, toStringz(text), &width, &height) == -1)
		{
			sdlException(__FILE__, __LINE__);
		}
	}
	
	static uint[] createTextImage(Font font, char[] text, ubyte r, ubyte g, ubyte b, ubyte a)
	{
		SDL_Surface* textSurface;
		try
		{
			textSurface = TTF_RenderUTF8_Blended(
				cast(TTF_Font*)font.handle, toStringz(text), toColor(r, g, b, a)
			);
		}
		catch (Object e)
		{
			// 時々作成に失敗するらしい、原因はよくわからない
			sdlException(__FILE__, __LINE__, null, "TTF_RenderUTF8_Blended " ~ text);
		}
		if (!textSurface) sdlException(__FILE__, __LINE__, null, text);
		
		uint[] result = createRgba32Pixels(textSurface);
		SDL_FreeSurface(textSurface);
		return result;
	}
	
	static assert(Uint8.max == ubyte.max);
	
	static SDL_Color toColor(ubyte r, ubyte g, ubyte b, ubyte a)
	{
		SDL_Color result;
		result.r = r;
		result.g = g;
		result.b = b;
		result.unused = a;
		return result;
	}
}

template SdlFont()
{
	import sdl_ttf;
	private static void*[char[]] fontMap;
	
	static void closeFont()
	{
		foreach (void* a; fontMap.values) TTF_CloseFont(cast(TTF_Font*)a);
		fontMap = null;
		TTF_Quit();
	}
	
	static void* getFont(char[] ttfFileName, uint size)
	{
		if (!fontMap) TTF_Init();
		char[] key = format("%s%d", ttfFileName, size);
		if (!(key in fontMap)) fontMap[key] = TTF_OpenFont(toStringz(ttfFileName), size);
		return fontMap[key];
	}
}

///
class ImageLoader : Rgba
{
	version (SDL_WINDOW) mixin SdlRgba32;
	
	private final uint _width, _height;
	private final uint[] _pixels;
	
	///
	this(char[] fileName)
	{
		_pixels = readImage(fileName, _width, _height);
	}
	
	uint width()
	{
		return _width;
	}
	
	uint height()
	{
		return _height;
	}
	
	uint[] pixels()
	{
		return _pixels;
	}
}

///
class Font
{
	version (SDL_WINDOW) mixin SdlFont;
	
	static void close() /// リソース開放
	{
		closeFont();
	}
	
	void* handle; ///
	
	///
	this(char[] ttfFileName, uint fontSize)
	{
		handle = getFont(ttfFileName, fontSize);
	}
	
	uint height() ///
	{
		return TTF_FontHeight(cast(TTF_Font*)handle);
	}
}

///
class DefaultFont : Font
{
	private const char[] TTF_FILE_NAME = `C:\WINDOWS\Fonts\msgothic.ttc`;
	private const uint FONT_SIZE = 32;
	
	///
	this()
	{
		super(TTF_FILE_NAME, FONT_SIZE);
	}
}

/// sizeが小さかったり、TTF fileによっては壊れたイメージになることがある
class TextLoader : Rgba
{
	version (SDL_WINDOW) mixin SdlRgba32;
	
	private final int _width, _height;
	private final uint[] _pixels;
	
	///
	this(Font font, char[] text, ubyte red, ubyte green, ubyte blue, ubyte alpha)
	{
		computeSize(font, text, _width, _height);
		_pixels = createTextImage(font, text, red, green, blue, alpha);
	}
	
	uint width()
	{
		return _width;
	}
	
	uint height()
	{
		return _height;
	}
	
	uint[] pixels()
	{
		return _pixels;
	}
}
