﻿/**
render stateはdetachで値を修復しておく。
textureやshaderなど、他に影響を与えるものもdetachで修復しておく。
他に影響を与えないものは、修復しなくてもいい。
*/
module coneneko.units;
import coneneko.scenegraph, coneneko.math;
import std.math, std.string;
version (DWT_WINDOW)
{
	import coneneko.glext;
}
version (SDL_WINDOW)
{
	import coneneko.glext;
	
	static assert(ubyte.max == GLboolean.max);
	static assert(int.max == GLint.max);
	static assert(int.max == GLsizei.max);
	static assert(uint.max == GLuint.max);
	static assert(uint.max == GLenum.max);
	static assert(uint.max == GLbitfield.max);
	static assert(uint.max == GLhandleARB.max);
	static assert(float.max == GLclampf.max);
	static assert(float.max == GLfloat.max);
	static assert(double.max == GLdouble.max);
}

///
class Viewport : Unit
{
	///
	int x, y, width, height;
	
	private int[4] stock;
	
	///
	this(int x, int y, int width, int height)
	{
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
	}
	
	void attach()
	{
		glGetIntegerv(GL_VIEWPORT, stock.ptr);
		glViewport(x, y, width, height);
	}
	
	void detach()
	{
		glViewport(stock[0], stock[1], stock[2], stock[3]);
	}
}

///
class Perspective : Unit
{
	///
	double fovy, aspect, zNear, zFar;
	
	///
	this(double fovy, double aspect, double zNear, double zFar)
	{
		this.fovy = fovy;
		this.aspect = aspect;
		this.zNear = zNear;
		this.zFar = zFar;
	}
	
	void attach()
	{
		glPushMatrix();
		gluPerspective(fovy, aspect, zNear, zFar);
	}
	
	void detach()
	{
		glPopMatrix();
	}
}

///
class MatrixMode : Unit
{
	///
	int mode;
	
	private int stock;
	
	/// GL_MODELVIEW GL_PROJECTION GL_TEXTURE
	this(uint mode)
	{
		this.mode = mode;
	}
	
	void attach()
	{
		glGetIntegerv(GL_MATRIX_MODE, &stock);
		glMatrixMode(mode);
	}
	
	void detach()
	{
		glMatrixMode(stock);
	}
}

///
class LoadIdentity : Unit
{
	void attach()
	{
		glPushMatrix();
		glLoadIdentity();
	}
	
	void detach()
	{
		glPopMatrix();
	}
}

/// GL_COLOR_BUFFER_BIT GL_DEPTH_BUFFER_BIT GL_ACCUM_BUFFER_BIT GL_STENCIL_BUFFER_BIT
class Clear : Unit
{
	///
	uint mask;
	
	/// GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT
	this()
	{
		this(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	}
	
	///
	this(uint mask)
	{
		this.mask = mask;
	}
	
	void attach()
	{
		glClear(mask);
	}
	
	void detach() {}
}

///
class ClearColor : Unit
{
	///
	float red, green, blue, alpha;
	
	private float[4] stock;
	
	///
	this(float red = 0.0, float green = 0.0, float blue = 0.0, float alpha = 0.0)
	{
		this.red = red;
		this.green = green;
		this.blue = blue;
		this.alpha = alpha;
	}
	
	void attach()
	{
		glGetFloatv(GL_COLOR_CLEAR_VALUE, stock.ptr);
		glClearColor(red, green, blue, alpha);
	}
	
	void detach()
	{
		glClearColor(stock[0], stock[1], stock[2], stock[3]);
	}
}

///
class Enable : Unit
{
	///
	uint cap;
	
	private ubyte stock;
	
	///
	this(uint cap)
	{
		this.cap = cap;
	}
	
	void attach()
	{
		stock = glIsEnabled(cap);
		glEnable(cap);
	}
	
	void detach()
	{
		stock == GL_TRUE ? glEnable(cap) : glDisable(cap);
		glErrorCheck(__FILE__, __LINE__, this);
	}
}

///
class CullFace : Unit
{
	///
	uint mode;
	
	private int stock;
	
	/// GL_FRONT GL_BACK GL_FRONT_AND_BACK
	this(uint mode)
	{
		this.mode = mode;
	}
	
	void attach()
	{
		glGetIntegerv(GL_CULL_FACE_MODE, &stock);
		glCullFace(mode);
	}
	
	void detach()
	{
		glCullFace(stock);
	}
}

///
class Rotate : Unit
{
	///
	float angle, x, y, z;
	
	///
	this(float angle, float x, float y, float z)
	{
		this.angle = angle;
		this.x = x;
		this.y = y;
		this.z = z;
	}
	
	void attach()
	{
		glPushMatrix();
		glRotatef(angle, x, y, z);
	}
	
	void detach()
	{
		glPopMatrix();
	}
}

///
class Translate : Unit
{
	///
	float x, y, z;
	
	///
	this(float x, float y, float z)
	{
		this.x = x;
		this.y = y;
		this.z = z;
	}
	
	void attach()
	{
		glPushMatrix();
		glTranslatef(x, y, z);
	}
	
	void detach()
	{
		glPopMatrix();
	}
}

/// エラーが発生していたら、throw new ConenekoException(__FILE__, __LINE__, this, "message");
void glErrorCheck(char[] file, long line, Object obj = null, char[] msg = null)
{
	char[] msg2;
	switch (glGetError())
	{
		case GL_NO_ERROR:          return;
		case GL_INVALID_ENUM:      msg2 = "GL_INVALID_ENUM";		break;
		case GL_INVALID_VALUE:     msg2 = "GL_INVALID_VALUE";		break;
		case GL_INVALID_OPERATION: msg2 = "GL_INVALID_OPERATION";	break;
		case GL_STACK_OVERFLOW:    msg2 = "GL_STACK_OVERFLOW";		break;
		case GL_STACK_UNDERFLOW:   msg2 = "GL_STACK_UNDERFLOW";		break;
		case GL_OUT_OF_MEMORY:     msg2 = "GL_OUT_OF_MEMORY";		break;
		default:                   msg2 = "OpenGL unknown error";	break;
	}
	exception(file, line, obj, msg ~ " " ~ msg2);
}

/// 使い終わったらdelete
class Texture : Unit, Rgba
{
	///
	protected final uint handle;
	
	private int stock;
	private final Unit enableTexture;
	private final uint index;
	
	///
	this(Rgba rgba, uint index)
	{
		this(rgba.width, rgba.height, rgba.pixels, index);
	}
	
	///
	this(uint width, uint height, uint[] pixels, uint index)
	{
		glErrorCheck(__FILE__, __LINE__, this);
		
		if (!glActiveTextureARB)
		{
			load_GL_ARB_multitexture();
			if (!glActiveTextureARB) exception(__FILE__, __LINE__, this, "load_GL_ARB_multitexture failed");
		}
		
		this.index = index;
		enableTexture = new Enable(GL_TEXTURE_2D);
		if (width != TextureSizeCorrector.correctLength(width)
			|| height != TextureSizeCorrector.correctLength(height))
		{
			exception(__FILE__, __LINE__, this, "texture size");
		}
		glGenTextures(1, &handle);
		attach();
		/*
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		//*/
		//*
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		//*/
		glTexImage2D(
			GL_TEXTURE_2D, 0, internalformat, width, height, 0, GL_RGBA, dataType, pixels
		);
		detach();
		glErrorCheck(__FILE__, __LINE__, this);
	}
	
	~this()
	{
		glDeleteTextures(1, &handle);
	}
	
	protected int internalformat() /// GL_RGBA8
	{
		return GL_RGBA8;
	}
	
	protected uint dataType() /// GL_UNSIGNED_BYTE
	{
		return GL_UNSIGNED_BYTE;
	}
	
	void attach()
	{
		glActiveTextureARB(GL_TEXTURE0_ARB + index);
		glGetIntegerv(GL_TEXTURE_BINDING_2D, &stock);
		glBindTexture(GL_TEXTURE_2D, handle);
		enableTexture.attach();
	}
	
	void detach()
	{
		glActiveTextureARB(GL_TEXTURE0_ARB + index);
		enableTexture.detach();
		glBindTexture(GL_TEXTURE_2D, stock);
	}
	
	uint width()
	{
		int result;
		attach();
		glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_WIDTH, &result);
		detach();
		return result;
	}
	
	uint height()
	{
		int result;
		attach();
		glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_HEIGHT, &result);
		detach();
		return result;
	}
	
	uint[] pixels()
	{
		uint[] result = new uint[width *  height];
		attach();
		glGetTexImage(GL_TEXTURE_2D, 0, GL_RGBA, dataType, result.ptr);
		detach();
		return result;
	}
}

///
class LoadMatrix : Unit
{
	Matrix matrix; ///
	
	///
	this(Matrix matrix)
	{
		this.matrix = matrix;
	}
	
	void attach()
	{
		glPushMatrix();
		glLoadMatrixf(cast(float*)&matrix);
	}
	
	void detach()
	{
		glPopMatrix();
	}
}

///
class MultMatrix : Unit
{
	Matrix matrix; ///
	
	///
	this(Matrix matrix)
	{
		this.matrix = matrix;
	}
	
	void attach()
	{
		glPushMatrix();
		glMultMatrixf(cast(float*)&matrix);
	}
	
	void detach()
	{
		glPopMatrix();
	}
}

///
class BlendFunc : Unit
{
	///
	uint sfactor, dfactor;
	
	private int stockSrc, stockDst;
	
	/// GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
	this()
	{
		this(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}
	
	///
	this(uint sfactor, uint dfactor)
	{
		this.sfactor = sfactor;
		this.dfactor = dfactor;
	}
	
	void attach()
	{
		glGetIntegerv(GL_BLEND_SRC, &stockSrc);
		glGetIntegerv(GL_BLEND_DST, &stockDst);
		glBlendFunc(sfactor, dfactor);
	}
	
	void detach()
	{
		glBlendFunc(stockSrc, stockDst);
	}
}

/// 使ったらdelete
class VertexBuffer : Unit
{
	private final uint handle;
	
	///
	this(void[] data)
	{
		if (!glBufferDataARB)
		{
			load_GL_ARB_vertex_buffer_object();
			if (!glBufferDataARB)
			{
				exception(__FILE__, __LINE__, this, "load_GL_ARB_vertex_buffer_object failed");
			}
		}
		
		glGenBuffersARB(1, &handle);
		bindBuffer(); // overrideされたものを間違えて呼ばないように注意
		glBufferDataARB(
			GL_ARRAY_BUFFER_ARB,
			void.sizeof * data.length,
			data.ptr,
			GL_STATIC_DRAW_ARB
		);
		
		glErrorCheck(__FILE__, __LINE__, this);
	}
	
	~this()
	{
		glDeleteBuffersARB(1, &handle);
	}
	
	void attach()
	{
		bindBuffer();
	}
	
	private void bindBuffer()
	{
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, handle);
	}
	
	void detach() {}
}

///
class PositionBuffer : VertexBuffer
{
	private final uint vertexLength;
	
	/// float3
	this(float[] positions)
	{
		super(positions);
		vertexLength = positions.length / 3;
	}
	
	override void attach()
	{
		super.attach();
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3, GL_FLOAT, 0, null);
		
		glDrawArrays(GL_TRIANGLES, 0, vertexLength);
	}
	
	override void detach()
	{
		glDisableClientState(GL_VERTEX_ARRAY);
		super.detach();
	}
}

///
class ColorBuffer : VertexBuffer
{
	/// float4
	this(float[] rgba)
	{
		super(rgba);
	}
	
	override void attach()
	{
		super.attach();
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_FLOAT, 0, null);
	}
	
	override void detach()
	{
		glDisableClientState(GL_COLOR_ARRAY);
		super.detach();
	}
}

///
class NormalBuffer : VertexBuffer
{
	/// float3
	this(float[] normals)
	{
		super(normals);
	}
	
	override void attach()
	{
		super.attach();
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(GL_FLOAT, 0, null);
	}
	
	override void detach()
	{
		glDisableClientState(GL_NORMAL_ARRAY);
		super.detach();
	}
}

///
class TexCoordBuffer : VertexBuffer
{
	/// float2
	this(float[] texCoords)
	{
		super(texCoords);
	}
	
	override void attach()
	{
		super.attach();
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, GL_FLOAT, 0, null);
	}
	
	override void detach()
	{
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		super.detach();
	}
}

///
class AttributeBuffer : VertexBuffer
{
	private final uint floatLength;
	
	/// shaderのglGetAttribLocationARB
	uint location;
	
	/// floatN, N = floatLength
	this(void[] data, uint floatLength)
	{
		glErrorCheck(__FILE__, __LINE__, this);
		
		if (!glEnableVertexAttribArrayARB)
		{
			load_GL_ARB_vertex_program();
			if (!glEnableVertexAttribArrayARB)
			{
				exception(__FILE__, __LINE__, this, "load_GL_ARB_vertex_program failed");
			}
		}
		
		super(data);
		this.floatLength = floatLength;
		
		glErrorCheck(__FILE__, __LINE__, this);
	}
	
	override void attach()
	{
		super.attach();
		glEnableVertexAttribArrayARB(location);
		glVertexAttribPointerARB(location, floatLength, GL_FLOAT, false, 0, null);
	}
	
	override void detach()
	{
		glDisableVertexAttribArrayARB(location);
		super.detach();
	}
}

/// glsl
class Shader : Unit
{
	private uint handle;
	
	///
	this(char[] vertexSource, char[] fragmentSource)
	{
		if (!glCreateProgramObjectARB)
		{
			load_GL_ARB_shader_objects();
			load_GL_ARB_vertex_shader();
			if (!glCreateProgramObjectARB) exception(__FILE__, __LINE__, this, "load_GL_ARB_shader_objects failed");
			if (!glGetAttribLocationARB) exception(__FILE__, __LINE__, this, "load_GL_ARB_vertex_shader");
		}
		
		handle = createShader(vertexSource, fragmentSource);
	}
	
	private int createShader(char[] vertexSource, char[] fragmentSource)
	{
		GLhandleARB result;
		GLhandleARB vsObj, fsObj;
		try
		{
			result = glCreateProgramObjectARB();
			if (!result) exception(__FILE__, __LINE__, this, "glCreateProgramObjectARB");
			
			vsObj = compileShader(GL_VERTEX_SHADER_ARB, vertexSource);
			fsObj = compileShader(GL_FRAGMENT_SHADER_ARB, fragmentSource);
			
			glAttachObjectARB(result, vsObj);
			glAttachObjectARB(result, fsObj);
			
			glLinkProgramARB(result);
			GLint resultL;
			glGetObjectParameterivARB(result, GL_OBJECT_LINK_STATUS_ARB, &resultL);
			if (!resultL) exception(__FILE__, __LINE__, this, "shader link error");
		}
		catch (Exception e) { throw e; }
		finally
		{
			if (fsObj) glDeleteObjectARB(fsObj);
			if (vsObj) glDeleteObjectARB(vsObj);
		}
		return result;
	}
	
	private int compileShader(uint shaderType, char[] source)
	{
		GLhandleARB result = glCreateShaderObjectARB(shaderType);
		if (!result) exception(__FILE__, __LINE__, this, "glCreateShaderObjectARB");
		
		char* sourcez = toStringz(source);
		int sourceLength = source.length;
		glShaderSourceARB(result, 1, &sourcez, &sourceLength);
		
		int resultC;
		glCompileShaderARB(result);
		glGetObjectParameterivARB(result, GL_OBJECT_COMPILE_STATUS_ARB, &resultC);
		if (!resultC)
		{
			int infoLogLength;
			glGetObjectParameterivARB(result, GL_OBJECT_INFO_LOG_LENGTH_ARB, &infoLogLength);
			assert(0 < infoLogLength);
			
			char[] infoLog = new char[infoLogLength];
			GLsizei r;
			glGetInfoLogARB(result, infoLog.length, &r, infoLog.ptr);
			assert(infoLog.length - 1 == r);
			
			exception(__FILE__, __LINE__, this, "shader compile error: " ~ infoLog);
		}
		return result;
	}
	
	~this()
	{
		glDeleteObjectARB(handle);
	}
	
	void attach()
	{
		glUseProgramObjectARB(handle);
	}
	
	void detach()
	{
		glUseProgramObjectARB(0);
	}
	
	protected int getUniformLocation(char[] key) /// 使わない値には代入できないようなので注意
	{
		int result = glGetUniformLocationARB(handle, toStringz(key));
		if (-1 == result) exception(__FILE__, __LINE__, this, "UniformLocation: " ~ key);
		return result;
	}
	
	void setFloat(char[] key, float a) ///
	{
		attach();
		glUniform1fARB(getUniformLocation(key), a);
		detach();
	}
	
	void setInt(char[] key, int a) ///
	{
		attach();
		glUniform1iARB(getUniformLocation(key), a);
		detach();
	}
	
	void setVector(char[] key, float x, float y, float z, float w) ///
	{
		attach();
		glUniform4fARB(getUniformLocation(key), x, y, z, w);
		detach();
	}
	
	void setVector(char[] key, Vector a) ///
	{
		setVector(key, a.x, a.y, a.z, a.w);
	}
	
	void setMatrixArray(char[] key, float* p, uint matrixLength) ///  Matrix[]
	{
		attach();
		glUniformMatrix4fvARB(getUniformLocation(key), matrixLength, GL_FALSE, p);
		detach();
	}
	
	void setMatrixArray(char[] key, Matrix[] a) ///
	{
		setMatrixArray(key, cast(float*)a.ptr, a.length);
	}
	
	uint getAttributeLocation(char[] key) ///
	{
		attach();
		int result = glGetAttribLocationARB(handle, toStringz(key));
		detach();
		if (-1 == result) exception(__FILE__, __LINE__, this, "getAttributeLocation: " ~ key);
		return result;
	}
}

///
class Fbo : Texture, RenderTarget
{
	private final uint fbHandle;
	private final Renderbuffer depthBuffer;
	
	///
	this(uint width, uint height, uint index, bool enableDepthBuffer)
	{
		super(width, height, null, index);
		
		if (!glGenFramebuffersEXT)
		{
			load_GL_EXT_framebuffer_object();
			if (!glGenFramebuffersEXT) exception(__FILE__, __LINE__, this, "load_GL_EXT_framebuffer_object failed");
		}
		
		if (glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT) == GL_FRAMEBUFFER_UNSUPPORTED_EXT)
		{
			exception(__FILE__, __LINE__, this, "GL_FRAMEBUFFER_UNSUPPORTED_EXT");
		}
		else if (glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT) != GL_FRAMEBUFFER_COMPLETE_EXT)
		{
			exception(__FILE__, __LINE__, this, "fbo");
		}
		
		glGenFramebuffersEXT(1, &fbHandle);
		
		if (enableDepthBuffer) depthBuffer = new Renderbuffer(this, width, height);
	}
	
	~this()
	{
		delete depthBuffer;
		glDeleteFramebuffersEXT(1, &fbHandle);
	}
	
	void draw(Node a)
	{
		attachFramebuffer();
		a();
		detachFramebuffer();
	}
	
	///
	protected void attachFramebuffer(ubyte attachmentIndex = 0)
	{
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbHandle);
		glFramebufferTexture2DEXT(
			GL_FRAMEBUFFER_EXT,
			GL_COLOR_ATTACHMENT0_EXT + attachmentIndex,
			GL_TEXTURE_2D,
			handle,
			0
		);
	}
	
	///
	protected void detachFramebuffer()
	{
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	}
	
	void flip() {}
}

/// fboのdepth
class Renderbuffer
{
	private uint depthHandle;
	
	///
	this(Fbo owner, uint width, uint height)
	{
		glGenRenderbuffersEXT(1, &depthHandle);
		glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, depthHandle);
		glRenderbufferStorageEXT(
			GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT24_ARB, width, height
		);
		owner.attachFramebuffer();
		glFramebufferRenderbufferEXT(
			GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, depthHandle
		);
		owner.detachFramebuffer();
		glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, 0);
		glErrorCheck(__FILE__, __LINE__, this);
	}
	
	~this()
	{
		glDeleteRenderbuffersEXT(1, &depthHandle);
	}
}

///
class FloatFbo : Fbo
{
	///
	this(uint width, uint height, uint index, bool enableDepthBuffer)
	{
		super(width, height, index, enableDepthBuffer);
	}
	
	override protected int internalformat() /// GL_RGBA32F_ARB
	{
		return GL_RGBA32F_ARB;
	}
	
	override protected uint dataType() /// GL_FLOAT
	{
		return GL_FLOAT;
	}
	
	///
	float[] floatPixels()
	{
		float[] result = new float[width * height * 4];
		attach();
		glGetTexImage(GL_TEXTURE_2D, 0, GL_RGBA, dataType, result.ptr);
		detach();
		return result;
	}
	
	/// 4倍のサイズになる
	override uint[] pixels()
	{
		return cast(uint[])floatPixels;
	}
}

///
class Disable : Unit
{
	///
	uint cap;
	
	private ubyte stock;
	
	///
	this(uint cap)
	{
		this.cap = cap;
	}
	
	void attach()
	{
		stock = glIsEnabled(cap);
		glDisable(cap);
	}
	
	void detach()
	{
		stock == GL_TRUE ? glEnable(cap) : glDisable(cap);
		glErrorCheck(__FILE__, __LINE__, this);
	}
}

///  左上(0, 0)、右下(virtualWidth, virtualHeight)、windowサイズに依存しない
interface VirtualSizeUnit : Unit
{
	static uint virtualWidth = 800, virtualHeight = 600;
}

///
class Quadrangle : VirtualSizeUnit
{
	///
	int x, y, width, height;
	
	Vector color; ///
	
	///
	float tcX = 0.0, tcY = 0.0, tcWidth = 1.0, tcHeight = 1.0;
	
	///
	this(int x, int y, int width, int height)
	{
		this(x, y, width, height, vector(1.0, 1.0, 1.0, 1.0));
	}
	
	///
	this(int x, int y, int width, int height, Vector color)
	{
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
		this.color = color;
	}
	
	void attach()
	{
		glPushMatrix();
		glTranslatef(-1.0, 1.0, 0.0);
		glScalef(2.0 / virtualWidth, -2.0 / virtualHeight, 1.0);
		glBegin(GL_QUADS);
		glColor4f(color.x, color.y, color.z, color.w);
		glNormal3f(0.0, 0.0, 0.0);
		
		glTexCoord2f(tcX, tcY);
		glVertex2f(x, y);
		
		glTexCoord2f(tcX + tcWidth, tcY);
		glVertex2f(x + width, y);
		
		glTexCoord2f(tcX + tcWidth, tcY + tcHeight);
		glVertex2f(x + width, y + height);
		
		glTexCoord2f(tcX, tcY + tcHeight);
		glVertex2f(x, y + height);
		
		glEnd();
		glPopMatrix();
	}
	
	void detach() {}
}

///
class AlphaBlend : UnitMerger
{
	///
	this()
	{
		super(new Enable(GL_BLEND), new BlendFunc());
	}
}

///
class WindowRgba : Rgba
{
	private uint _width, _height;
	private uint[] _pixels;
	
	///
	this(Window wnd)
	{
		this._width = wnd.width;
		this._height = wnd.height;
		
		uint[] bbp = readBackBufferPixels(_width, _height);
		_pixels = createAndTranslateFromBmpPixels(bbp, _width, _height);
		delete bbp;
		
		setAlpha(_pixels);
	}
	
	private static uint[] readBackBufferPixels(uint width, uint height)
	{
		uint[] result = new uint[width * height];
		glReadBuffer(GL_BACK);
		glReadPixels(0, 0, width, height, GL_RGBA, GL_UNSIGNED_BYTE, result.ptr);
		return result;
	}
	
	private static uint[] createAndTranslateFromBmpPixels(uint[] src, uint width, uint height)
	{
		uint[] result = new uint[width * height];
		for (int y = 0; y < height; y++)
		{
			int y2 = (height - 1) - y;
			int begin = width * y;
			int end = begin + width;
			int begin2 = width * y2;
			int end2 = begin2 + width;
			result[begin..end] = src[begin2..end2].dup;
		}
		return result;
	}
	
	private static void setAlpha(inout uint[] src)
	{
		const uint ALPHA_FILTER = 0xff000000;
		foreach (inout uint a; src) { a |= ALPHA_FILTER; }
	}
	
	uint width()
	{
		return _width;
	}
	
	uint height()
	{
		return _height;
	}
	
	uint[] pixels()
	{
		return _pixels;
	}
}

///
class ProgressBar : UnitMerger
{
	private Quadrangle fore;
	private float _value = 0.0;
	private uint max;
	
	///
	this(uint x, uint y, uint width, uint height, Vector foreColor, Vector backColor)
	{
		fore = new Quadrangle(x, y, 0, height, foreColor);
		max = width;
		super(
			new Quadrangle(x, y, width, height, backColor),
			fore
		);
	}
	
	void value(float a) /// 0.0 <= value <= 1.0
	{
		_value = a;
		fore.width = cast(uint)(max * a);
	}
	
	float value() /// 0.0 <= value <= 1.0
	{
		return _value;
	}
}

///
class VirtualMousePoint : MousePoint, VirtualSizeUnit
{
	private final Window wnd;
	
	///
	this(Window wnd)
	{
		this.wnd = wnd;
	}
	
	int x()
	{
		return wnd.mousePoint.x * virtualWidth / wnd.width;
	}
	
	int y()
	{
		return wnd.mousePoint.y * virtualHeight / wnd.height;
	}
	
	void attach() {}
	void detach() {}
	
	bool isInner(int rectX, int rectY, int rectWidth, int rectHeight) ///
	{
		return
			rectX <= x && x <= rectX + rectWidth &&
			rectY <= y && y <= rectY + rectHeight;
	}
}

///
class LineQuadrangle : VirtualSizeUnit
{
	///
	int x, y, width, height;
	
	Vector color; ///
	
	///
	this(int x, int y, int width, int height)
	{
		this(x, y, width, height, vector(1.0, 1.0, 1.0, 1.0));
	}
	
	///
	this(int x, int y, int width, int height, Vector color)
	{
		this.x = x;
		this.y = y;
		this.width = width;
		this.height = height;
		this.color = color;
	}
	
	void attach()
	{
		glPushMatrix();
		glTranslatef(-1.0, 1.0, 0.0);
		glScalef(2.0 / virtualWidth, -2.0 / virtualHeight, 1.0);
		glBegin(GL_LINES);
		glColor4f(color.x, color.y, color.z, color.w);
		glNormal3f(0.0, 0.0, 0.0);
		glTexCoord2f(0.0, 0.0);
		
		glVertex2f(x, y);
		glVertex2f(x + width, y);
		glVertex2f(x + width, y);
		glVertex2f(x + width, y + height);
		glVertex2f(x + width, y + height);
		glVertex2f(x, y + height);
		glVertex2f(x, y + height);
		glVertex2f(x, y);
		
		glEnd();
		glPopMatrix();
	}
	
	void detach() {}
}

///
class BlendQuadrangle : Quadrangle
{
	private Unit state;
	
	///
	this(int x, int y, int width, int height, Vector color = vector(1.0, 1.0, 1.0, 1.0))
	{
		super(x, y, width, height, color);
		state = new UnitMerger(new Enable(GL_BLEND), new BlendFunc());
	}
	
	override void attach()
	{
		state.attach();
		super.attach();
		super.detach();
		state.detach();
	}
}
