﻿import std.string;
import std.cstream;

int main(char[][] args)
{
	try
	{
		if (args.length == 2 && args[1] == "-header")
		{
			printHeader();
			return 0;
		}
		
		char[] caption;
		char[][] functionList;
		while (true)
		{
			char[] line = din.readLine();
			if (!line) break;
			switch (typeOf(line))
			{
				case Type.CAPTION:
					dout.writeLine(toComment(line));
					if (caption) throw new ErrorLine(__LINE__);
					caption = line;
					break;
					
				case Type.URL:
					dout.writeLine(toComment(line));
					break;
					
				case Type.ENUM:
					dout.writeLine(toGlenum(line));
					break;
					
				case Type.FUNCTION:
					dout.writeLine(toFunction(line));
					functionList ~= getFunctionName(line);
					break;
					
				case Type.TYPEDEF:
					dout.writeLine(toAlias(line));
					break;
			}
		}
		if (!caption) throw new ErrorLine(__LINE__);
		printLoadFunction(caption, functionList);
	}
	catch (Exception e)
	{
		e.print();
	}
	return 0;
}

class ErrorLine : Error
{
	this(long line)
	{
		super(format("%s(%d): ", __FILE__, line));
	}
}

void printLoadFunction(char[] caption, char[][] functionList)
{
	dout.writeLine("");
	dout.writefln ("void load_%s() /// loader", caption);
	dout.writeLine("{");
	foreach (char[] name; functionList)
	{
		dout.writefln(`	loadGlextFunctions(cast(void**)&%s, "%s");`, name, name);
	}
	dout.writeLine("}");
	dout.writeLine("");
}

void printHeader()
{
	dout.writeLine("module coneneko.glext;");
	dout.writeLine("import std.string;");
	dout.writeLine("version (DWT_WINDOW)");
	dout.writeLine("{");
	dout.writeLine("	public import dwt.opengl.all;");
	dout.writeLine("}");
	dout.writeLine("else");
	dout.writeLine("{");
	dout.writeLine("	public import opengl, openglu;");
	dout.writeLine("}");
	dout.writeLine("extern(Windows):");
	dout.writeLine("void* wglGetProcAddress(char* proc);");
	dout.writeLine("private void loadGlextFunctions(void** f, char[] name)");
	dout.writeLine("{");
	dout.writeLine("	*f = wglGetProcAddress(toStringz(name));");
	dout.writeLine("}");
	dout.writeLine("");
}

unittest
{
	assert(Type.CAPTION == typeOf("GL_ARB_vertex_buffer_object"));
	assert(Type.URL == typeOf("http://oss.sgi.com/projects/ogl-sample/registry/ARB/vertex_buffer_object.txt"));
	assert(Type.ENUM == typeOf("GL_BUFFER_SIZE_ARB 0x8764"));
	assert(Type.FUNCTION == typeOf("void glBindBufferARB (GLenum target, GLuint buffer)"));
	assert(Type.TYPEDEF == typeOf("typedef ptrdiff_t GLsizeiptrARB"));
}

enum Type
{
	CAPTION, URL, ENUM, FUNCTION, TYPEDEF
}

Type typeOf(char[] line)
{
	char[][] sl = split(line);
	if (line[0..4] == "http") return Type.URL;
	else if (sl.length == 2 && sl[1][0..2] == "0x") return Type.ENUM;
	else if (line.length != 0 && line[$ - 1] == ')') return Type.FUNCTION;
	else if (sl.length > 2 && sl[0] == "typedef") return Type.TYPEDEF;
	else return Type.CAPTION;
}

unittest
{
	assert("//hoge" == toComment("hoge"));
}

char[] toComment(char[] a)
{
	return "//" ~ a;
}

unittest
{
	assert(
		"const GLenum GL_BUFFER_SIZE_ARB = 0x8764;"
		== toGlenum("GL_BUFFER_SIZE_ARB 0x8764")
	);
}

char[] toGlenum(char[] a)
{
	char[][] sl = split(a);
	return format("const GLenum %s = %s;", sl[0], sl[1]);
}

unittest
{
	assert(
		"void function (GLenum target, GLuint buffer) glBindBufferARB;"
		== toFunction("void glBindBufferARB (GLenum target, GLuint buffer)")
	);
	assert(
		"GLvoid * function (GLenum target, GLenum access) glMapBufferARB;"
		== toFunction("GLvoid * glMapBufferARB (GLenum target, GLenum access)")
	);
	assert(
		"GLhandleARB function () glCreateProgramObjectARB;"
		== toFunction("GLhandleARB glCreateProgramObjectARB (void)")
	);
}

char[] toFunction(char[] a)
{
	const char[] delim = [ '(' ];
	char[][] b = split(a, delim);
	assert(2 == b.length);
	char[][] c = split(b[0]);
	char[] result;
	for (int i = 0; i < c.length - 1; i++)
	{
		result ~= c[i];
		result ~= " ";
	}
	result ~= "function (";
	result ~= b[1][0..5] == "void)" ? ")" : b[1];
	result ~= " ";
	result ~= c[$ - 1];
	result ~= ";";
	return replace(result, "const", "");
}

unittest
{
	assert(
		"alias ptrdiff_t GLsizeiptrARB;"
		== toAlias("typedef ptrdiff_t GLsizeiptrARB")
	);
	assert(
		"alias uint GLhandleARB;"
		== toAlias("typedef unsigned int GLhandleARB")
	);
}

char[] toAlias(char[] a)
{
	return removechars(
		replace(
			replace(a, "typedef", "alias"),
			"unsigned int",
			"uint"
		),
		"\t"
	) ~ ";";
}

unittest
{
	assert(
		"glBindBufferARB"
		== getFunctionName("void glBindBufferARB (GLenum target, GLuint buffer)")
	);
	assert(
		"glMapBufferARB"
		== getFunctionName("GLvoid * glMapBufferARB (GLenum target, GLenum access)")
	);
}

char[] getFunctionName(char[] a)
{
	const char[] delim = [ '(' ];
	char[][] b = split(a, delim);
	assert(2 == b.length);
	char[][] c = split(b[0]);
	return c[$ - 1];
}
