﻿module editview;
import coneneko.dwtwindow;
import std.string, std.file, std.utf, std.path;

private alias dwt.widgets.text.Text Text;

interface Observer
{
	void update(char[] src, int position); // positionは日本語3文字でカウントしているので注意
}

class EditView : Shell, DisposeListener, ShellListener
{
	private const int INIT_WIDTH = 310, INIT_HEIGHT = 560;
	private Observer observer;
	private Text text;
	private bool textModified;
	private char[] currentFileName;
	private long currentFileLastWriteTime;
	
	this(Observer observer)
	{
		super(display);
		this.observer = observer;
		setLocation(0, 0);
		setSize(INIT_WIDTH, INIT_HEIGHT);
		setLayout(new FillLayout());
		initializeMenu();
		
		text = new Text(this, DWT.MULTI | DWT.H_SCROLL | DWT.V_SCROLL | DWT.BORDER);
		text.addModifyListener(new ModifyText(&modifyText));
		text.setForeground(display.getSystemColor(DWT.COLOR_WHITE));
		text.setBackground(display.getSystemColor(DWT.COLOR_BLACK));
		text.setFont(
			new dwt.graphics.font.Font(display, new FontData("ＭＳ ゴシック", 10, DWT.NORMAL))
		);
		text.addDisposeListener(this);
		newFile();
		
		text.addKeyListener(new KeyAccelerator('o', DWT.CTRL, &openFile));
		text.addKeyListener(new KeyAccelerator('s', DWT.CTRL, &saveFile));
		text.addKeyListener(new KeyAccelerator(DWT.F5, DWT.NULL, &update));
		
		addShellListener(this);
		open();
	}
	
	void widgetDisposed(DisposeEvent e)
	{
		preClose();
	}
	
	private void preClose() // 未保存なら確認して保存
	{
		if (!textModified) return;
		const char[] MSG = "未保存です、上書き保存しますか?";
		const int STYLE = DWT.ICON_QUESTION | DWT.YES | DWT.NO;
		if (DWT.YES == MessageBox.showMsg(MSG, null, null, STYLE)) saveFile();
	}
	
	void shellActivated(ShellEvent e)
	{
		if (currentFileName == "") return;
		if (!isfile2(currentFileName))
		{
			textModified = true;
			updateCaption();
			return;
		}
		if (getLastWriteTime(currentFileName) <= currentFileLastWriteTime) return;
		
		const int STYLE = DWT.ICON_INFORMATION | DWT.YES;
		MessageBox.showMsg("コードが修正されました、読み直します。", null, null, STYLE);
		openFile(currentFileName);
	}
	
	private static long getLastWriteTime(char[] fileName)
	{
		long result = -1;
		listdir(
			fileName.getDirName(),
			delegate bool(DirEntry* de)
			{
				if (de.name != fileName) return true;
				result = de.lastWriteTime;
				return false;
			}
		);
		return result;
	}
	
	void shellClosed(ShellEvent e) {}
	void shellDeactivated(ShellEvent e) {}
	void shellDeiconified(ShellEvent e) {}
	void shellIconified(ShellEvent e) {}
	
	private void initializeMenu()
	{
		Menu root = new Menu(this, DWT.BAR);
		setMenuBar(root);
		
		
		MenuItem filei = new MenuItem(root, DWT.CASCADE);
		Menu file = new Menu(filei);
		filei.setText("ファイル(&F)");
		filei.setMenu(file);
		
		addPushMenuItem(file, "新規作成(&N)", &newFile);
		addPushMenuItem(file, "開く(&O)...\tCtrl+O", &openFile);
		addPushMenuItem(file, "上書き保存(&S)\tCtrl+S", &saveFile);
		addPushMenuItem(file, "名前を付けて保存(&A)...", &saveAsFile);
		new MenuItem(file, DWT.SEPARATOR);
		addPushMenuItem(file, "終了(&X)", &close);
		
		
		MenuItem editi = new MenuItem(root, DWT.CASCADE);
		Menu edit = new Menu(editi);
		editi.setText("編集(&E)");
		editi.setMenu(edit);
		
		addPushMenuItem(edit, "更新(&U)\tF5", &update);
	}
	
	private void addPushMenuItem(Menu parent, char[] text, void delegate() dg)
	{
		MenuItem mi = new MenuItem(parent, DWT.PUSH);
		mi.setText(text);
		mi.addSelectionListener(new SelectionListenerFromDelegate(dg));
	}
	
	void modifyText()
	{
		textModified = true;
		updateCaption();
	}
	
	private void updateCaption()
	{
		setText(
			format("Scene Editor - %s%s", currentFileName, textModified ? "*" : "")
		);
	}
	
	void newFile()
	{
		preClose();
		const char[] INIT_TEXT =
"// `c:\\foo`
template Scene() { void doScene() {

}}
";
		text.setText(INIT_TEXT);
		currentFileName = "";
		textModified = false;
		updateCaption();
	}
	
	void openFile()
	{
		preClose();
		try
		{
			FileDialogForD dlg = new FileDialogForD(this, DWT.OPEN);
			char[] fileName = dlg.open();
			if (!fileName) return;
			openFile(fileName);
		}
		catch (Exception e)
		{
			MessageBox.showMsg(e.toString());
		}
	}
	
	void openFile(char[] fileName)
	{
		if (!isfile2(fileName))
		{
			MessageBox.showMsg(format("%s", fileName));
			return;
		}
		if (!isabs(fileName))
		{
			fileName = std.path.join(getcwd(), fileName);
		}
		text.setText(BomUtf8File.read(fileName));
		currentFileName = fileName;
		textModified = false;
		updateCaption();
		currentFileLastWriteTime = getLastWriteTime(currentFileName);
	}
	
	void saveFile()
	{
		if (isfile2(currentFileName))
		{
			BomUtf8File.write(currentFileName, text.getText());
			textModified = false;
			updateCaption();
			currentFileLastWriteTime = getLastWriteTime(currentFileName);
		}
		else saveAsFile();
	}
	
	private static bool isfile2(char[] fileName)
	{
		try
		{
			return isfile(fileName) ? true : false;
		}
		catch (FileException e)
		{
			return false;
		}
	}
	
	void saveAsFile()
	{
		FileDialogForD dlg = new FileDialogForD(this, DWT.SAVE);
		char[] fileName = dlg.open();
		if (!fileName) return;
		if (isfile2(fileName))
		{
			const char[] MSG = "上書き保存しますか?";
			const int STYLE = DWT.ICON_QUESTION | DWT.OK | DWT.CANCEL;
			if (DWT.CANCEL == MessageBox.showMsg(MSG, null, null, STYLE)) return;
		}
		BomUtf8File.write(fileName, text.getText());
		currentFileName = fileName;
		textModified = false;
		updateCaption();
		currentFileLastWriteTime = getLastWriteTime(currentFileName);
	}
	
	void update()
	{
		wchar[] t = toUTF16(text.getText());
		char[] t2 = toUTF8(t[0..text.getCaretPosition()]);
		observer.update(text.getText(), t2.length);
		//observer.update(text.getText(), text.getCaretPosition());
	}
}

class SelectionListenerFromDelegate : SelectionAdapter
{
	private void delegate() dg;
	
	this(void delegate() dg)
	{
		this.dg = dg;
	}
	
	override void widgetSelected(SelectionEvent e)
	{
		dg();
	}
}

class KeyAccelerator : KeyAdapter
{
	private int keyCode;
	private int stateMask;
	private void delegate() dg;
	
	this(int keyCode, int stateMask, void delegate() dg) // keyCodeは小文字
	{
		this.keyCode = keyCode;
		this.stateMask = stateMask;
		this.dg = dg;
	}
	
	override void keyPressed(KeyEvent e)
	{
		//dout.writefln("%d %d %d", e.character, e.keyCode, e.stateMask), dout.flush();
		if (keyCode == e.keyCode && stateMask == e.stateMask)
		{
			dg();
			e.doit = false;
		}
	}
}

class ModifyText : ModifyListener
{
	private void delegate() dg;
	
	this(void delegate() dg)
	{
		this.dg = dg;
	}
	
	void modifyText(ModifyEvent e)
	{
		dg();
	}
}

class FileDialogForD : FileDialog
{
	private const char[][]
		FILTER_EXTENSIONS = [ "*.d" ],
		FILTER_NAMES = [ "D (*.d)" ];
	
	this(Shell parent, int style)
	{
		super(parent, style);
		setFilterExtensions(FILTER_EXTENSIONS);
		setFilterNames(FILTER_NAMES);
	}
}

class BomUtf8File
{
	const char[] BOM = [ 0xEF, 0xBB, 0xBF ];
	
	static char[] read(char[] fileName)
	{
		char[] result = cast(char[])std.file.read(fileName);
		if (result.length <= 3 || result[0..3] != BOM) throw new Exception("bom付きutf8ではない");
		return result[3..$];
	}
	
	static void write(char[] fileName, void[] buffer)
	{
		std.file.write(fileName, BOM ~ buffer);
	}
}
