﻿import editview;
import coneneko.dwtwindow, coneneko.sceneview;
import std.date, std.regexp, std.cstream, std.conv, std.stream;

alias coneneko.math.Vector Vector;

int main(char[][] args)
{
	try
	{
		auto SceneViewObserver svo = new SceneViewObserver();
		EditView ev = new EditView(svo);
		if (args.length == 2) ev.openFile(args[1]);
		
		Waiter waiter = new Waiter(svo);
		waiter.addEvent(1, new DwtClickEvent(0, 0, 640, 480));
		waiter.killTime = delegate uint(RenderTarget rt)
		{
			while (true)
			{
				try
				{
					rt.draw(svo.rootNode);
				}
				catch (Exception e)
				{
					char[] s = e.toString();
					for (int i = 50; i < s.length; i += 50) s = std.string.insert(s, i, "\n");
					svo.trace = s;
				}
				rt.flip();
				if (ev.isDisposed) throw new WindowDeadException();
			}
			return 0;
		};
		while (true)
		{
			uint id = waiter.wait();
			if (id == 1) svo.timeStamp = getUTCtime();
		}
	}
	catch (Exception e)
	{
		e.print();
	}
	return 0;
}

class SceneViewObserver : SceneView, Observer
{
	final DwtWindow shell;
	
	this()
	{
		shell = new DwtWindow();
		shell.setText("Scene Editor View");
		shell.setLocation(310, 0);
		shell.setSize(480, 360);
		super(shell);
	}
	
	void update(char[] src, int position)
	{
		initialize();
		try
		{
			CodeReader reader = new CodeReader(src, position);
			baseDir = reader.readBaseDir();
			reader.seekStartPosition();
			while (true)
			{
				if (reader.eof) break;
				reader.readNext();
				if (reader.left == "wc()" && reader.endOfCode) break;
				setValue(reader.left, reader.right);
			}
		}
		catch (Object e)
		{
			MessageBox.showMsg(e.toString());
		}
		timeStamp = getUTCtime();
	}
	
	private void setValue(char[] left, char[] right) // left = right;、追加はここで
	{
		uint[] indices;
		RegExp re = new RegExp(`\[[0-9]+\]`, "g");
		foreach (char[] a; re.match(left)) indices ~= toUint(a[1..$ - 1]);
		uint i0 = indices.length >= 1 ? indices[0] : 0;
		uint i1 = indices.length >= 2 ? indices[1] : 0;
		
		if (left == "resourceDir") resourceDir = toString(right);
		if (left == "te") te = toTexts(right);
		if (left == "bg") bg = toString(right);
		if (left == "cameraPoint") cameraPoint = toVector(right);
		if (left == "cameraAt") cameraAt = toVector(right);
		if (left == "lightDirection") lightDirection = toVector(right);
		if (left == "bgm") bgm = toString(right);
		
		// ModelInfo
		if (left == "ms.length") ms.length = toUint(right);
		const char[] MI = `^ms\[[0-9]+\]\.`;
		if (left.find(MI ~ `enable`) != -1) ms[i0].enable = toBool(right);
		if (left.find(MI ~ `file`) != -1) ms[i0].file = toString(right);
		if (left.find(MI ~ `point`) != -1) ms[i0].point = toVector(right);
		if (left.find(MI ~ `angle`) != -1) ms[i0].angle = toFloat(right);
		if (left.find(MI ~ `ambient`) != -1) ms[i0].ambient = toVector(right);
		if (left.find(MI ~ `fmi`) != -1) ms[i0].fmi = toByte(right);
		if (left.find(MI ~ `smi`) != -1) ms[i0].smi = toByte(right);
		
		if (left.find(MI ~ `vis.length`) != -1) ms[i0].vis.length = toUint(right);
		if (left.find(MI ~ `vis\[[0-9]+\]`) != -1) ms[i0].vis[i1] = toBool(right);
		
		// MorphInfo
		if (left.find(MI ~ `mps.length`) != -1) ms[i0].mps.length = toUint(right);
		const char[] MPI = MI ~ `mps\[[0-9]+\]\.`;
		if (left.find(MPI ~ `enable`) != -1) ms[i0].mps[i1].enable = toBool(right);
		if (left.find(MPI ~ `frameRate`) != -1) ms[i0].mps[i1].frameRate = toUbyte(right);
		if (left.find(MPI ~ `frequency`) != -1) ms[i0].mps[i1].frequency = toUbyte(right);
		if (left.find(MPI ~ `syncTe`) != -1) ms[i0].mps[i1].syncTe = toBool(right);
		
		// EffectInfo
		if (left == "es.length") es.length = toUint(right);
		const char[] EI = `^es\[[0-9]+\]\.`;
		if (left.find(EI ~ `enable`) != -1) es[i0].enable = toBool(right);
		if (left.find(EI ~ `file`) != -1) es[i0].file = toString(right);
		if (left.find(EI ~ `x`) != -1) es[i0].x = toShort(right);
		if (left.find(EI ~ `y`) != -1) es[i0].y = toShort(right);
		if (left.find(EI ~ `width`) != -1) es[i0].width = toShort(right);
		if (left.find(EI ~ `height`) != -1) es[i0].height = toShort(right);
		if (left.find(EI ~ `t`) != -1) es[i0].t = toUbyte(right);
		if (left.find(EI ~ `frameRate`) != -1) es[i0].frameRate = toUbyte(right);
		if (left.find(EI ~ `loop`) != -1) es[i0].loop = toBool(right);
		
		// SoundEffectInfo
		if (left == "ses.length") ses.length = toUint(right);
		const char[] SEI = `^ses\[[0-9]+\]\.`;
		if (left.find(SEI ~ `enable`) != -1) ses[i0].enable = toBool(right);
		if (left.find(SEI ~ `file`) != -1) ses[i0].file = toString(right);
		if (left.find(SEI ~ `t`) != -1) ses[i0].t = toUbyte(right);
		if (left.find(SEI ~ `loop`) != -1) ses[i0].loop = toBool(right);
	}
	
	unittest
	{
		assert("a\nb" == toTexts(`"a\nb"`));
		assert("a\\nb" == toTexts(`"a\\nb"`));
	}
	
	private static char[] toTexts(char[] right)
	{
		char[] result = toString(right);
		result = result.sub(`\\n`, "\n", "g");
		result = result.sub(`\\\n`, `\n`, "g");
		return result;
	}
	
	private static char[] toString(char[] right)
	{
		return right[1..$ - 1];
	}
	
	private static bool toBool(char[] right)
	{
		return right == "true";
	}
	
	unittest
	{
		const char[] TEST = "vector(0, 1, 2)";
		char[][] a = TEST[7..$ - 1].split(", *");
		assert("0" == a[0]);
		assert("1" == a[1]);
		assert("2" == a[2]);
	}
	
	private static Vector toVector(char[] right)
	{
		char[][] xyz = right[7..$ - 1].split(", *");
		return vector(xyz[0].toFloat(), xyz[1].toFloat(), xyz[2].toFloat());
	}
}

class CodeReader : MemoryStream
{
	unittest
	{
		if (RegExp m = search("// `c:\\foo`", "`.+`")) assert("`c:\\foo`" == m.match(0));
	}
	
	unittest
	{
		MemoryStream src = new MemoryStream();
		src.writeLine("// `c:\\foo`");
		src.writeLine("template Scene() { void doScene() {");
		src.writeLine(`te = "abc def\nあ";`);
		src.writeLine("//name = `aaa`;");
		src.writeLine("wc();");
		src.writeLine("}}");
		
		CodeReader reader = new CodeReader(src.toString(), src.size - 6);
		assert(`c:\foo` == reader.readBaseDir());
		reader.seekStartPosition();
		
		assert(!reader.endOfCode);
		reader.readNext();
		assert("te" == reader.left);
		assert(`"abc def\nあ"` == reader.right);
		reader.readNext();
		assert("wc()" == reader.left);
		assert(reader.endOfCode);
	}
	
	private int cursorPosition;
	char[] left, right;
	
	this(char[] src, int cursorPosition)
	{
		super(src);
		this.cursorPosition = cursorPosition;
	}
	
	char[] readBaseDir()
	{
		while (true)
		{
			if (eof) throw new Exception("baseDirを読めない");
			char[] line = readLine();
			if (line.length < 2 || line[0..2] != "//") continue;
			if (RegExp m = search(line, "`.+`")) return m.match(0)[1..$ - 1];
		}
	}
	
	void seekStartPosition()
	{
		int count = 0;
		while (true)
		{
			if (eof) throw new Exception("startPositionに移動できない");
			char a;
			read(a);
			if (a == '{') count++;
			if (count == 2) return;
		}
	}
	
	bool endOfCode()
	{
		return position >= cursorPosition;
	}
	
	void readNext()
	{
		left = "";
		right = "";
		
		// left
		while (true)
		{
			if (eof) return;
			char a;
			read(a);
			if (a == ';') return;
			if (a == '=') break;
			if (!iswhite(a)) left ~= a;
			if (left.length == 2 && left == "//")
			{
				seekNextLine();
				left = "";
			}
		}
		
		// right
		bool whiteSkip = true;
		while (true)
		{
			if (eof) return;
			char a;
			read(a);
			if (a == ';') return;
			if (a == '"' || a == '`') whiteSkip = false;
			if (whiteSkip && iswhite(a)) continue;
			right ~= a;
		}
	}
	
	private void seekNextLine()
	{
		while (true)
		{
			if (eof) return;
			char a;
			read(a);
			if (a == '\n') return;
		}
	}
}
