// collections.dRpCKv
import hdk.d3dframe;
import hdk.joystick;
import hdk.string;
import std.random;
import hdk.math;
import hdk.collections;

enum
{
	WIDTH_OF_DRAW_PUYO = 6,
	HEIGHT_OF_DRAW_PUYO = 12,
}

enum PuyoColor
{
	RED, GREEN, BLUE, YELLOW, PURPLE
}

class PuyoView : D3dFrame
{
	public static void test()
	{
		PuyoView view = new PuyoView();
		view.fps = 10;
		for (int x = 0; x < WIDTH_OF_DRAW_PUYO; x++)
		{
			for (int y = 0; y < HEIGHT_OF_DRAW_PUYO; y++)
			{
				view.drawBackground();
				view.drawPuyo(x, y, PuyoColor.RED);
				view.drawPuyo(0, 0, PuyoColor.GREEN);
				view.drawPuyo(WIDTH_OF_DRAW_PUYO - 1, HEIGHT_OF_DRAW_PUYO - 1, PuyoColor.BLUE);
				view.drawNextPuyo(PuyoColor.YELLOW, PuyoColor.PURPLE);
				if (view.eof) throw new Error("f");
				view.update();
				fullCollect();
			}
		}
		view.drawGameOver();
		view.update();
		Sleep(1000);
	}
	
	public void drawBackground()
	{
		drawRectangle(0, 0, width, height, Color.DARK_GREEN);
		drawRectangle(margin, 0, fieldWidth, height, Color.BLACK);
	}
	
	public void drawPuyo(int x, int y, PuyoColor color)
	in
	{
		assert(0 <= x && x < WIDTH_OF_DRAW_PUYO);
		assert(-1 <= y && y < HEIGHT_OF_DRAW_PUYO);
	}
	body
	{
		drawRectangle(margin + blockWidth * x, blockWidth * y, blockWidth, blockWidth, toColor(color));
	}
	
	public void drawNextPuyo(PuyoColor first, PuyoColor second)
	{
		drawRectangle(margin + fieldWidth + margin_4, height_2, blockWidth, blockWidth, toColor(first));
		drawRectangle(margin + fieldWidth + margin_4, height_2 + blockWidth, blockWidth, blockWidth, toColor(second));
	}
	
	public void drawGameOver()
	{
		drawString("game over", width * 0.5, height * 0.5, Color.WHITE);
	}
	
	private Color toColor(PuyoColor a)
	{
		switch(a)
		{
			case a.RED: return Color.RED;
			case a.GREEN: return Color.GREEN;
			case a.BLUE: return Color.BLUE;
			case a.YELLOW: return Color.create(127, 127, 0);
			case a.PURPLE: return Color.create(127, 0, 127);
		}
	}
	
	private int margin() { return width * 0.5 - fieldWidth * 0.5; }
	private int fieldWidth() { return blockWidth * WIDTH_OF_DRAW_PUYO; }
	private int blockWidth() { return height / HEIGHT_OF_DRAW_PUYO; }
	private int margin_4() { return margin * 0.25; }
	private int height_2() { return height * 0.5; }
}

class Puyo
{
	unittest
	{
		printf("Puyo test ");
		Puyo a = new Puyo(0, 0, PuyoColor.RED);
		a.save();
		assert(!a.isOutside);
		a.moveLeft();
		assert(-1 == a.x && 0 == a.y);
		assert(a.isOutside);
		a.moveRight();
		assert(!a.isOutside);
		for (int i = 0; i < 15; i++) a.moveDown();
		assert(a.isOutside);
		a.load();
		assert(0 == a.x && 0 == a.y);
		printf("ok\n");
	}

	public this(int x, int y, PuyoColor color)
	in
	{
		assert(0 <= x && x < WIDTH_OF_DRAW_PUYO);
		assert(0 <= y && y < HEIGHT_OF_DRAW_PUYO);
	}
	body
	{
		this.x = x;
		this.y = y;
		this.color = color;
	}
	
	public void moveLeft() { x--; }	
	public void moveRight() { x++; }
	public void moveDown() { y++; }

	public bit isOutside()
	{
		return x < 0 || WIDTH_OF_DRAW_PUYO <= x || y < -1 || HEIGHT_OF_DRAW_PUYO <= y;
	}
	
	public void save()
	{
		stockX = x;
		stockY = y;
	}
	
	public void load()
	{
		x = stockX;
		y = stockY;
	}

	public int x, y, stockX, stockY;
	public PuyoColor color;
}

class PuyoPuyo
{
	public this()
	{
		first = new Puyo(2, 0, getRandomPuyoColor());
		second = new Puyo(3, 0, getRandomPuyoColor());
	}
	
	private PuyoColor getRandomPuyoColor()
	{
		real r = rrand() * 5;
		if (r < 1) return PuyoColor.RED;
		else if (r < 2) return PuyoColor.GREEN;
		else if (r < 3) return PuyoColor.BLUE;
		else if (r < 4) return PuyoColor.YELLOW;
		else return PuyoColor.PURPLE;
	}
	
	public void moveRight()
	{
		first.moveRight();
		second.moveRight();
	}
	
	public void moveLeft()
	{
		first.moveLeft();
		second.moveLeft();
	}
	
	public void moveDown()
	{
		first.moveDown();
		second.moveDown();
	}
	
	public bit isOutside()
	{
		return first.isOutside() || second.isOutside();
	}
	
	public void save()
	{
		first.save();
		second.save();
	}
	
	public void load()
	{
		first.load();
		second.load();
	}

	public void rotateRight()
	{
		rotate(PI_2);
	}
	
	public void rotateLeft()
	{
		rotate(-PI_2);
	}
	
	private void rotate(real radian)
	{
		Vector v = Vector.create(second.x - first.x, second.y - first.y) * Matrix.rotationZ(radian);
		second.x = first.x + round(v.x);
		second.y = first.y + round(v.y);
	}
	
	private int round(float a)
	{
		return floor(a + 0.5);		
	}

	public Puyo first, second;
}

class PuyoQueue : instance Collections(Puyo).Queue
{
	public bit contains(Puyo a)
	{
		for (int i = 0; i < count; i++) if (at(i) == a) return true;
		return false;
	}
	
	public void[Puyo] toSet()
	{
		void[Puyo] result;
		for (int i = 0; i < count; i++) result[at(i)];
		return result;
	}
}

class StaticPuyoSet
{
	public bit isIntersect(PuyoPuyo puyoPuyo)
	{
		foreach (Puyo a; set.keys)
		{
			if (a.x == puyoPuyo.first.x && a.y == puyoPuyo.first.y) return true;
			if (a.x == puyoPuyo.second.x && a.y == puyoPuyo.second.y) return true;
		}
		return false;
	}
	
	public int opApply(int delegate(inout Puyo) dg)
	{
		int result = 0;
		foreach (Puyo a; set.keys)
		{
		    result = dg(a);
		    if (result) break;
		}
		return result;
	}
	
	public void add(PuyoPuyo puyoPuyo)
	{
		set[puyoPuyo.first];
		set[puyoPuyo.second];
	}
	
	public void drop()
	{
		while (canDrop)
		{
			foreach (Puyo a; set.keys)
			{
				if (a.y + 1 >= HEIGHT_OF_DRAW_PUYO) continue;
				if (isEmpty(a.x, a.y + 1)) a.moveDown();
			}
		}
	}
	
	public void eraseScan()
	{
		void[Puyo] erasePuyoSet;
		foreach (Puyo a; set.keys)
		{
			foreach (Puyo b; getErasiblePuyo(a).keys) erasePuyoSet[b];
		}
		foreach (Puyo a; erasePuyoSet.keys) delete set[a];
	}
	
	private void[Puyo] getErasiblePuyo(Puyo a)
	{
		PuyoQueue unsettledQueue = new PuyoQueue();
		PuyoQueue processedQueue = new PuyoQueue();
		unsettledQueue.enqueue(a);

		while (unsettledQueue.count != 0)
		{
			Puyo currentPuyo = unsettledQueue.dequeue();
			void[Puyo] neighbor = getNeighbor(currentPuyo);
			processedQueue.enqueue(currentPuyo);
			foreach (Puyo b; neighbor.keys)
			{
				if (!unsettledQueue.contains(b) && !processedQueue.contains(b))
				{
					unsettledQueue.enqueue(b);
				}
			}
		}

		void[Puyo] result = processedQueue.toSet();
		return result.length < 4 ? null : result;
	}
	
	private void[Puyo] getNeighbor(Puyo a)
	{
		void[Puyo] result;
		foreach (Puyo b; set.keys)
		{
			if (a.color == b.color && distance(a, b) - 1 < 0.001) result[b];
		}
		return result;
	}
	
	private real distance(Puyo a, Puyo b)
	{
		Vector va = Vector.create(a.x, a.y);
		Vector vb = Vector.create(b.x, b.y);
		return hdk.math.distance(va, vb);
	}
	
	public bit canDrop()
	{
		foreach (Puyo a; set.keys)
		{
			if (a.y + 1 >= HEIGHT_OF_DRAW_PUYO) continue;
			if (isEmpty(a.x, a.y + 1)) return true;
		}
		return false;
	}
	
	private bit isEmpty(int x, int y)
	{
		foreach (Puyo a; set.keys) if (a.x == x && a.y == y) return false;
		return true;
	}

	public bit isGameOver()
	{
		foreach (Puyo a; set.keys)
		{
			if (a.x == 2 && a.y == 0) return true;
			if (a.x == 3 && a.y == 0) return true;
		}
		return false;
	}

	private void[Puyo] set;
}

int main()
{
	//PuyoView.test();

	//*
	PuyoView view = new PuyoView();
	Joystick joystick = new Joystick();
	PuyoPuyo nextPuyoPuyo = new PuyoPuyo();
	PuyoPuyo currentPuyoPuyo = new PuyoPuyo();
	StaticPuyoSet staticPuyoSet = new StaticPuyoSet();
	while (!view.eof)
	{
		bit downFlag = false;
		while (true)
		{
			if (view.eof) throw new Error("interrupt");
			while (true)
			{
				JoyInfo ji = joystick.read();
				if (ji == JoyInfo.empty) break;
				if (ji == JoyInfo.downIsOn) downFlag = true;
				if (ji == JoyInfo.downIsOff) downFlag = false;
				currentPuyoPuyo.save();
				if (ji == JoyInfo.rightIsOn) currentPuyoPuyo.moveRight();
				if (ji == JoyInfo.leftIsOn) currentPuyoPuyo.moveLeft();
				if (ji == JoyInfo.oneIsOn) currentPuyoPuyo.rotateLeft();
				if (ji == JoyInfo.twoIsOn) currentPuyoPuyo.rotateRight();
				if (currentPuyoPuyo.isOutside() || staticPuyoSet.isIntersect(currentPuyoPuyo)) currentPuyoPuyo.load();
			}
			if (downFlag)
			{
				currentPuyoPuyo.save();
				currentPuyoPuyo.moveDown();
				if (currentPuyoPuyo.isOutside() || staticPuyoSet.isIntersect(currentPuyoPuyo))
				{
					currentPuyoPuyo.load();
					break;
				}
			}

			draw(view, currentPuyoPuyo, nextPuyoPuyo, staticPuyoSet);
		}

		staticPuyoSet.add(currentPuyoPuyo);
		do
		{
			draw(view, null, nextPuyoPuyo, staticPuyoSet);
			Sleep(100);
			staticPuyoSet.drop();
			draw(view, null, nextPuyoPuyo, staticPuyoSet);
			Sleep(100);
			staticPuyoSet.eraseScan();
		}
		while (staticPuyoSet.canDrop)
		currentPuyoPuyo = nextPuyoPuyo;
		nextPuyoPuyo = new PuyoPuyo();

		if (staticPuyoSet.isGameOver())
		{
			view.drawGameOver();
			view.update();
			Sleep(1500);
			break;
		}
	}
	//*/

	return 0;
}

void draw(PuyoView view, PuyoPuyo currentPuyoPuyo, PuyoPuyo nextPuyoPuyo, StaticPuyoSet staticPuyoSet)
{
	view.drawBackground();
	foreach (Puyo a; staticPuyoSet) view.drawPuyo(a.x, a.y, a.color);
	if (currentPuyoPuyo !== null)
	{
		view.drawPuyo(currentPuyoPuyo.first.x, currentPuyoPuyo.first.y, currentPuyoPuyo.first.color);
		view.drawPuyo(currentPuyoPuyo.second.x, currentPuyoPuyo.second.y, currentPuyoPuyo.second.color);
	}
	view.drawNextPuyo(nextPuyoPuyo.first.color, nextPuyoPuyo.second.color);
	view.update();
	fullCollect();
}
