import std.stream;
import std.conv;
import std.gc;
import hdk.d3dframe;
import hdk.mouse;

struct Point
{
	int x, y;
}

class UserInterface : D3dFrame
{
	public static void test()
	{
		printf("UserInterface test ");
		UserInterface ui = new UserInterface();
		Mouse mouse = new Mouse(ui.winHandle());
		while (!ui.eof)
		{
			ui.drawBackground();
			ui.drawStone(0, 0, Color.BLACK);
			ui.drawStone(7, 7, Color.WHITE);
			ui.drawString("11͂", 0, 0, Color.LIGHT_RED);
			Point p = UserInterface.cursorPosition(mouse);
			ui.drawCursor(p);
			ui.update();
			fullCollect();

			if (mouse.read() == MouseInfo.leftIsOn
				&& p.x == 1 && p.y == 1) break;
		}
		printf("ok\n");
	}

	public void drawBackground()
	{
		const float CELL_WIDTH = 0.24;
		const float LINE_WIDTH = 0.01;
		VertexData billboard = VertexData.createBillboard(Color.GREEN);
		for (int i = 0; i < 8 * 8; i++)
		{
			int x = i / 8;
			int y = i % 8;
			drawVertices(billboard, null,
				Matrix.scaling(CELL_WIDTH, CELL_WIDTH, 1)
				* Matrix.translation(-1 + CELL_WIDTH * 0.5, 1 - CELL_WIDTH * 0.5, 0)
				* Matrix.translation((CELL_WIDTH + LINE_WIDTH) * x, -(CELL_WIDTH + LINE_WIDTH) * y, 0),
				Matrix.identity, Matrix.identity);
		}
	}

	public void drawCursor(Point p)
	{
		this.drawStone(p.x, p.y, Color.create(Color.GRAY, 0.8));
	}

	public void drawStone(int x, int y, Vector color)
	in
	{
		assert(0 <= x && x <= 7);
		assert(0 <= y && y <= 7);
	}
	body
	{
		const float CELL_WIDTH = 0.24;
		const float LINE_WIDTH = 0.01;
		const float STONE_WIDTH = 0.23;
		drawVertices(VertexData.createBillboard(color), null,
			Matrix.scaling(STONE_WIDTH, STONE_WIDTH, 1)
			* Matrix.translation(-1 + CELL_WIDTH * 0.5, 1 - CELL_WIDTH * 0.5, 0)
			* Matrix.translation((CELL_WIDTH + LINE_WIDTH) * x, -(CELL_WIDTH + LINE_WIDTH) * y, 0),
			Matrix.identity, Matrix.identity);
	}
	
	public static Point cursorPosition(Mouse mouse)
	{
		Point result;
		result.x = floor((mouse.x + 1) * 4);
		result.y = floor((-mouse.y + 1) * 4);
		return result;
	}
}

enum StoneColor
{
	EMPTY, BLACK, WHITE
}

class Board
{
	public this()
	{
		buffer[3][3] = StoneColor.BLACK;
		buffer[4][4] = StoneColor.BLACK;
		buffer[3][4] = StoneColor.WHITE;
		buffer[4][3] = StoneColor.WHITE;
	}

	public void setStoneColor(int x, int y, StoneColor sc) // reverse
	in
	{
		assert(StoneColor.EMPTY == buffer[x][y]);
		assert(canAccept(x, y, sc));
	}
	body
	{
		buffer[x][y] = sc;
		for (int i = 1; i <= 9; i++)
		{
			if (i == 5) continue;
			Point[] line = getLine(x, y, i);
			if (reversible(sc, line))
			{
				for (int j = 0; j < line.length; j++)
				{
					if (getStoneColor(line[j]) == sc) break;
					buffer[line[j].x][line[j].y] = sc;
				}
			}
		}
	}

	public StoneColor getStoneColor(int x, int y)
	{
		return buffer[x][y];
	}
	
	public StoneColor getStoneColor(Point p)
	{
		return buffer[p.x][p.y];
	}

	private StoneColor[8][8] buffer;

	public bit canAccept(int x, int y, StoneColor sc)
	{
		if (buffer[x][y] != StoneColor.EMPTY) return false;
		for (int i = 1; i <= 9; i++)
		{
			if (i == 5) continue;
			Point[] line = getLine(x, y, i);
			if (reversible(sc, line)) return true;
		}
		return false;
	}
	
	// (x, y)5Ƃtenkeyւ̔zԂ
	private Point[] getLine(int x, int y, int tenkey)
	in
	{
		assert(0 <= tenkey && tenkey <= 9 && 5 != tenkey);
	}
	body
	{
		Point[] result = new Point[7];
		int resultLength = 0;
		int dx = (tenkey - 1) % 3 - 1;
		int dy = -((tenkey - 1) / 3 - 1);
		for (int i = 0; i < 7; i++)
		{
			resultLength++;
			if (!onBoard(x + dx * resultLength, y + dy * resultLength))
			{
				resultLength--;
				break;
			}

			result[i].x = x + dx * resultLength;
			result[i].y = y + dy * resultLength;
		}
		result.length = resultLength;
		return result;
	}
	
	private bit onBoard(int x, int y)
	{
		return 0 <= x && x <= 7 && 0 <= y && y <= 7;
	}
	
	private bit reversible(StoneColor myColor, Point[] line)
	{
		if (line.length < 2) return false;
		StoneColor enemyColor = myColor == StoneColor.BLACK ? StoneColor.WHITE : StoneColor.BLACK;
		if (getStoneColor(line[0]) != enemyColor) return false;
		for (int i = 1; i < line.length; i++)
		{
			if (getStoneColor(line[i]) == StoneColor.EMPTY) return false;
			if (getStoneColor(line[i]) == myColor) return true;
		}
		return false;
	}

	unittest
	{
		printf("Board test ");
		Board board = new Board();
		Point[] p = board.getLine(1, 1, 1);
		assert(1 == p.length);
		p = board.getLine(1, 1, 2);
		assert(6 == p.length);
		printf("ok\n");
	}
	
	public bit cannotChoose(StoneColor sc)
	{
		for (int i = 0; i < 8 * 8; i++)
		{
			int x = i / 8;
			int y = i % 8;
			if (canAccept(x, y, sc)) return false;
		}
		return true;
	}
	
	public bit isGameEnd()
	{
		return cannotChoose(StoneColor.BLACK) && cannotChoose(StoneColor.WHITE);
	}
	
	public int count(StoneColor sc)
	{
		int result = 0;
		for (int i = 0; i < 8 * 8; i++)
		{
			int x = i / 8;
			int y = i % 8;
			if (buffer[x][y] == sc) result++;
		}
		return result;
	}
}

void drawBoard(UserInterface ui, Board b)
{
	ui.drawBackground();
	for (int i = 0; i < 8 * 8; i++)
	{
		int x = i / 8;
		int y = i % 8;
		StoneColor color = b.getStoneColor(x, y);
		if (color != StoneColor.EMPTY)
		{
			Vector color2 = color == StoneColor.BLACK ? Color.BLACK : Color.WHITE;
			ui.drawStone(x, y, color2);
		}
	}
}

bit onBoard(Point cursorPosition)
{
	return 0 <= cursorPosition.x && cursorPosition.x <= 7
		&& 0 <= cursorPosition.y && cursorPosition.y <= 7;
}

void inputPlayer(UserInterface ui, Mouse mouse, Board board)
{
	if (board.cannotChoose(StoneColor.BLACK)) return;
	while (!ui.eof)
	{
		Point cursorPosition = UserInterface.cursorPosition(mouse);
		drawBoard(ui, board);
		if (onBoard(cursorPosition)) ui.drawCursor(cursorPosition);
		ui.update();
		fullCollect();
		if (MouseInfo.leftIsOn == mouse.read() && onBoard(cursorPosition)
			&& board.canAccept(cursorPosition.x, cursorPosition.y, StoneColor.BLACK))
		{
			board.setStoneColor(cursorPosition.x, cursorPosition.y, StoneColor.BLACK);
			break;
		}
	}
}

void inputComputer(Board board)
{
	if (board.cannotChoose(StoneColor.WHITE)) return;
	const Point[] p =
	[
		{0, 0}, {7, 0}, {0, 7}, {7, 7},
		
		// [
		{2, 0}, {3, 0}, {4, 0}, {5, 0},
		{2, 7}, {3, 7}, {4, 7}, {5, 7},
		{0, 2}, {0, 3}, {0, 4}, {0, 5},
		{7, 2}, {7, 3}, {7, 4}, {7, 5},

		// 
		{2, 2}, {3, 2}, {4, 2}, {5, 2},
		{2, 3},                 {5, 3},
		{2, 4},                 {5, 4},
		{2, 5}, {3, 5}, {4, 5}, {5, 5},

		// ̑
		{2, 1}, {3, 1}, {4, 1}, {5, 1},
		{2, 6}, {3, 6}, {4, 6}, {5, 6},
		{1, 2}, {1, 3}, {1, 4}, {1, 5},
		{6, 2}, {6, 3}, {6, 4}, {6, 5},

		// ܂łȂ[
		        {1, 0}, {6, 0},
		{0, 1}, {1, 1}, {6, 1}, {7, 1},
		{0, 6}, {1, 6}, {6, 6}, {7, 6},
		        {1, 7}, {6, 7},
	];
	foreach (Point a; p)
	{
		if (board.canAccept(a.x, a.y, StoneColor.WHITE))
		{
			board.setStoneColor(a.x, a.y, StoneColor.WHITE);
			break;
		}
	}
}

int main()
{
	//UserInterface.test();
	//*
	UserInterface ui = new UserInterface();
	Mouse mouse = new Mouse(ui.winHandle());
	Board board = new Board();
	while (true)
	{
		inputPlayer(ui, mouse, board);
		if (ui.eof) return 0;

		drawBoard(ui, board);
		ui.update();
		fullCollect();
		Sleep(250);

		inputComputer(board);

		if (board.isGameEnd()) break;
	}
	char[] text = board.count(StoneColor.BLACK) > board.count(StoneColor.WHITE) ? "̏" :
		board.count(StoneColor.BLACK) < board.count(StoneColor.WHITE) ? "̏" :
		"";
	while (!ui.eof)
	{
		drawBoard(ui, board);
		ui.drawString(text, 0, 0, Color.LIGHT_RED);
		ui.update();
		fullCollect();
	}
	//*/
	return 0;
}
