﻿module anchor;
import
	std.cstream, std.string, std.stream, std.file,
	coneneko.math, coneneko.sjis2utf8,
	mqo, skeleton;

interface _Anchor
{
	static Anchor[] createList(Mqo.Material[] material, Mqo.Object anchorObject);
	//Anchor.this(Vector[] triangleList, char[] materialName)
	bit has(Vector point);
	char[] materialName();
	static bit isAnchor(Mqo.Object obj);
	static char[] extractRelationalObjectName(Mqo.Object anchorObject);
}


class Anchor : _Anchor
{
	unittest
	{
		assert(isAnchor("anchor1|hoge"));
		assert(isAnchor("anchor2|hoge"));
		assert(!isAnchor("hoge"));
		Vector[] vertex = new Vector[8];
		vertex[0] = Vector.create(-100.0000, 100.0000, -100.0000);
		vertex[1] = Vector.create(100.0000, 100.0000, -100.0000);
		vertex[2] = Vector.create(-100.0000, 100.0000, 100.0000);
		vertex[3] = Vector.create(100.0000, 100.0000, 100.0000);
		vertex[4] = Vector.create(-100.0000, -100.0000, 100.0000);
		vertex[5] = Vector.create(100.0000, -100.0000, 100.0000);
		vertex[6] = Vector.create(-100.0000, -100.0000, -100.0000);
		vertex[7] = Vector.create(100.0000, -100.0000, -100.0000);
		const uint[] indexList =
		[
			1, 3, 2,  1, 2, 0,  3, 5, 4,  3, 4, 2,
			5, 7, 6,  5, 6, 4,  7, 1, 0,  7, 0, 6,
			7, 5, 3,  7, 3, 1,  0, 2, 4,  0, 4, 6,
		];
		Vector[] boxData = new Vector[indexList.length];
		for (int i = 0; i < indexList.length; i++)
		{
			boxData[i] = vertex[indexList[i]];
		}
		Anchor box = new Anchor(boxData, "hoge");
		assert(box.has(Vector.create(0, 0, 0)));
		assert(!box.has(Vector.create(0, 200, 0)));
		assert(!box.has(Vector.create(0, -200, 0)));
		assert(!box.has(Vector.create(150, 0, 0)));
		
		Vector[] polyData = new Vector[3];
		polyData[0] = Vector.create(0, 1, 0);
		polyData[1] = Vector.create(1, 0, 0);
		polyData[2] = Vector.create(0, 0, 1);
		Anchor poly = new Anchor(polyData, "hoge");
		assert(!poly.has(Vector.create(0.5, 0.5, 0.5)));
	}
	
	private final Vector[] triangleList;
	private final char[] _materialName;
	
	this(Vector[] triangleList, char[] materialName)
	in
	{
		assert(0 == triangleList.length % 3);
	}
	body
	{
		this.triangleList = triangleList;
		this._materialName = materialName;
	}
	
	char[] materialName() { return _materialName; }
	
	static bit isAnchor(Mqo.Object obj) { return isAnchor(obj.name); }
	private static bit isAnchor(char[] name)
	{
		return 6 <= name.length && name[0..6] == "anchor" ? true : false;
	}
	
	bit has(Vector point)
	{
		Vector[] dir = new Vector[3];
		dir[0] = Vector.create(1, 0, 0);
		dir[1] = Vector.create(0, 1, 0);
		dir[2] = Vector.create(0, 0, 1);
		bit[] bits = new bit[6];
		assert(false == bits[0]);
		for (int i = 0; i < triangleList.length; i += 3)
		{
			Vector p0 = triangleList[i];
			Vector p1 = triangleList[i + 1];
			Vector p2 = triangleList[i + 2];
			float dist;
			if (!(bits[0] && bits[1])
				&& triangleIntersectRay2(p0, p1, p2, point, dir[0], dist))
			{
				if (0 <= dist) bits[0] = true;
				if (dist <= 0) bits[1] = true;
			}
			if (!(bits[2] && bits[3])
				&& triangleIntersectRay2(p0, p1, p2, point, dir[1], dist))
			{
				if (0 <= dist) bits[2] = true;
				if (dist <= 0) bits[3] = true;
			}
			if (!(bits[4] && bits[5])
				&& triangleIntersectRay2(p0, p1, p2, point, dir[2], dist))
			{
				if (0 <= dist) bits[4] = true;
				if (dist <= 0) bits[5] = true;
			}
		}
		return bits[0] && bits[1] && bits[2] && bits[3] && bits[4] && bits[5] ? true : false;
	}
	
	static Anchor[] createList(Mqo.Material[] material, Mqo.Object anchorObject)
	in
	{
		assert(isAnchor(anchorObject));
	}
	body
	{
		alias Vector[] VectorList;
		alias char[] MaterialName;
		VectorList[MaterialName] triangleMap;
		foreach (Mqo.Face face; anchorObject.face)
		{
			triangleMap[material[face.M].name] ~= createTriangle(anchorObject, face);
		}
		
		Anchor[] result;
		foreach (char[] a; triangleMap.keys)
		{
			result ~= new Anchor(triangleMap[a], a);
		}
		return result;
	}
	
	static Vector[] createTriangle(Mqo.Object obj, Mqo.Face face)
	{
		const uint[] indexList3 = [ 0, 1, 2 ];
		const uint[] indexList4 = [ 0, 1, 2,  0, 2, 3 ];
		uint[] indexList = face.vlength == 3 ? indexList3 : indexList4;
		Vector[] result;
		foreach (uint a; indexList) result ~= toVector(obj.vertex[face.V[a]]);
		return result;
	}
	
	private static Vector toVector(Mqo.Vector3 a) { return Vector.create(a.x, a.y, a.z); }
	
	
	unittest
	{
		assert("hoge" == extractRelationObjectName("hoge|hoge"));
		assert("aaa" == extractRelationObjectName("abc|aaa"));
		assert("abc" != extractRelationObjectName("abc|aaa"));
	}
	static char[] extractRelationalObjectName(Mqo.Object anchorObject)
	in
	{
		assert(isAnchor(anchorObject));
	}
	body
	{
		return extractRelationObjectName(anchorObject.name);
	}
	private static char[] extractRelationObjectName(char[] name)
	{
		char[][] a = split(name, "|");
		assert(2 == a.length);
		return a[1];
	}
}

class AnchorSet
{
	Anchor[] anchorList;
	
	unittest
	{
		assert(isSdef("sdef:hoge"));
		assert(!isSdef("aaa:hoge"));
	}
	
	static bit isSdef(char[] name)
	{
		return "sdef:".length <= name.length && "sdef:" == name[0..5] ? true : false;
	}
	
	static bit isSdef(Mqo.Object obj) { return isSdef(obj.name); }
	
	private Mqo.Object[] objList;
	private Mqo.Material[] matList;
	private Skeleton skeleton;
	
	this(Mqo mqo, Mqo.Object obj, Skeleton skeleton)
	{
		objList = mqo.object;
		matList = mqo.material;
		this.skeleton = skeleton;
		anchorList = createRelationalAnchorList(obj);
	}
	
	// objとつながっているAnchor[]を返す
	private Anchor[] createRelationalAnchorList(Mqo.Object obj)
	in
	{
		assert(isSdef(obj));
	}
	body
	{
		Anchor[] result;
		foreach (Mqo.Object a; objList)
		{
			if (Anchor.isAnchor(a)
				&& removeSdef(obj.name) == Anchor.extractRelationalObjectName(a))
			{
				result ~= Anchor.createList(matList, a);
			}
		}
		return result;
	}
	
	unittest
	{
		assert("hoge" == removeSdef("sdef:hoge"));
		assert("aaa" == removeSdef("aaa"));
	}
	
	private static char[] removeSdef(char[] a)
	{
		if (a.length <= 5) return a;
		if (a[0..5] == "sdef:") return a[5..a.length];
		return a;
	}
	
	Vector getMatrixIndex(Vector p)
	{
		uint[] ownerBoneIndexList = getOwnerBoneIndexList(p);
		uint first, second;
		if (ownerBoneIndexList.length >= 2)
		{
			float[] distanceList = new float[ownerBoneIndexList.length];
			for (int i = 0; i < distanceList.length; i++)
			{
				distanceList[i] = Bone.distanceFromCenter(
					skeleton.boneList[ownerBoneIndexList[i]], p
				);
			}
			first = ownerBoneIndexList[firstIndexOf(distanceList)];
			second = ownerBoneIndexList[secondIndexOf(distanceList)];
		}
		else if (ownerBoneIndexList.length == 1)
		{
			first = ownerBoneIndexList[0];
			second = ownerBoneIndexList[0];
		}
		else
		{
			first = second = getNearBoneIndex(p);
		}
		return Vector.create(cast(float)first, cast(float)second);
	}
	
	// pの所有ボーン候補リスト
	uint[] getOwnerBoneIndexList(Vector p)
	{
		uint[] result;
		foreach (Anchor b; anchorList)
		{
			if (!b.has(p)) continue;
			int boneIndex = getBoneIndex(b.materialName, p);
			if (boneIndex == -1)
			{
				//throw new Error("not found " ~ materialName ~ " from bone material name");
				continue;
			}
			result ~= cast(uint)boneIndex;
		}
		return result;
	}
	
	unittest
	{
		float[] distanceList;
		distanceList ~= 2.0;
		distanceList ~= 0.0;
		distanceList ~= 0.5;
		distanceList ~= 1.0;
		assert(1 == firstIndexOf(distanceList));
		assert(2 == secondIndexOf(distanceList));
		
		float[] list2;
		list2 ~= 0.1;
		list2 ~= 0.2;
		assert(0 == firstIndexOf(list2));
		assert(1 == secondIndexOf(list2));
	}
	
	private static ubyte firstIndexOf(float[] distanceList)
	in
	{
		assert(1 <= distanceList.length);
	}
	body
	{
		ubyte result = 0;
		for (int i = 0; i < distanceList.length; i++)
		{
			if (distanceList[result] > distanceList[i]) result = cast(ubyte)i;
		}
		return result;
	}
	
	private static ubyte secondIndexOf(float[] distanceList)
	in
	{
		assert(2 <= distanceList.length);
	}
	body
	{
		ubyte firstIndex = firstIndexOf(distanceList);
		ubyte result = firstIndex == 0 ? 1 : 0;
		for (int i = 0; i < distanceList.length; i++)
		{
			if (i == firstIndex) continue;
			if (distanceList[result] > distanceList[i]) result = cast(ubyte)i;
		}
		return result;
	}
	
	uint getNearBoneIndex(Vector p)
	{
		float[] distanceList = new float[skeleton.boneList.length];
		for (int i = 0; i < distanceList.length; i++)
		{
			distanceList[i] = Bone.distanceFromLineSegment(skeleton.boneList[i], p);
		}
		
		uint result;
		bit hasInner = false;
		for (int i = 0; i < distanceList.length; i++)
		{
			bit isInner = distanceList[i] < skeleton.boneList[i].height;
			if (!isInner) continue;
			if (!hasInner)
			{
				hasInner = true;
				result = i;
			}
			else
			{
				if (distanceList[i] < distanceList[result]) result = i;
			}
		}
		
		return hasInner ? result : firstIndexOf(distanceList);
	}
	
	
	private bit hasBone()
	{
		foreach (Mqo.Object a; objList)
		{
			if ("bone:".length <= a.name.length && a.name[0..5] == "bone:") return true;
		}
		return false;
	}
	
	private int getBoneIndex(char[] materialName, Vector position)
	{
		int result = indexOfBoneName(skeleton, getBoneName(materialName, position));
		if (result == -1) result = indexOfBoneMaterialName(skeleton, materialName);
		return result;
	}
	
	private int indexOfBoneName(Skeleton skeleton, char[] name)
	{
		for (int i = 0; i < skeleton.boneList.length; i++)
		{
			if (skeleton.boneList[i].name == name) return i;
		}
		return -1;
	}
	
	private int indexOfBoneMaterialName(Skeleton skeleton, char[] materialName)
	{
		for (int i = 0; i < skeleton.boneList.length; i++)
		{
			if (skeleton.boneList[i].materialName == materialName) return i;
		}
		return -1;
	}
	
	private static char[] getBoneName(char[] materialName, Vector position) // []の中のR, Lを入れる
	{
		return insertRorL(materialName, position.x);
	}
	
	private static char[] insertRorL(char[] a, float x)
	{
		if (-1 == find(a, "[]")) return a;
		if (x >= 0) return replace(a, "[]", "[L]");
		else return replace(a, "[]", "[R]");
	}
}

void printOwnerBone(char[] mqoFileName, Vector position)
{
	dout.writeLine(mqoFileName);
	dout.writeLine(position.toString());
	
	char[] mqoSrc = cast(char[])read(mqoFileName);
	mqoSrc = sjisToUtf8(mqoSrc);
	Mqo mqo = new Mqo(new MemoryStream(mqoSrc));
	
	Skeleton skeleton = new Skeleton(mqo);
	
	AnchorSet anchorSet;
	foreach (Mqo.Object a; mqo.object)
	{
		if (!AnchorSet.isSdef(a)) continue;
		anchorSet = new AnchorSet(mqo, a, skeleton);
		dout.writeLine(a.name);
		break;
	}
	
	if (!anchorSet) throw new Error("sdef not found");
	
	dout.writeLine("owner bone index list");
	foreach (uint a; anchorSet.getOwnerBoneIndexList(position))
	{
		dout.writeLine(skeleton.boneList[a].name);
	}
	dout.writeLine("");
	
	Vector mi = anchorSet.getMatrixIndex(position);
	
	dout.writeLine("result");
	dout.writeLine(skeleton.boneList[cast(uint)mi.x].name);
	dout.writeLine(skeleton.boneList[cast(uint)mi.y].name);
}
