﻿module blackboard;
private import mqo;
private import mkm;
private import skeleton;
private import coneneko.math;
private import std.stream;
private import std.cstream;
private import coneneko.sdlwindow;

interface KnowledgeSource
{
	bit condition();
	void opCall();
}

class Blackboard
{
	char[] mqoFileName, mkmFileName, mqoDirectory;
	bit flatvertex;
	
	void[] result;
	
	Mqo mqo;
	Mkm mkm;
	Skeleton skeleton;
	
	alias char[] string;
	string[][] textureFileName;
	
	TextureInfo[][] texture;
	VertexBufferInfo[][] vertexBuffer;
	
	int opApply(int delegate(inout VertexBufferInfo vb, inout uint x, inout uint y) dg)
	{
		for (uint y = 0; y < vertexBuffer.length; y++)
		{
			for (uint x = 0; x < vertexBuffer[y].length; x++)
			{
				if (!vertexBuffer[y][x]) continue;
				int result = dg(vertexBuffer[y][x], x, y);
				if (result != 0) return result;
			}
		}
		return 0;
	}
	
	MotionMatrix[] motionMatrix;
}

class TextureInfo
{
	uint[] rgba;
	uint width, height;
	invariant { assert(rgba.length == width * height); }
	
	this(char[] textureFileName)
	{
		rgba = ImageLoader.readImage(textureFileName, width, height);
	}
	
	this(uint[] rgba, uint width, uint height)
	{
		this.rgba = rgba;
		this.width = width;
		this.height = height;
	}
	
	static TextureInfo createNullTexture()
	{
		const uint[] rgba = [ uint.max ];
		return new TextureInfo(rgba, 1, 1);
	}
	
	static TextureInfo lerp(TextureInfo previous, TextureInfo next, float zero_one)
	{
		uint width = max(previous.width, next.width);
		uint height = max(previous.height, next.height);
		uint[] rgba = new uint[width * height];
		for (int y = 0; y < height; y++)
		{
			float fy = cast(float)y / cast(float)height;
			for (int x = 0; x < width; x++)
			{
				float fx = cast(float)x / cast(float)width;
				rgba[y * width + x] = lerp(previous[fx, fy], next[fx, fy], zero_one);
			}
		}
		return new TextureInfo(rgba, width, height);
	}
	
	private static uint max(uint a, uint b) { return a >= b ? a : b; }
	
	private uint opIndex(float x, float y)
	in
	{
		assert(0.0 <= x && x <= 1.0);
		assert(0.0 <= y && y <= 1.0);
	}
	body
	{
		uint x2 = cast(uint)(x * width);
		uint y2 = cast(uint)(y * height);
		return rgba[y2 * width + x2];
	}
	
	private static uint lerp(uint pixelA, uint pixelB, float s)
	{
		ubyte[] result = new ubyte[4];
		ubyte[] ubaA = toUbyteArray(pixelA);
		ubyte[] ubaB = toUbyteArray(pixelB);
		result[0] = lerp(ubaA[0], ubaB[0], s);
		result[1] = lerp(ubaA[1], ubaB[1], s);
		result[2] = lerp(ubaA[2], ubaB[2], s);
		result[3] = lerp(ubaA[3], ubaB[3], s);
		return toUint(result);
	}
	
	private static ubyte lerp(ubyte a, ubyte b, float s)
	{
		return cast(ubyte)(a + (b - a) * s);
	}
	
	unittest
	{
		ubyte[] rgbaPixel = new ubyte[4];
		rgbaPixel[0] = 10;
		rgbaPixel[1] = 1;
		rgbaPixel[2] = 2;
		rgbaPixel[3] = 3;
		assert(rgbaPixel[0] == toUbyteArray(toUint(rgbaPixel))[0]);
		assert(rgbaPixel[1] == toUbyteArray(toUint(rgbaPixel))[1]);
		assert(rgbaPixel[2] == toUbyteArray(toUint(rgbaPixel))[2]);
		assert(rgbaPixel[3] == toUbyteArray(toUint(rgbaPixel))[3]);
	}
	
	private static ubyte[] toUbyteArray(uint a)
	out (result)
	{
		assert(4 == result.length);
	}
	body
	{
		ubyte[] result = new ubyte[4];
		result[3] = cast(ubyte)(a >> 24);
		result[2] = cast(ubyte)(a << 8 >> 24);
		result[1] = cast(ubyte)(a << 16 >> 24);
		result[0] = cast(ubyte)(a << 24 >> 24);
		return result;
	}
		
	private static uint toUint(ubyte[] rgbaPixel)
	in
	{
		assert(4 == rgbaPixel.length);
	}
	body
	{
		return (rgbaPixel[3] << 24) | (rgbaPixel[2] << 16)
			| (rgbaPixel[1] << 8) | rgbaPixel[0];
	}
}

class VertexBufferInfo
{
	uint sizeOfVertex = 0;
	float[] buffer;
	void add(float[] a) { buffer ~= a; }
	
	static VertexBufferInfo lerp(VertexBufferInfo previous, VertexBufferInfo next, float zero_one)
	in
	{
		assert(previous.sizeOfVertex == next.sizeOfVertex);
		assert(previous.buffer.length == next.buffer.length);
	}
	out (result)
	{
		assert(previous.buffer.length == result.buffer.length);
	}
	body
	{
		VertexBufferInfo result = new VertexBufferInfo();
		result.sizeOfVertex = previous.sizeOfVertex;
		result.buffer.length = previous.buffer.length;
		
		uint normalPosition = 7 * previous.vertexLength;
		uint texCoordPosition = 10 * previous.vertexLength;
		uint i = 0;
		
		while (i < normalPosition)
		{
			result.buffer[i] = .lerp(previous.buffer[i], next.buffer[i], zero_one);
			i++;
		}
		while (i < texCoordPosition)
		{
			float x = .lerp(previous.buffer[i], next.buffer[i], zero_one);
			float y = .lerp(previous.buffer[i + 1], next.buffer[i + 1], zero_one);
			float z = .lerp(previous.buffer[i + 2], next.buffer[i + 2], zero_one);
			Vector n = normalize(Vector.create(x, y, z));
			result.buffer[i] = n.x;
			result.buffer[i + 1] = n.y;
			result.buffer[i + 2] = n.z;
			i += 3;
		}
		while (i < previous.buffer.length)
		{
			result.buffer[i] = .lerp(previous.buffer[i], next.buffer[i], zero_one);
			i++;
		}
		
		return result;
	}
	
	uint vertexLength()
	{
		if (sizeOfVertex == 0) return 0;
		return buffer.length / sizeOfVertex;
	}
}
