﻿import blackboard;
import
	std.cstream, std.stream, std.file, std.path, std.string, std.conv, std.math,
	coneneko.math, coneneko.sjis2utf8, coneneko.nanami,
	mqo, mkm, skeleton, anchor, mmgenerator;

int main(char[][] args)
{
	try
	{
		foreach (inout char[] a; args) a = sjisToUtf8(a);
		chdir(getDirName(args[0]));
		
		ArgsInfo ai = new ArgsInfo(args);
		assert(getcwd() == ai.exeDirectory);
		
		if (ai.printOwnerBone)
		{
			printOwnerBone(ai.mqoFileName, ai.position);
			return 0;
		}
		
		ToNanami tn = new ToNanami(ai);
		tn.loop();
		if (!tn.result) throw new Error("failed");
		
		write(ai.nanamiFileName, tn.result);
		dout.writefln(`created "%s"`, ai.nanamiFileName);
		dout.flush();
	}
	catch (Exception e)
	{
		dout.flush();
		derr.writeLine(e.toString());
		dout.writeLine("---- failed ----");
		din.readLine();
		dout.writeLine("end of process");
	}
	return 0;
}

// できるだけ絶対パスを返す
// toon shaderと toon textureはjoin(exeDirectory, ...)
// mqo textureはjoin(mqoDirectory, ...)
private class ArgsInfo
{
	unittest
	{
		char[][] args;
		args ~= `C:\test\foo.exe`;
		args ~= `C:\test\mqo\test.mqo`;
		args ~= `mqo\test.mkm`;
		args ~= `mqo\test.773`;
		args ~= `-flatvertex`;
		ArgsInfo info = new ArgsInfo(args);
		assert(`C:\test` == info.exeDirectory);
		assert(`C:\test\mqo` == info.mqoDirectory);
		assert(`C:\test\mqo\test.mqo` == info.mqoFileName);
		assert(`C:\test\mqo\test.mkm` == info.mkmFileName);
		assert(`C:\test\mqo\test.773` == info.nanamiFileName);
		assert(true == info.flatvertex);
		
		char[][] args2;
		args2 ~= `C:\テスト\foo.exe`;
		args2 ~= `C:\データ\test.mqo`;
		args2 ~= `C:\データ\test.mkm`;
		ArgsInfo info2 = new ArgsInfo(args2);
		assert(`C:\テスト` == info2.exeDirectory);
		assert(`C:\データ` == info2.mqoDirectory);
		assert(`C:\データ\test.mqo` == info2.mqoFileName);
		assert(`C:\データ\test.mkm` == info2.mkmFileName);
		assert(`C:\データ\test.773` == info2.nanamiFileName);
		assert(false == info2.flatvertex);
	}
	
	private final char[][] args;
	this(char[][] args) { this.args = args; }
	
	char[] mqoFileName()
	{
		const char[] MSG = "tonanami mqoFileName [mkmFileName] [mmzFileName] [nanamiFileName] [-flatvertex]";
		char[] result = getFileName("mqo");
		if (!result) throw new Error(MSG);
		return result;
	}
	
	char[] mkmFileName()
	{
		return getFileName("mkm");
	}
	
	char[] nanamiFileName()
	{
		char[] result = getFileName("773");
		return result ? result : addExt(mqoFileName, "773");
	}
	
	private char[] getFileName(char[] ext)
	{
		foreach (char[] a; args)
		{
			if (0 != icmp(ext, getExt(a))) continue;
			return isabs(a) ? a : std.path.join(exeDirectory, a);
		}
		return null;
	}
	
	bit flatvertex()
	{
		foreach (char[] a; args)
		{
			if ("-flatvertex" == a) return true;
		}
		return false;
	}
	
	char[] exeDirectory()
	{
		return getDirName(args[0]);
	}
	
	char[] mqoDirectory()
	{
		foreach (char[] a; args)
		{
			if (0 != icmp("mqo", getExt(a))) continue;
			return getDirName(isabs(a) ? a : std.path.join(exeDirectory, a));
		}
		throw new Error("mqo directory not found");
	}
	
	bit printOwnerBone()
	{
		foreach (char[] a; args)
		{
			if ("-print_owner_bone" == a) return true;
		}
		return false;
	}
	
	Vector position()
	{
		return vector(atof(args[3]), atof(args[4]), atof(args[5]));
	}
}

private class ToNanami
{
	private Blackboard blackboard;
	private KnowledgeSource[] ksList;
	
	this(ArgsInfo argsInfo)
	{
		blackboard = new Blackboard();
		blackboard.mqoFileName = argsInfo.mqoFileName;
		blackboard.mkmFileName = argsInfo.mkmFileName;
		blackboard.flatvertex = argsInfo.flatvertex;
		blackboard.mqoDirectory = argsInfo.mqoDirectory;
		
		ksList ~= new MqoLoader(blackboard);
		ksList ~= new MkmLoader(blackboard);
		ksList ~= new SkeletonGenerator(blackboard);
		ksList ~= new TextureFileNameExtractor(blackboard);
		ksList ~= new TextureLoader(blackboard);
		ksList ~= new TextureCompletion(blackboard);
		ksList ~= new PositionExtractor(blackboard);
		ksList ~= new ColorExtractor(blackboard);
		ksList ~= new FlatNormalExtractor(blackboard);
		ksList ~= new SmoothNormalExtractor(blackboard);
		ksList ~= new TexCoordExtractor(blackboard);
		ksList ~= new VertexBufferCompletion(blackboard);
		ksList ~= new ZipGenerator(blackboard);
		ksList ~= new MatrixIndexExtractor(blackboard);
		ksList ~= new WeightExtractor(blackboard);
		ksList ~= new MotionMatrixExtractor(blackboard);
	}
	
	void loop()
	{
		uint loopCount = 0;
		while (true)
		{
			loopCount++;
			if (loopCount > 100) throw new Error("loop?");
			
			KnowledgeSource source = nextSource();
			if (!source) return;
			source();
			
			dout.flush();
		}
	}
	
	private KnowledgeSource nextSource()
	{
		foreach (KnowledgeSource a; ksList)
		{
			if (a.condition) return a;
		}
		return null;
	}
	
	void[] result() { return blackboard.result; }
}

class MqoLoader : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		return bb.mqoFileName && !bb.mqo ? true : false;
	}
	
	void opCall()
	{
		dout.writef(`"%s" loading...`, bb.mqoFileName);
		dout.flush();
		
		char[] src = cast(char[])read(bb.mqoFileName);
		src = sjisToUtf8(src);
		bb.mqo = new Mqo(new MemoryStream(src));
		
		dout.writeLine("ok");
	}
}

class MkmLoader : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		return bb.mkmFileName && !bb.mkm ? true : false;
	}
	
	void opCall()
	{
		dout.writef(`"%s" loading...`, bb.mkmFileName);
		dout.flush();
		
		char[] src = cast(char[])read(bb.mkmFileName);
		src = sjisToUtf8(src);
		bb.mkm = new Mkm(new MemoryStream(src));
		
		dout.writeLine("ok");
	}
}

class SkeletonGenerator : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		return bb.mqo && bb.mkm && !bb.skeleton ? true : false;
	}
	
	void opCall()
	{
		dout.writeString("skeleton generating...");
		dout.flush();
		
		bb.skeleton = new Skeleton(bb.mqo);
		
		dout.writeLine("ok");
	}
}

class TextureFileNameExtractor : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		return bb.mqo && !bb.textureFileName ? true : false;
	}
	
	void opCall()
	{
		dout.writeLine("texture file name extracting...");
		dout.flush();
		
		SubsetName[] snlist;
		foreach (char[] a; objectNameList)
		{
			if (SubsetName.isSubset(a)) snlist ~= new SubsetName(a);
		}
		
		bb.textureFileName.length = SubsetName.maxY(snlist) + 1;
		
		bb.textureFileName[0] = join(bb.mqoDirectory, createFirstTextureFileNameList());
		
		for (int i = 1; i < bb.textureFileName.length; i++)
		{
			if (!SubsetName.hasY(snlist, i)) continue;
			
			char[][] textureNameList;
			textureNameList.length = SubsetName.maxX(snlist, i) + 1;
			
			foreach (SubsetName a; snlist)
			{
				if (a.y != i) continue;
				char[] tn = getObjectTextureName(a);
				if (!tn) continue;
				textureNameList[a.x] = std.path.join(bb.mqoDirectory, tn);
			}
			
			bb.textureFileName[i] = textureNameList;
		}
		
		printTextureFileName();
		dout.writeLine("ok");
	}
	
	private static bit has(char[][] list, char[] text)
	{
		foreach (char[] a; list) if (a == text) return true;
		return false;
	}
	
	private char[][] createFirstTextureFileNameList()
	{
		char[][] result;
		result ~= null;
		foreach (Mqo.Material a; bb.mqo.material)
		{
			if (!result.has(a.tex)) result ~= a.tex;
		}
		return result;
	}
	
	private static char[][] join(char[] path, char[][] list)
	{
		char[][] result;
		foreach (char[] a; list)
		{
			result ~= a == null ? null : std.path.join(path, a);
		}
		return result;
	}
	
	private char[][] objectNameList()
	{
		char[][] result;
		foreach (_Mqo.Object a; bb.mqo.object) result ~= a.name;
		return result;
	}
	
	private char[] getObjectTextureName(SubsetName sn)
	{
		_Mqo.Object obj;
		foreach (_Mqo.Object a; bb.mqo.object)
		{
			if (a.name != sn.objectName) continue;
			obj = a;
			break;
		}
		
		if (bb.mqo.material == null) return "";
		
		foreach (_Mqo.Face a; obj.face)
		{
			if (bb.mqo.material[a.M].tex != null) return bb.mqo.material[a.M].tex;
		}
		
		return null;
	}
	
	private void printTextureFileName()
	{
		dout.writefln("textureFileName.length = %d;", bb.textureFileName.length);
		for (int y = 0; y < bb.textureFileName.length; y++)
		{
			dout.writefln("textureFileName[%d].length = %d;",
				y, bb.textureFileName[y].length);
			for (int x = 0; x < bb.textureFileName[y].length; x++)
			{
				if (bb.textureFileName[y][x] == null) continue;
				dout.writefln(`textureFileName[%d][%d] = "%s";`,
					y, x, bb.textureFileName[y][x]);
			}
		}
	}
}

class SubsetName
{
	unittest
	{
		assert(!isSubset("hoge"));
		assert(!isSubset("subset[0][99]:hoge"));
		assert(isSubset("sdef:subset[0][99]:hoge"));
		assert(isSubset("bdef:subset[0][99]:hoge"));
	}
	
	// 繰り返し呼ぶので高速に
	static bool isSubset(char[] objectName)
	{
		const char[] DEF_HEAD = "def:subset[";
		const char[] SDEF_HEAD = "sdef:subset[][]:";
		const char[] BDEF_HEAD = "bdef:subset[][]:";
		const int MIN_LENGTH = SDEF_HEAD.length + 2;
		const int I_BEGIN = "_def:subset[".length;
		static assert('0' < '9');
		
		if (objectName.length < MIN_LENGTH) return false;
		if (objectName[0] != 's' && objectName[0] != 'b') return false;
		if (objectName[1..1 + DEF_HEAD.length] != DEF_HEAD) return false;
		
		int i = I_BEGIN;
		if (objectName[i] < '0' || '9' < objectName[i]) return false;
		i++;
		while ('0' <= objectName[i] && objectName[i] <= '9') i++;
		if (objectName[i] != ']') return false;
		i++;
		if (objectName[i] != '[') return false;
		i++;
		if (objectName[i] < '0' || '9' < objectName[i]) return false;
		i++;
		while ('0' <= objectName[i] && objectName[i] <= '9') i++;
		if (objectName[i] != ']') return false;
		i++;
		return objectName[i] == ':';
	}
	
	final char[] objectName;
	
	this(char[] objectName)
	{
		if (!isSubset(objectName)) throw new Error("SubsetName");
		this.objectName = objectName;
	}
	
	unittest
	{
		SubsetName sn = new SubsetName("sdef:subset[4][99]:hoge");
		assert(4 == sn.y);
		assert(99 == sn.x);
	}
	
	uint y()
	{
		int b = objectName.find('[');
		int e = objectName.find(']');
		return toUint(objectName[b + 1..e]);
	}
	
	uint x()
	{
		int b = objectName.find("][") + 1;
		int e = b + objectName[b..$].find(']');
		return toUint(objectName[b + 1..e]);
	}
	
	unittest
	{
		SubsetName[] list;
		list ~= new SubsetName("sdef:subset[0][0]:hoge");
		list ~= new SubsetName("sdef:subset[1][0]:hoge");
		list ~= new SubsetName("sdef:subset[1][1]:hoge");
		list ~= new SubsetName("sdef:subset[2][0]:hoge");
		list ~= new SubsetName("sdef:subset[2][2]:hoge");
		assert(2 == maxY(list));
		assert(0 == maxX(list, 0));
		assert(1 == maxX(list, 1));
		assert(2 == maxX(list, 2));
		assert(hasY(list, 2));
		assert(!hasY(list, 3));
	}
	
	private static uint maxY(SubsetName[] list)
	{
		uint result = 0;
		foreach (SubsetName a; list)
		{
			if (a.y > result) result = a.y;
		}
		return result;
	}
	
	private static uint maxX(SubsetName[] list, uint indexY)
	{
		if (!hasY(list, indexY)) throw new Error("SubsetName.maxX");
		uint result = 0;
		foreach (SubsetName a; list)
		{
			if (a.y != indexY) continue;
			if (a.x > result) result = a.x;
		}
		return result;
	}
	
	private static bit hasY(SubsetName[] list, uint indexY)
	{
		foreach (SubsetName a; list)
		{
			if (a.y == indexY) return true;
		}
		return false;
	}
}

class TextureLoader : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		return bb.textureFileName && !bb.texture ? true : false;
	}
	
	void opCall()
	{
		dout.writeLine("texture loading...");
		dout.flush();
		
		bb.texture.length = bb.textureFileName.length;
		for (int y = 0; y < bb.texture.length; y++)
		{
			bb.texture[y].length = bb.textureFileName[y].length;
			for (int x = 0; x < bb.texture[y].length; x++)
			{
				if (!bb.textureFileName[y][x]) continue;
				
				bb.texture[y][x] = exists(bb.textureFileName[y][x]) ?
					new TextureInfo(bb.textureFileName[y][x])
					: TextureInfo.createNullTexture;
				dout.writefln(`texture[%d][%d] width=%d height=%d "%s"`,
					y, x, bb.texture[y][x].width, bb.texture[y][x].height,
					bb.textureFileName[y][x]);
				
				throwIfTextureSizeError(bb.texture[y][x].width, bb.texture[y][x].height);
			}
		}
		
		dout.writeLine("ok");
	}
	
	private void throwIfTextureSizeError(uint width, uint height)
	{
		bit widthIsCorrect = false, heightIsCorrect = false;
		const uint[] list = [ 1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048 ];
		foreach (uint a; list)
		{
			if (a == width) widthIsCorrect = true;
			if (a == height) heightIsCorrect = true;
		}
		if (widthIsCorrect && heightIsCorrect) return;
		throw new Error("TextureSizeError");
	}
}

class TextureCompletion : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		if (!bb.texture) return false;
		for (int y = 0; y < bb.texture.length; y++)
		{
			for (int x = 0; x < bb.texture[y].length; x++)
			{
				if (!bb.texture[y][x]) return true;
			}
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("texture completing...");
		dout.flush();
		
		for (int x = 0; x < bb.texture[0].length; x++)
		{
			if (!bb.texture[0][x])
			{
				bb.texture[0][x] = TextureInfo.createNullTexture();
				dout.writefln("texture[%d][%d] null texture", 0, x);
			}
		}
		
		for (int y = 1; y < bb.texture.length; y++)
		{
			if (!bb.texture[y]) continue;
			if (!bb.texture[y][0])
			{
				bb.texture[y][0] = TextureInfo.createNullTexture();
				dout.writefln("texture[%d][%d] null texture", y, 0);
			}
			if (!bb.texture[y][$ - 1])
			{
				bb.texture[y][$ - 1] = TextureInfo.createNullTexture();
				dout.writefln("texture[%d][%d] set null texture", y, bb.texture[y].length - 1);
			}
		}
		
		for (int y = 1; y < bb.texture.length; y++)
		{
			if (!bb.texture[y]) continue;
			
			for (int x = 1; x < bb.texture[y].length - 1; x++)
			{
				if (bb.texture[y][x]) continue;
				
				uint previousIndex = x - 1;
				
				uint nextIndex = x;
				while (true)
				{
					if (bb.texture[y][++nextIndex]) break;
				}
				
				TextureInfo previousTexture = bb.texture[y][previousIndex];
				TextureInfo nextTexture = bb.texture[y][nextIndex];
				float zero_one = 1.0 / cast(float)(nextIndex - previousIndex);
				
				bb.texture[y][x] = TextureInfo.lerp(previousTexture, nextTexture, zero_one);
				dout.writefln("texture[%d][%d] width=%d height=%d lerp(texture[%d][%d], texture[%d][%d], %f)",
					y, x, bb.texture[y][x].width, bb.texture[y][x].height,
					y, previousIndex, y, nextIndex, zero_one);
			}
		}
		
		dout.writeLine("ok");
	}
}

abstract class ElementExtractor
{
	protected Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	protected float[] getElementList(uint subsetIndexX, uint subsetIndexY)
	{
		MemoryStream result = new MemoryStream();
		MqoIterator i = new MqoIterator(bb.mqo);
		if (subsetIndexY == 0)
		{
			char[] textureFileName = getBaseName(bb.textureFileName[0][subsetIndexX]);
			while (!i.eof)
			{
				if (i.visible && i.textureFileName == textureFileName)
				{
					float[] a = getElement(i);
					result.writeExact(a.ptr, float.sizeof * a.length);
				}
				i++;
			}
		}
		else
		{
			while (!i.eof)
			{
				if (SubsetName.isSubset(i.objectName))
				{
					SubsetName sn = new SubsetName(i.objectName);
					if (sn.x == subsetIndexX && sn.y == subsetIndexY)
					{
						float[] a = getElement(i);
						result.writeExact(a.ptr, float.sizeof * a.length);
					}
				}
				i++;
			}
		}
		return cast(float[])result.data;
	}
	
	protected float[] getElement(MqoIterator i);
}

class PositionExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (!bb.texture) return false;
		for (int y = 0; y < bb.texture.length; y++)
		{
			for (int x = 0; x < bb.texture[y].length; x++)
			{
				if (!bb.texture) return false;
			}
		}
		return !bb.vertexBuffer ? true : false;
	}
	
	void opCall()
	{
		dout.writeLine("position extracting...");
		dout.flush();
		
		bb.vertexBuffer.length = bb.texture.length;
		for (int y = 0; y < bb.vertexBuffer.length; y++)
		{
			bb.vertexBuffer[y].length = bb.texture[y].length;
			for (int x = 0; x < bb.vertexBuffer[y].length; x++)
			{
				float[] list = getElementList(x, y);
				if (list.length == 0) continue;
				
				bb.vertexBuffer[y][x] = new VertexBufferInfo();
				bb.vertexBuffer[y][x].sizeOfVertex += 3;
				bb.vertexBuffer[y][x].add(list);
				dout.writefln("VertexBuffer[%d][%d] position floatLength=%d",
					y, x, list.length);
			}
		}
		
		dout.writeLine("ok");
	}
	
	protected override float[] getElement(MqoIterator i)
	{
		float[] result;
		Vector p = i.position;
		result ~= p.x;
		result ~= p.y;
		result ~= p.z;
		return result;
	}
}

class MqoIterator
{
	private Mqo mqo;
	private uint objectIndex = 0, faceIndex = 0, vIndex = 0;
	private bit _eof = false;
	this(Mqo mqo) { this.mqo = mqo; }
	
	// 高速化
	void opPostInc()
	{
		if (_eof) throw new Error("MqoIterator.opPostInc");
		vIndex++;
		if (face_vlength == 2 && vIndex < 3) return;
		if (face_vlength == 3 && vIndex < 3) return;
		if (face_vlength == 4 && vIndex < 6) return;
		vIndex = 0;
		faceIndex++;
		if (faceIndex < mqo.object[objectIndex].face.length) return;
		faceIndex = 0;
		while (true)
		{
			objectIndex++;
			if (mqo.object.length == objectIndex)
			{
				_eof = true;
				return;
			}
			else if (mqo.object[objectIndex].face.length != 0) return;
		}
	}
	
	char[] toString()
	{
		return format("MqoIterator objectIndex=%d faceIndex=%d vIndex=%d",
			objectIndex, faceIndex, vIndex);
	}
	
	bit eof() { return _eof; }
	
	char[] textureFileName()
	{
		if (mqo.material == null) return "";
		return mqo.material[face_M].tex;
	}
	
	bit visible()
	{
		return 0 != mqo.object[objectIndex].visible ? true : false;
	}
	
	char[] objectName()
	{
		return mqo.object[objectIndex].name;
	}
	
	
	private int face_vlength()
	{
		return mqo.object[objectIndex].face[faceIndex].vlength;
	}
	
	private int face_M()
	{
		return mqo.object[objectIndex].face[faceIndex].M;
	}
	
	private int face_V(int index)
	{
		return mqo.object[objectIndex].face[faceIndex].V[index];
	}
	
	private int face_UV_length()
	{
		return mqo.object[objectIndex].face[faceIndex].UV.length;
	}
	
	private _Mqo.Vector2 face_UV(int index)
	{
		return mqo.object[objectIndex].face[faceIndex].UV[index];
	}
	
	
	uint vertexIndex()
	{
		return face_V(toFacevIndex(face_vlength, vIndex));
	}
	
	private static uint toFacevIndex(uint faceVlength, uint vIndex)
	out (result)
	{
		assert(0 <= result && result < 4);
	}
	body
	{
		if (faceVlength == 2)
		{
			const uint[] a = [ 0, 1, 0 ];
			return a[vIndex];
		}
		else if (faceVlength == 3)
		{
			return vIndex;
		}
		else if (faceVlength == 4)
		{
			const uint[] a = [ 0, 1, 2, 0, 2, 3 ];
			return a[vIndex];
		}
		throw new Error("toFacevIndex");
	}
	
	Vector position()
	{
		_Mqo.Vector3 v3 = mqo.object[objectIndex].vertex[vertexIndex];
		return Vector.create(v3.x, v3.y, v3.z);
	}
	
	Vector color()
	{
		if (mqo.material == null) return vector(1.0, 1.0, 1.0, 1.0);
		_Mqo.Vector4 v4 = mqo.material[face_M].col;
		return Vector.create(v4.x, v4.y, v4.z, v4.w);
	}
	
	Vector flatNormal()
	{
		return flatNormal(
			mqo.object[objectIndex],
			mqo.object[objectIndex].face[faceIndex]
		);
	}
	
	static Vector flatNormal(_Mqo.Object obj, _Mqo.Face face)
	{
		Vector a = toVector(obj.vertex[face.V[toFacevIndex(face.vlength, 0)]]);
		Vector b = toVector(obj.vertex[face.V[toFacevIndex(face.vlength, 1)]]);
		Vector c = toVector(obj.vertex[face.V[toFacevIndex(face.vlength, 2)]]);
		Vector result = cross(b - a, c - a);
		return length(result) != 0.0 ? normalize(result) : vector(0, 0, 0);
	}
	
	private static Vector toVector(_Mqo.Vector3 a)
	{
		return Vector.create(a.x, a.y, a.z);
	}
	
	Vector texCoord()
	{
		if (face_UV_length == 0) return Vector.create(0.0, 0.0);
		Mqo.Vector2 v2 = face_UV(toFacevIndex(face_vlength, vIndex));
		return Vector.create(v2.x, v2.y);
	}
}

class ColorExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (!bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 3) return true;
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("color extracting...");
		dout.flush();
		
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			vb.sizeOfVertex += 4;
			float[] list = getElementList(x, y);
			vb.add(list);
			dout.writefln("VertexBuffer[%d][%d] color floatLength=%d", y, x, list.length);
		}
		
		dout.writeLine("ok");
	}
	
	protected override float[] getElement(MqoIterator i)
	{
		float[] result;
		Vector c = i.color;
		result ~= c.x;
		result ~= c.y;
		result ~= c.z;
		result ~= c.w;
		return result;
	}
}

class FlatNormalExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (!bb.flatvertex || !bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 7) return true;
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("flat normal extracting...");
		dout.flush();
		
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			vb.sizeOfVertex += 3;
			float[] list = getElementList(x, y);
			vb.add(list);
			dout.writefln("VertexBuffer[%d][%d] flat normal floatLength=%d", y, x, list.length);
		}
		
		dout.writeLine("ok");
	}
	
	protected override float[] getElement(MqoIterator i)
	{
		float[] result;
		Vector c = i.flatNormal;
		result ~= c.x;
		result ~= c.y;
		result ~= c.z;
		return result;
	}
}

class SmoothNormalExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (bb.flatvertex || !bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 7) return true;
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("smooth normal extracting...");
		dout.flush();
		
		initializeSmoothNormalMap();
		
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			vb.sizeOfVertex += 3;
			float[] list = getElementList(x, y);
			vb.add(list);
			dout.writefln("VertexBuffer[%d][%d] smooth normal floatLength=%d", y, x, list.length);
		}
		
		dout.writeLine("ok");
	}
	
	private Vector[][char[]] smoothNormalMap;
	
	private void initializeSmoothNormalMap()
	{
		foreach (Mqo.Object obj; bb.mqo.object)
		{
			smoothNormalMap[obj.name] = computeSmoothNormalList(obj);
		}
	}
	
	protected override float[] getElement(MqoIterator i)
	{
		float[] result;
		Vector n = (smoothNormalMap[i.objectName])[i.vertexIndex];
		bool nIsZero = length(n) == 0.0;
		bool flatIsZero = length(i.flatNormal) == 0.0;
		if (nIsZero && flatIsZero) // 両方とも存在しない困った状態
		{
			n = vector(0.0, 1.0, 0.0);
		}
		else if (nIsZero && !flatIsZero) n = i.flatNormal;
		else if (!nIsZero && !flatIsZero && dot(n, i.flatNormal) < 0.0) n = -n;
		// !nIsZero && flatIsZeroの場合はそのまま
		result ~= n.x;
		result ~= n.y;
		result ~= n.z;
		return result;
	}
	
	private static Vector[] computeSmoothNormalList(Mqo.Object obj)
	{
		Vector[] result = new Vector[obj.vertex.length];
		result[] = Vector.create(0, 0, 0);
		foreach (Mqo.Face face; obj.face) // トータルを計算する
		{
			if (face.vlength == 2) continue;
			assert(face.vlength == 3 || face.vlength == 4);
			
			Vector v = MqoIterator.flatNormal(obj, face);
			void addSmooth(inout Vector r, inout Vector a) // 120度以上はスムースとしない
			{
				if (length(r) != 0.0 && dot(normalize(r), a) < -0.5) return;
				r += a;
			}
			addSmooth(result[face.V[0]], v);
			addSmooth(result[face.V[1]], v);
			addSmooth(result[face.V[2]], v);
			if (face.vlength == 4) addSmooth(result[face.V[3]], v);
		}
		foreach (inout Vector a; result) // 全て正規化
		{
			if (length(a) != 0.0) a = normalize(a); // 0のときは通常の値を使う(getElement)
		}
		return result;
	}
}

class TexCoordExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (!bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 10) return true;
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("tex coord extracting...");
		dout.flush();
		
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			vb.sizeOfVertex += 2;
			float[] list = getElementList(x, y);
			vb.add(list);
			dout.writefln("VertexBuffer[%d][%d] tex coord floatLength=%d", y, x, list.length);
		}
		
		dout.writeLine("ok");
	}
	
	protected override float[] getElement(MqoIterator i)
	{
		float[] result;
		Vector t = i.texCoord;
		result ~= t.x;
		result ~= t.y;
		return result;
	}
}

class VertexBufferCompletion : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		if (!bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 12 && !bb.skeleton) break;
			if (vb.sizeOfVertex == 15 && bb.skeleton) break;
			return false;
		}
		for (int y = 0; y < bb.vertexBuffer.length; y++)
		{
			for (int x = 0; x < bb.vertexBuffer[y].length; x++)
			{
				if (!bb.vertexBuffer[y][x]) return true;
			}
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("vertex buffer completing...");
		dout.flush();
		
		for (int x = 0; x < bb.vertexBuffer[0].length; x++)
		{
			if (!bb.vertexBuffer[0][x]) bb.vertexBuffer[0][x] = new VertexBufferInfo();
		}
		
		for (int y = 1; y < bb.vertexBuffer.length; y++)
		{
			for (int x = 1; x < bb.vertexBuffer[y].length; x++)
			{
				if (bb.vertexBuffer[y][x]) continue;
				
				uint previousIndex = x - 1;
				
				uint nextIndex = x;
				while (true)
				{
					if (bb.vertexBuffer[y][++nextIndex]) break;
				}
				
				VertexBufferInfo previousVb = bb.vertexBuffer[y][previousIndex];
				VertexBufferInfo nextVb = bb.vertexBuffer[y][nextIndex];
				if (!previousVb || !nextVb) throw new Error("VertexBufferCompletion");
				
				float zero_one = 1.0 / cast(float)(nextIndex - previousIndex);
				
				bb.vertexBuffer[y][x] = VertexBufferInfo.lerp(previousVb, nextVb, zero_one);
				dout.writefln("vertexBuffer[%d][%d] sizeOfVertex=%d vertexLength=%d lerp(vertexBuffer[%d][%d], vertexBuffer[%d][%d], %f)",
					y, x, bb.vertexBuffer[y][x].sizeOfVertex, bb.vertexBuffer[y][x].vertexLength,
					y, previousIndex, y, nextIndex, zero_one);
			}
		}
		
		dout.writeLine("ok");
	}
}

class ZipGenerator : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		if (bb.result || !bb.texture || !bb.vertexBuffer) return false;
		if (bb.mkm && bb.skeleton && !bb.motionMatrix) return false;
		for (int y = 0; y < bb.texture.length; y++)
		{
			for (int x = 0; x < bb.texture[y].length; x++)
			{
				if (!bb.texture[y][x] || !bb.vertexBuffer[y][x]) return false;
			}
		}
		return true;
	}
	
	void opCall()
	{
		dout.writeString("zip generating...");
		dout.flush();
		
		NanamiFile encoder = new NanamiFile();
		
		for (int y = 0; y < bb.vertexBuffer.length; y++)
		{
			for (int x = 0; x < bb.vertexBuffer[y].length; x++)
			{
				char[] yx = format("[%d][%d]", y, x);
				
				VertexBufferInfo vb = bb.vertexBuffer[y][x];
				encoder.addVertexBuffer(x, y, vb.sizeOfVertex, vb.vertexLength, vb.buffer);
				
				TextureInfo texture = bb.texture[y][x];
				encoder.addTexture(x, y, texture.width, texture.height, texture.rgba);
			}
		}
		
		if (bb.skeleton)
		{
			encoder.add("skeleton_bones", bb.skeleton.extractBone());
			encoder.add("skeleton_relations", bb.skeleton.extractRelation());
		}
		
		if (bb.motionMatrix)
		{
			foreach (int i, MotionMatrix a; bb.motionMatrix)
			{
				encoder.add(format("motion_matrix[%d]", i), a.toData());
			}
		}
		
		bb.result = encoder.build;
		
		dout.writeLine("ok");
	}
}

class MatrixIndexExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (!bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 12 && bb.skeleton) return true;
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("matrix index extracting...");
		dout.flush();
		
		foreach (Mqo.Object a; bb.mqo.object)
		{
			anchorMap[a] = isSdef(a) ? createRelationalAnchorList(a) : null;
		}
		
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			vb.sizeOfVertex += 2;
			float[] list = getElementList(x, y);
			vb.add(list);
			dout.writefln("VertexBuffer[%d][%d] matrix index floatLength=%d", y, x, list.length);
		}
		
		dout.writeLine("ok");
	}
	
	protected override float[] getElement(MqoIterator i)
	{
		float[] result;
		Vector mi = getMatrixIndex(anchorMap[i.mqo.object[i.objectIndex]], i.position);
		result ~= mi.x;
		result ~= mi.y;
		return result;
	}
	
	private Skeleton skeleton() { return bb.skeleton; }
	private Mqo.Object[] object() { return bb.mqo.object; }
	private Mqo.Material[] material() { return bb.mqo.material; }
	
	private Anchor[][Mqo.Object] anchorMap;
	
	private Vector getMatrixIndex(Anchor[] anchorList, Vector p)
	{
		return getOwnerBoneIndex(getOwnerBoneIndexList(anchorList, p), p);
	}
	
	// pの所有ボーン候補リスト
	private uint[] getOwnerBoneIndexList(Anchor[] anchorList, Vector p)
	{
		uint[] result;
		foreach (Anchor b; anchorList)
		{
			if (!b.has(p)) continue;
			int boneIndex = getBoneIndex(b.materialName, p);
			if (boneIndex == -1)
			{
				//throw new Error("not found " ~ materialName ~ " from bone material name");
				continue;
			}
			result ~= cast(uint)boneIndex;
		}
		return result;
	}
	
	private Vector getOwnerBoneIndex(uint[] ownerBoneIndexList, Vector p)
	{
		uint first, second;
		if (ownerBoneIndexList.length >= 2)
		{
			float[] distanceList = new float[ownerBoneIndexList.length];
			for (int i = 0; i < distanceList.length; i++)
			{
				distanceList[i] = Bone.distanceFromCenter(
					skeleton.boneList[ownerBoneIndexList[i]], p
				);
			}
			first = ownerBoneIndexList[firstIndexOf(distanceList)];
			second = ownerBoneIndexList[secondIndexOf(distanceList)];
		}
		else if (ownerBoneIndexList.length == 1)
		{
			first = ownerBoneIndexList[0];
			second = ownerBoneIndexList[0];
		}
		else
		{
			first = second = getNearBoneIndex(p);
		}
		return Vector.create(cast(float)first, cast(float)second);
	}
	
	
	private bit hasBone()
	{
		foreach (Mqo.Object a; object)
		{
			if ("bone:".length <= a.name.length && a.name[0..5] == "bone:") return true;
		}
		return false;
	}
	
	
	// objとつながっているAnchor[]を返す
	private Anchor[] createRelationalAnchorList(Mqo.Object obj)
	in
	{
		assert(isSdef(obj));
	}
	body
	{
		Anchor[] result;
		foreach (Mqo.Object a; object)
		{
			if (Anchor.isAnchor(a)
				&& removeSdef(obj.name) == Anchor.extractRelationalObjectName(a))
			{
				result ~= Anchor.createList(material, a);
			}
		}
		return result;
	}
	
	private static bit isSdef(Mqo.Object obj) { return isSdef(obj.name); }
	unittest
	{
		assert(isSdef("sdef:hoge"));
		assert(!isSdef("aaa:hoge"));
	}
	private static bit isSdef(char[] name)
	{
		return "sdef:".length <= name.length && "sdef:" == name[0..5] ? true : false;
	}
	
	unittest
	{
		assert("hoge" == removeSdef("sdef:hoge"));
		assert("aaa" == removeSdef("aaa"));
	}
	private static char[] removeSdef(char[] a)
	{
		if (a.length <= 5) return a;
		if (a[0..5] == "sdef:") return a[5..a.length];
		return a;
	}
	
	private int getBoneIndex(char[] materialName, Vector position)
	{
		int result = indexOfBoneName(skeleton, getBoneName(materialName, position));
		if (result == -1) result = indexOfBoneMaterialName(skeleton, materialName);
		return result;
	}
	private int indexOfBoneName(Skeleton skeleton, char[] name)
	{
		for (int i = 0; i < skeleton.boneList.length; i++)
		{
			if (skeleton.boneList[i].name == name) return i;
		}
		return -1;
	}
	private int indexOfBoneMaterialName(Skeleton skeleton, char[] materialName)
	{
		for (int i = 0; i < skeleton.boneList.length; i++)
		{
			if (skeleton.boneList[i].materialName == materialName) return i;
		}
		return -1;
	}
	private static char[] getBoneName(char[] materialName, Vector position) // []の中のR, Lを入れる
	{
		return insertRorL(materialName, position.x);
	}
	private static char[] insertRorL(char[] a, float x)
	{
		if (-1 == find(a, "[]")) return a;
		if (x >= 0) return replace(a, "[]", "[L]");
		else return replace(a, "[]", "[R]");
	}
	
	
	unittest
	{
		float[] distanceList;
		distanceList ~= 2.0;
		distanceList ~= 0.0;
		distanceList ~= 0.5;
		distanceList ~= 1.0;
		assert(1 == firstIndexOf(distanceList));
		assert(2 == secondIndexOf(distanceList));
	}
	private static ubyte firstIndexOf(float[] distanceList)
	in
	{
		assert(1 <= distanceList.length);
	}
	body
	{
		ubyte result = 0;
		for (int i = 0; i < distanceList.length; i++)
		{
			if (distanceList[result] > distanceList[i]) result = cast(ubyte)i;
		}
		return result;
	}
	private static ubyte secondIndexOf(float[] distanceList)
	in
	{
		assert(2 <= distanceList.length);
	}
	body
	{
		ubyte firstIndex = firstIndexOf(distanceList);
		ubyte result = firstIndex == 0 ? 1 : 0;
		for (int i = 0; i < distanceList.length; i++)
		{
			if (i == firstIndex) continue;
			if (distanceList[result] > distanceList[i]) result = cast(ubyte)i;
		}
		return result;
	}
	
	// カプセル、
	uint getNearBoneIndex(Vector p)
	{
		float[] distanceList = new float[skeleton.boneList.length];
		for (int i = 0; i < distanceList.length; i++)
		{
			distanceList[i] = Bone.distanceFromLineSegment(skeleton.boneList[i], p);
		}
		
		uint result;
		bit hasInner = false;
		for (int i = 0; i < distanceList.length; i++)
		{
			bit isInner = distanceList[i] < skeleton.boneList[i].height;
			if (!isInner) continue;
			if (!hasInner)
			{
				hasInner = true;
				result = i;
			}
			else
			{
				if (distanceList[i] < distanceList[result]) result = i;
			}
		}
		
		return hasInner ? result : firstIndexOf(distanceList);
	}
}

class WeightExtractor : ElementExtractor, KnowledgeSource
{
	this(Blackboard bb) { super(bb); }
	
	bit condition()
	{
		if (!bb.vertexBuffer) return false;
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			if (vb.sizeOfVertex == 14) return true;
		}
		return false;
	}
	
	void opCall()
	{
		dout.writeLine("weight extracting...");
		dout.flush();
		
		foreach (VertexBufferInfo vb, uint x, uint y; bb)
		{
			currentX = x, currentY = y, miIndex = 0;
			float[] list = getElementList(x, y);
			vb.sizeOfVertex += 1; // getElementのvb.vertexLengthに影響がでるのでここに書く
			vb.add(list);
			dout.writefln("VertexBuffer[%d][%d] weight floatLength=%d", y, x, list.length);
		}
		
		dout.writeLine("ok");
	}
	
	private uint currentX, currentY, miIndex;
	
	protected override float[] getElement(MqoIterator i)
	{
		VertexBufferInfo vb = bb.vertexBuffer[currentY][currentX];
		float x = vb.buffer[12 * vb.vertexLength + miIndex];
		float y = vb.buffer[12 * vb.vertexLength + miIndex + 1];
		miIndex += 2;
		
		float[] result;
		result ~= computeWeight(i.position, Vector.create(x, y));
		return result;
	}
	
	private Skeleton skeleton() { return bb.skeleton; }
	
	private float computeWeight(Vector p, Vector matrixIndex)
	{
		/* 古いほう
		uint firstBoneIndex = cast(uint)matrixIndex.x;
		uint secondBoneIndex = cast(uint)matrixIndex.y;
		float firstDistance = Bone.distanceFromCenter(skeleton.boneList[firstBoneIndex], p);
		float secondDistance = Bone.distanceFromCenter(skeleton.boneList[secondBoneIndex], p);
		assert(0.0 != firstDistance + secondDistance);
		return secondDistance / (firstDistance + secondDistance);
		//*/
		
		if (matrixIndex.x == matrixIndex.y) return 0.0;
		
		Bone firstBone = skeleton.boneList[cast(uint)matrixIndex.x];
		Bone secondBone = skeleton.boneList[cast(uint)matrixIndex.y];
		bit parentIsFirst = distance(firstBone.endPosition, secondBone.startPosition)
			< distance(secondBone.endPosition, firstBone.startPosition);
		Bone parentBone = parentIsFirst ? firstBone : secondBone;
		Bone childBone = parentIsFirst ? secondBone : firstBone;
		Vector indirectSpherePosition = parentBone.endPosition;
		float indirectSphereRadius = childBone.height * 1.0; //
		Vector parentBoneDirection = normalize(parentBone.endPosition - parentBone.startPosition);
		Vector childBoneDirection = normalize(childBone.endPosition - childBone.startPosition);
		Vector parentIntersection = indirectSpherePosition - parentBoneDirection * indirectSphereRadius;
		Vector childIntersection = indirectSpherePosition + childBoneDirection * indirectSphereRadius;
		
		float parentDistance = distance(parentIntersection, p);
		float childDistance = distance(childIntersection, p);
		//parentDistance = pow(parentDistance, 0.5); //
		//childDistance = pow(childDistance, 0.5); //
		assert(0.0 != parentDistance + childDistance);
		float result = childDistance / (parentDistance + childDistance);
		return parentIsFirst ? result : 1.0 - result;
	}
	
	private static float min(float a, float b)
	{
		return a < b ? a : b;
	}
}

class MotionMatrixExtractor : KnowledgeSource
{
	private Blackboard bb;
	this(Blackboard bb) { this.bb = bb; }
	
	bit condition()
	{
		return bb.mkm && bb.skeleton && !bb.motionMatrix ? true : false;
	}
	
	void opCall()
	{
		dout.writeLine("motion matrix list extracting...");
		dout.flush();
		
		MotionMatrixFormat[] mmf = createMotionMatrixFormatList(bb.skeleton, bb.mkm);
		bb.motionMatrix.length = mmf.length;
		for (int i = 0; i < mmf.length; i++)
		{
			bb.motionMatrix[i] = new MotionMatrix(mmf[i].boneLength, mmf[i].frameLength);
			for (int fi = 0; fi < mmf[i].frameLength; fi++)
			{
				for (int bi = 0; bi < mmf[i].boneLength; bi++)
				{
					bb.motionMatrix[i].setLrt(bi, fi,
						mmf[i].getLookAt(fi, bi),
						mmf[i].getRotation(fi, bi),
						mmf[i].getTranslation(fi, bi)
						);
				}
			}
			
			dout.writefln("motionMatrix[%d] boneLength=%d frameLength%d",
				i, bb.motionMatrix[i].boneLength, bb.motionMatrix[i].frameLength);
		}
		
		dout.writeLine("ok");
	}
}
