﻿module mkm;
private import std.stream;
private import std.string;
private import std.file;
private import std.math;


enum MkmTokenType
{
	INTEGER = 200, FLOAT, STRING,
	Mikoto, Motion, Ver,
	name, endframe, loop, Vector, Quaternion,
	_class, member, curve,
	Eof,
}
static assert(34 == '\"');
static assert(40 == '(');
static assert(41 == ')');
static assert(61 == '=');
static assert(123 == '{');
static assert(125 == '}');
static assert(126 <= MkmTokenType.min);

interface _MkmLexicalAnalyzer
{
	//MkmLexicalAnalyzer.this(Stream src)
	MkmTokenType read();
	int intBuffer();
	float floatBuffer();
	char[] stringBuffer();
}

interface _Mkm
{
	struct float3 { float x, y, z; }
	struct float4 { float a, b, c, d; }
	struct Motion
	{
		char[] name;
		int endframe, loop;
		Vector vector;
		Quaternion[] quaternion;
	}
	struct Vector
	{
		int index;
		char[] name, _class, member, curve;
		IntAndFloat3[] keyFrame;
	}
	struct IntAndFloat3
	{
		int keyFrameNumber;
		float3 position;
	}
	struct Quaternion
	{
		int index;
		char[] name, _class, member, curve;
		IntAndFloat4[] keyFrame;
	}
	struct IntAndFloat4
	{
		int keyFrameNumber;
		float4 rotation;
	}
	//Mkm.this(Stream src)
	Motion[] motion();
}

class MkmLexicalAnalyzer : _MkmLexicalAnalyzer
{
	private final Stream src;
	private final MkmTokenType[char[]] tokenMap;
	
	unittest // tokenMapのvalueは一意
	{
		MkmLexicalAnalyzer a = new MkmLexicalAnalyzer(new MemoryStream(" "));
		bit[MkmTokenType] bitMap;
		foreach (MkmTokenType b; a.tokenMap.values)
		{
			assert(!(b in bitMap));
			bitMap[b] = true;
		}
	}
	
	alias MkmTokenType Mtt;
	
	this(Stream src)
	{
		this.src = src;
		
		tokenMap["INTEGER"] = Mtt.INTEGER;
		tokenMap["FLOAT"] = Mtt.FLOAT;
		tokenMap["STRING"] = Mtt.STRING;
		tokenMap["Mikoto"] = Mtt.Mikoto;
		tokenMap["Motion"] = Mtt.Motion;
		tokenMap["Ver"] = Mtt.Ver;
		tokenMap["name"] = Mtt.name;
		tokenMap["endframe"] = Mtt.endframe;
		tokenMap["loop"] = Mtt.loop;
		tokenMap["Vector"] = Mtt.Vector;
		tokenMap["Quaternion"] = Mtt.Quaternion;
		tokenMap["class"] = Mtt._class;
		tokenMap["member"] = Mtt.member;
		tokenMap["curve"] = Mtt.curve;
		tokenMap["Eof"] = Mtt.Eof;
		
		current = src.getc();
	}
	
	unittest
	{
		Stream src = new MemoryStream(
			"0 1.0 \"2\"\n"
			"Motion { 1 (0.000000 -2.12) = }\n"
			"Eof\n"
			);
		MkmLexicalAnalyzer a = new MkmLexicalAnalyzer(src);
		alias MkmTokenType Mtt;
		assert(Mtt.INTEGER == a.read());
		assert(0 == a.intBuffer);
		assert(Mtt.FLOAT == a.read());
		assert(1.0 == a.floatBuffer);
		assert(Mtt.STRING == a.read());
		assert("2" == a.stringBuffer);
		assert(Mtt.Motion == a.read());
		assert('{' == a.read());
		assert(Mtt.INTEGER == a.read());
		assert(1 == a.intBuffer);
		assert('(' == a.read());
		assert(Mtt.FLOAT == a.read());
		assert(0.0 == a.floatBuffer);
		assert(Mtt.FLOAT == a.read());
		assert(fabs(-2.12 - a.floatBuffer) < 0.000001);
		assert(')' == a.read());
		assert('=' == a.read());
		assert('}' == a.read());
		assert(Mtt.Eof == a.read());
	}
	
	private char current = ' ';
	MkmTokenType read()
	{
		token = "";
		while (isWhiteSpace(current)) current = src.getc();
		
		if (isDigit(current) || current == '-')
		{
			while (true)
			{
				token ~= current;
				current = src.getc();
				if (!isDigit(current) && '.' != current)
				{
					return isFloat(token) ? Mtt.FLOAT : Mtt.INTEGER;
				}
				assert(!src.eof);
			}
		}
		else if ('\"' == current)
		{
			while (true)
			{
				current = src.getc();
				if ('\"' == current)
				{
					current = src.getc();
					return Mtt.STRING;
				}
				assert(!src.eof);
				token ~= current;
			}
		}
		else if ('{' == current || '}' == current
			|| '(' == current || ')' == current
			|| '=' == current)
		{
			char result = current;
			token ~= current;
			current = src.getc();
			return cast(Mtt)result;
		}
		
		while (true)
		{
			token ~= current;
			current = src.getc();
			if (isWhiteSpace(current) || '(' == current)
			{
				if (!(token in tokenMap))
				{
					throw new Error("unknown token: " ~ token);
				}
				return tokenMap[token];
			}
			assert(!src.eof);
		}
		
		throw new Error("a");
	}
	
	unittest
	{
		assert(isFloat("1.0"));
		assert(!isFloat("1"));
		assert(isWhiteSpace(10));
	}
	
	static assert(0x0D == '\r');
	static assert(0x0A == '\n');
	
	static bit isWhiteSpace(char a) { return -1 != find(whitespace, a) ? true : false; }
	static bit isDigit(char a) { return -1 != find(digits, a) ? true : false; }
	static bit isFloat(char[] a) { return -1 != find(a, '.') ? true : false; }
	
	private char[] token = "";
	int intBuffer() { return cast(int)atoi(token); }
	float floatBuffer() { return atof(token); }
	char[] stringBuffer() { return token; }
}

// Mikoto Motion Ver 2 だけ
// 正確な仕様がわからないのでMotionがVector, Quaternion...と続くものだけ
class Mkm : _Mkm
{
	Motion[] motion() { return _motion; }
	private Motion[] _motion;
	
	private MkmLexicalAnalyzer lex;
	private alias MkmTokenType Mtt;
	
	this(Stream src)
	{
		lex = new MkmLexicalAnalyzer(src);
		readHeader();
		readMotion();
	}
	
	private void readHeader()
	{
		read(Mtt.Mikoto);
		read(Mtt.Motion);
		read(Mtt.Ver);
		if (2 != readInt()) throw new Error("not supported file");
	}
	
	private void readMotion()
	{
		Mtt type = lex.read();
		assert(Mtt.Motion == type);
		while (type == Mtt.Motion)
		{
			int index = 0;
			Motion a;
			read('{');
			read(Mtt.name); read('='); a.name = readString();
			read(Mtt.endframe); read('='); a.endframe = readInt();
			read(Mtt.loop); read('='); a.loop = readInt();
			while (true)
			{
				type = lex.read();
				if (type == Mtt.Vector) readVector(a, index++);
				else if (type == Mtt.Quaternion) readQuaternion(a, index++);
				else break;
			}
			assert('}' == type);
			_motion ~= a;
			type = lex.read();
		}
		assert(Mtt.Eof == type);
	}
	
	private void readVector(inout Motion a, int index)
	{
		a.vector.index = index;
		read('{');
		read(Mtt.name); read('='); a.vector.name = readString();
		read(Mtt._class); read('='); a.vector._class = readString();
		read(Mtt.member); read('='); a.vector.member = readString();
		read(Mtt.curve); read('='); a.vector.curve = readString();
		Mtt type = lex.read();
		while (type != '}')
		{
			IntAndFloat3 b;
			b.keyFrameNumber = lex.intBuffer;
			read('(');
			b.position.x = readFloat();
			b.position.y = readFloat();
			b.position.z = readFloat();
			read(')');
			a.vector.keyFrame ~= b;
			type = lex.read();
		}
	}
	
	private void readQuaternion(inout Motion a, int index)
	{
		Quaternion b;
		b.index = index;
		read('{');
		read(Mtt.name); read('='); b.name = readString();
		read(Mtt._class); read('='); b._class = readString();
		read(Mtt.member); read('='); b.member = readString();
		read(Mtt.curve); read('='); b.curve = readString();
		Mtt type = lex.read();
		while (type != '}')
		{
			IntAndFloat4 c;
			c.keyFrameNumber = lex.intBuffer;
			read('(');
			c.rotation.a = readFloat();
			c.rotation.b = readFloat();
			c.rotation.c = readFloat();
			c.rotation.d = readFloat();
			read(')');
			b.keyFrame ~= c;
			type = lex.read();
		}
		assert('}' == type);
		a.quaternion ~= b;
	}
	
	char[] toString() // mikoto側がVectorとQuaternionを同じリストに入れているようなので
	{
		MemoryStream result = new MemoryStream();
		result.writeLine("Mikoto Motion Ver 2");
		foreach (Motion a; _motion)
		{
			result.writeLine("Motion {");
			result.writefln("\tname = \"%s\"", a.name);
			result.writefln("\tendframe = %d", a.endframe);
			result.writefln("\tloop = %d", a.loop);
			int vectorAndQuaternionLength = 1 + a.quaternion.length;
			for (int i = 0; i < vectorAndQuaternionLength; i++)
			{
				if (a.vector.index == i)
				{
					result.writeLine("\tVector {");
					result.writefln("\t\tname = \"%s\"", a.vector.name);
					result.writefln("\t\tclass = \"%s\"", a.vector._class);
					result.writefln("\t\tmember = \"%s\"", a.vector.member);
					result.writefln("\t\tcurve = \"%s\"", a.vector.curve);
					foreach (IntAndFloat3 b; a.vector.keyFrame)
					{
						result.writefln("\t\t%d (%f %f %f)",
							b.keyFrameNumber, b.position.x, b.position.y, b.position.z);
					}
					result.writefln("\t}");
				}
				else // quaternion
				{
					int qIndex = 0;
					for (int j = 0; j < a.quaternion.length; j++)
					{
						if (a.quaternion[j].index == i)
						{
							qIndex = j;
							break;
						}
					}
					
					Quaternion b = a.quaternion[qIndex];
					result.writefln("\tQuaternion {");
					result.writefln("\t\tname = \"%s\"", b.name);
					result.writefln("\t\tclass = \"%s\"", b._class);
					result.writefln("\t\tmember = \"%s\"", b.member);
					result.writefln("\t\tcurve = \"%s\"", b.curve);
					foreach (IntAndFloat4 c; b.keyFrame)
					{
						result.writefln("\t\t%d (%f %f %f %f)",
							c.keyFrameNumber,
							c.rotation.a, c.rotation.b,
							c.rotation.c, c.rotation.d);
					}
					result.writeLine("\t}");
				}
			}
			result.writeLine("}");
		}
		result.writeLine("Eof");
		return result.toString();
	}
	
	
private:
	class SyntaxError : Error
	{
		this(Mtt a, char[] b)
		{
			super("syntax error: " ~ .toString(cast(int)a) ~ " " ~ b);
		}
	}
	void read(Mtt a) { if (a != lex.read()) throw new SyntaxError(a, lex.stringBuffer); }
	void read(char a) { read(cast(Mtt)a); }
	float readFloat()
	{
		if (Mtt.FLOAT != lex.read()) throw new SyntaxError(Mtt.FLOAT, lex.stringBuffer);
		return lex.floatBuffer;
	}
	int readInt()
	{
		if (Mtt.INTEGER != lex.read()) throw new SyntaxError(Mtt.INTEGER, lex.stringBuffer);
		return lex.intBuffer;
	}
	char[] readString()
	{
		if (Mtt.STRING != lex.read()) throw new SyntaxError(Mtt.STRING, lex.stringBuffer);
		return lex.stringBuffer;
	}
}
