﻿module mqo;
private import std.stream;
private import std.string;
private import std.file;

// 新しいものはtokenMapに追加すること
enum MqoTokenType
{
	INTEGER = 200, FLOAT, STRING,
	Metasequoia, Document, Format, Text, Ver,
	Scene, Material, Object,
	pos, lookat, head, pich, ortho, zoom2, amb,
	shader, vcol, col, dif, /*amb,*/ emi, spc, power, tex, aplane, bump,
	proj_type, proj_pos, proj_scale, proj_angle,
	patch, segment, visible, locking, shading, facet, color, color_type,
	mirror, mirror_axis, mirror_dis, lathe, lathe_axis, lathe_seg, vertex, face,
	V, M, UV,
	Blob, blob1,
	sphere, cylinder, box, torus, cone, pyramid, tube, roundcube, roundcylinder,
	weight, /*pos,*/ scale, angle, /*power,*/
	Eof,
	depth, folding, rotation, translation,
	BackImage, pers,
}
static assert(34 == '\"');
static assert(40 == '(');
static assert(41 == ')');
static assert(123 == '{');
static assert(125 == '}');
static assert(126 <= MqoTokenType.min);

interface _MqoLexicalAnalyzer
{
	//MqoLexicalAnalyzer.this(Stream src)
	MqoTokenType read();
	int intBuffer();
	float floatBuffer();
	char[] stringBuffer();
}

interface _Mqo
{
	struct Vector2 { float x, y; }
	struct Vector3 { float x, y, z; }
	struct Vector4 { float x, y, z, w; }
	struct Scene
	{
		Vector3 pos, lookat;
		float head, pich;
		int ortho;
		float zoom2;
		Vector3 amb;
	}
	struct Material
	{
		char[] name;
		int shader, vcol;
		Vector4 col;
		float dif, amb, emi, spc, power;
		char[] tex, aplane, bump;
		int proj_type;
		Vector3 proj_pos, proj_scale, proj_angle;
	}
	struct Object
	{
		char[] name;
		int depth, folding;
		Vector3 scale, rotation, translation;
		int patch, segment, visible, locking, shading;
		float facet;
		Vector3 color;
		int color_type, mirror, mirror_axis;
		float mirror_dis;
		int lathe, lathe_axis, lathe_seg;
		Vector3[] vertex;
		Face[] face;
	}
	struct Face
	{
		int vlength; // V.length, UV.length, 2 or 3 or 4
		int[] V;
		int M;
		Vector2[] UV;
	}
	struct Blob
	{
		char[] name; // どれか sphere cylinder box torus cone pyramid tube roundcube roundcylinder
		float weight; // torus tubeでのみ使用
		Vector3 pos, scale, angle;
		float power;
	}
	//Mqo.this(Stream src)
	Scene scene();
	Material[] material();
	Object[] object();
	Blob[] blob();
}


class MqoLexicalAnalyzer : _MqoLexicalAnalyzer
{
	private final Stream src;
	private final MqoTokenType[char[]] tokenMap;
	
	unittest // tokenMapのvalueは一意
	{
		MqoLexicalAnalyzer a = new MqoLexicalAnalyzer(new MemoryStream(" "));
		bit[MqoTokenType] bitMap;
		foreach (MqoTokenType b; a.tokenMap.values)
		{
			assert(!(b in bitMap));
			bitMap[b] = true;
		}
	}
	
	this(Stream src)
	{
		this.src = src;
		
		tokenMap["Metasequoia"] = MqoTokenType.Metasequoia;
		tokenMap["Document"] = MqoTokenType.Document;
		tokenMap["Format"] = MqoTokenType.Format;
		tokenMap["Text"] = MqoTokenType.Text;
		tokenMap["Ver"] = MqoTokenType.Ver;
		tokenMap["Scene"] = MqoTokenType.Scene;
		tokenMap["Material"] = MqoTokenType.Material;
		tokenMap["Object"] = MqoTokenType.Object;
		tokenMap["pos"] = MqoTokenType.pos;
		tokenMap["lookat"] = MqoTokenType.lookat;
		tokenMap["head"] = MqoTokenType.head;
		tokenMap["pich"] = MqoTokenType.pich;
		tokenMap["ortho"] = MqoTokenType.ortho;
		tokenMap["zoom2"] = MqoTokenType.zoom2;
		tokenMap["amb"] = MqoTokenType.amb;
		tokenMap["shader"] = MqoTokenType.shader;
		tokenMap["vcol"] = MqoTokenType.vcol;
		tokenMap["col"] = MqoTokenType.col;
		tokenMap["dif"] = MqoTokenType.dif;
		tokenMap["emi"] = MqoTokenType.emi;
		tokenMap["spc"] = MqoTokenType.spc;
		tokenMap["power"] = MqoTokenType.power;
		tokenMap["tex"] = MqoTokenType.tex;
		tokenMap["aplane"] = MqoTokenType.aplane;
		tokenMap["bump"] = MqoTokenType.bump;
		tokenMap["proj_type"] = MqoTokenType.proj_type;
		tokenMap["proj_pos"] = MqoTokenType.proj_pos;
		tokenMap["proj_scale"] = MqoTokenType.proj_scale;
		tokenMap["proj_angle"] = MqoTokenType.proj_angle;
		tokenMap["patch"] = MqoTokenType.patch;
		tokenMap["segment"] = MqoTokenType.segment;
		tokenMap["visible"] = MqoTokenType.visible;
		tokenMap["locking"] = MqoTokenType.locking;
		tokenMap["shading"] = MqoTokenType.shading;
		tokenMap["facet"] = MqoTokenType.facet;
		tokenMap["color"] = MqoTokenType.color;
		tokenMap["color_type"] = MqoTokenType.color_type;
		tokenMap["mirror"] = MqoTokenType.mirror;
		tokenMap["mirror_axis"] = MqoTokenType.mirror_axis;
		tokenMap["mirror_dis"] = MqoTokenType.mirror_dis;
		tokenMap["lathe"] = MqoTokenType.lathe;
		tokenMap["lathe_axis"] = MqoTokenType.lathe_axis;
		tokenMap["lathe_seg"] = MqoTokenType.lathe_seg;
		tokenMap["vertex"] = MqoTokenType.vertex;
		tokenMap["face"] = MqoTokenType.face;
		tokenMap["V"] = MqoTokenType.V;
		tokenMap["M"] = MqoTokenType.M;
		tokenMap["UV"] = MqoTokenType.UV;
		tokenMap["Blob"] = MqoTokenType.Blob;
		tokenMap["blob1"] = MqoTokenType.blob1;
		tokenMap["sphere"] = MqoTokenType.sphere;
		tokenMap["cylinder"] = MqoTokenType.cylinder;
		tokenMap["box"] = MqoTokenType.box;
		tokenMap["torus"] = MqoTokenType.torus;
		tokenMap["cone"] = MqoTokenType.cone;
		tokenMap["pyramid"] = MqoTokenType.pyramid;
		tokenMap["tube"] = MqoTokenType.tube;
		tokenMap["roundcube"] = MqoTokenType.roundcube;
		tokenMap["roundcylinder"] = MqoTokenType.roundcylinder;
		tokenMap["weight"] = MqoTokenType.weight;
		tokenMap["scale"] = MqoTokenType.scale;
		tokenMap["angle"] = MqoTokenType.angle;
		tokenMap["Eof"] = MqoTokenType.Eof;
		tokenMap["depth"] = MqoTokenType.depth;
		tokenMap["folding"] = MqoTokenType.folding;
		tokenMap["BackImage"] = MqoTokenType.BackImage;
		tokenMap["pers"] = MqoTokenType.pers;
		tokenMap["rotation"] = MqoTokenType.rotation;
		tokenMap["translation"] = MqoTokenType.translation;
		
		current = src.getc();
	}
	
	unittest
	{
		Stream src = new MemoryStream(
			"0 1.0 \"2\"\n"
			"Scene {\n"
			"	pos 0.0000\n"
			"}\n"
			"tex(\"hoge\")\n"
			"Eof\n"
			);
		MqoLexicalAnalyzer a = new MqoLexicalAnalyzer(src);
		assert(MqoTokenType.INTEGER == a.read());
		assert(0 == a.intBuffer);
		assert(MqoTokenType.FLOAT == a.read());
		assert(1.0 == a.floatBuffer);
		assert(MqoTokenType.STRING == a.read());
		assert("2" == a.stringBuffer);
		assert(MqoTokenType.Scene == a.read());
		assert('{' == a.read());
		assert(MqoTokenType.pos == a.read());
		assert(MqoTokenType.FLOAT == a.read());
		assert(0.0 == a.floatBuffer);
		assert('}' == a.read());
		assert(MqoTokenType.tex == a.read());
		assert('(' == a.read());
		assert(MqoTokenType.STRING == a.read());
		assert("hoge" == a.stringBuffer);
		assert(')' == a.read());
		assert(MqoTokenType.Eof == a.read());
	}
	
	private char _current = ' ';
	uint line = 1;
	private char current() { return _current; }
	private void current(char a)
	{
		if ('\n' == a) line++;
		_current = a;
	}
	
	MqoTokenType read()
	{
		token = "";
		while (isWhiteSpace(current)) current = src.getc();
		
		if (isDigit(current) || current == '-')
		{
			while (true)
			{
				token ~= current;
				current = src.getc();
				if (!isDigit(current) && '.' != current)
				{
					return isFloat(token) ? MqoTokenType.FLOAT : MqoTokenType.INTEGER;
				}
				assert(!src.eof);
			}
		}
		else if ('\"' == current)
		{
			while (true)
			{
				current = src.getc();
				if ('\"' == current)
				{
					current = src.getc();
					return MqoTokenType.STRING;
				}
				assert(!src.eof);
				token ~= current;
			}
		}
		else if ('{' == current || '}' == current
			|| '(' == current || ')' == current)
		{
			char result = current;
			current = src.getc();
			return cast(MqoTokenType)result;
		}
		
		while (true)
		{
			token ~= current;
			current = src.getc();
			if (isWhiteSpace(current) || '(' == current)
			{
				if (!(token in tokenMap))
				{
					throw new Error("unknown token: " ~ token);
				}
				return tokenMap[token];
			}
			if (src.eof) throw new Error("a");
		}
		
		throw new Error("a");
	}
	
	unittest
	{
		assert(isFloat("1.0"));
		assert(!isFloat("1"));
		assert(isWhiteSpace(10));
	}
	
	static assert(0x0D == '\r');
	static assert(0x0A == '\n');
	
	static bit isWhiteSpace(char a) { return -1 != find(whitespace, a) ? true : false; }
	static bit isDigit(char a) { return -1 != find(digits, a) ? true : false; }
	static bit isFloat(char[] a) { return -1 != find(a, '.') ? true : false; }
	
	private char[] token = "";
	int intBuffer() { return cast(int)atoi(token); }
	float floatBuffer() { return atof(token); }
	char[] stringBuffer() { return token; }
}

class Mqo : _Mqo
{
	Scene scene() { return _scene; }
	Material[] material() { return _material; }
	Object[] object() { return _object; }
	Blob[] blob() { return _blob; }
	
	private Scene _scene;
	private Material[] _material;
	private Object[] _object;
	private Blob[] _blob;
	
	private MqoLexicalAnalyzer lex;
	private alias MqoTokenType Mtt;
	
	this(Stream src)
	{
		lex = new MqoLexicalAnalyzer(src);
		readHeader();
		readScene();
		readMaterialAndObjectAndBlob();
	}
	
	private void readHeader()
	{
		read(Mtt.Metasequoia);
		read(Mtt.Document);
		read(Mtt.Format);
		read(Mtt.Text);
		read(Mtt.Ver);
		if (1.0 != readFloat()) throw new Error("version");
	}
	
	private void readScene()
	{
		read(Mtt.Scene);
		read('{');
		read(Mtt.pos); _scene.pos = readVector3();
		read(Mtt.lookat); _scene.lookat = readVector3();
		read(Mtt.head); _scene.head = readFloat();
		read(Mtt.pich); _scene.pich = readFloat();
		read(Mtt.ortho); _scene.ortho = readInt();
		read(Mtt.zoom2); _scene.zoom2 = readFloat();
		read(Mtt.amb); _scene.amb = readVector3();
		read('}');
	}
	
	private void readMaterial() // Materialを呼んだ状態で呼ぶ
	{
		_material.length = readInt();
		read('{');
		read(Mtt.STRING);
		foreach (inout Material a; _material)
		{
			// ここでlexがnameを読み込んだ状態に
			a.name = lex.stringBuffer;
			Mtt type = lex.read();
			type = readIntP(type, Mtt.shader, a.shader);
			type = readIntP(type, Mtt.vcol, a.vcol);
			type = readVector4P(type, Mtt.col, a.col);
			type = readFloatP(type, Mtt.dif, a.dif);
			type = readFloatP(type, Mtt.amb, a.amb);
			type = readFloatP(type, Mtt.emi, a.emi);
			type = readFloatP(type, Mtt.spc, a.spc);
			type = readFloatP(type, Mtt.power, a.power);
			type = readStringP(type, Mtt.tex, a.tex);
			type = readStringP(type, Mtt.aplane, a.aplane);
			type = readStringP(type, Mtt.bump, a.bump);
			type = readIntP(type, Mtt.proj_type, a.proj_type);
			type = readVector3P(type, Mtt.proj_pos, a.proj_pos);
			type = readVector3P(type, Mtt.proj_scale, a.proj_scale);
			type = readVector3P(type, Mtt.proj_angle, a.proj_angle);
			if ('}' == type) return;
			assert(Mtt.STRING == type);
		}
	}
	
	// 読めるなら読む
	private Mtt readIntP(Mtt current, Mtt name, out int result) // name(int)
	{
		if (current != name) return current;
		read('('); result = readInt(); read(')');
		return lex.read();
	}
	private Mtt readVector4P(Mtt current, Mtt name, out Vector4 result) // name(Vector4)
	{
		if (current != name) return current;
		read('('); result = readVector4(); read(')');
		return lex.read();
	}
	private Mtt readFloatP(Mtt current, Mtt name, out float result) // name(float)
	{
		if (current != name) return current;
		read('('); result = readFloat(); read(')');
		return lex.read();
	}
	private Mtt readStringP(Mtt current, Mtt name, out char[] result) // name(string)
	{
		if (current != name) return current;
		read('('); result = readString(); read(')');
		return lex.read();
	}
	private Mtt readVector3P(Mtt current, Mtt name, out Vector3 result) // name(Vector3)
	{
		if (current != name) return current;
		read('('); result = readVector3(); read(')');
		return lex.read();
	}
	
	
	private void readMaterialAndObjectAndBlob()
	{
		Mtt type = lex.read();
		if (type == Mtt.BackImage)
		{
			while (true)
			{
				type = lex.read();
				if (type == Mtt.Material) break;
			}
		}
		if (type == Mtt.Material)
		{
			readMaterial();
			type = lex.read();
		}
		while (Mtt.Object == type)
		{
			Object a;
			a.name = readString();
			read('{');
			type = lex.read();
			type = readInt(type, Mtt.depth, a.depth);
			type = readInt(type, Mtt.folding, a.folding);
			type = readVector3(type, Mtt.scale, a.scale);
			type = readVector3(type, Mtt.rotation, a.rotation);
			type = readVector3(type, Mtt.translation, a.translation);
			type = readInt(type, Mtt.patch, a.patch);
			type = readInt(type, Mtt.segment, a.segment);
			type = readInt(type, Mtt.visible, a.visible);
			type = readInt(type, Mtt.locking, a.locking);
			type = readInt(type, Mtt.shading, a.shading);
			type = readFloat(type, Mtt.facet, a.facet);
			type = readVector3(type, Mtt.color, a.color);
			type = readInt(type, Mtt.color_type, a.color_type);
			type = readInt(type, Mtt.mirror, a.mirror);
			type = readInt(type, Mtt.mirror_axis, a.mirror_axis);
			type = readFloat(type, Mtt.mirror_dis, a.mirror_dis);
			type = readInt(type, Mtt.lathe, a.lathe);
			type = readInt(type, Mtt.lathe_axis, a.lathe_axis);
			type = readInt(type, Mtt.lathe_seg, a.lathe_seg);
			if (Mtt.vertex != type) { throwSyntaxError(); }
			readVertex(a);
			readFace(a);
			read('}');
			_object ~= a;
			type = lex.read();
		}
		if (type == Mtt.Eof) return;
		
		assert(type == Mtt.Blob);
		read(Mtt.blob1);
		_blob.length = readInt();
		read('{');
		foreach (inout Blob a; _blob)
		{
			lex.read();
			a.name = lex.stringBuffer;
			read('{');
			if ("torus" == a.name || "tube" == a.name)
			{
				read(Mtt.weight); a.weight = readFloat();
			}
			read(Mtt.pos); a.pos = readVector3();
			read(Mtt.scale); a.scale = readVector3();
			read(Mtt.angle); a.angle = readVector3();
			read(Mtt.power); a.power = readFloat();
			read('}');
		}
		read('}');
		read(Mtt.Eof);
	}
	
	// 読めるなら読む
	private Mtt readInt(Mtt current, Mtt name, out int result) // name int
	{
		if (current != name) return current;
		result = readInt();
		return lex.read();
	}
	private Mtt readFloat(Mtt current, Mtt name, out float result) // name float
	{
		if (current != name) return current;
		result = readFloat();
		return lex.read();
	}
	private Mtt readVector3(Mtt current, Mtt name, out Vector3 result) // name Vector3
	{
		if (current != name) return current;
		result = readVector3();
		return lex.read();
	}
	
	private void readVertex(inout Object a)
	{
		// vertexは読まれている
		a.vertex.length = readInt();
		read('{');
		foreach (inout Vector3 b; a.vertex) b = readVector3();
		read('}');
	}
	
	private void readFace(inout Object a)
	{
		read(Mtt.face);
		a.face.length = readInt();
		read('{');
		Mtt type = lex.read();
		if ('}' == type) return;
		foreach (inout Face b; a.face)
		{
			// INTEGERが読まれている
			b.vlength = lex.intBuffer;
			type = lex.read();
			if (type == Mtt.V)
			{
				b.V.length = b.vlength;
				read('(');
				for (int i = 0; i < b.vlength; i++) b.V[i] = readInt();
				read(')');
				type = lex.read();
			}
			if (type == Mtt.M)
			{
				read('('); b.M = readInt(); read(')');
				type = lex.read();
			}
			if (type == Mtt.UV)
			{
				b.UV.length = b.vlength;
				read('(');
				for (int i = 0; i < b.vlength; i++) b.UV[i] = readVector2();
				read(')');
				type = lex.read();
			}
			assert(type == Mtt.INTEGER || type == '}');
		}
		assert(type == '}');
	}
	
	char[] toString()
	{
		MemoryStream result = new MemoryStream();
		result.writeLine("Metasequoia Document");
		result.writeLine("Format Text Ver 1.0");
		result.writeLine("");
		
		result.writeLine("Scene {");
		with (_scene)
		{
			result.writefln("\tpos %.4f %.4f %.4f", pos.x, pos.y, pos.z);
			result.writefln("\tlookat %.4f %.4f %.4f", lookat.x, lookat.y, lookat.z);
			result.writefln("\thead %.4f", head);
			result.writefln("\tpich %.4f", pich);
			result.writefln("\tortho %d", ortho);
			result.writefln("\tzoom2 %.4f", zoom2);
			result.writefln("\tamb %.3f %.3f %.3f", amb.x, amb.y, amb.z);
		}
		result.writeLine("}");
		
		result.writefln("Material %d {", _material.length);
		foreach (Material a; _material)
		{
			result.writef("\t\"%s\" ", a.name);
			result.writef("shader(%d) ", a.shader);
			if (a.vcol != 0) result.writef("vcol(%d) ", a.vcol);
			result.writef("col(%.3f %.3f %.3f %.3f) ", a.col.x, a.col.y, a.col.z, a.col.w);
			result.writef("dif(%.3f) ", a.dif);
			result.writef("amb(%.3f) ", a.amb);
			result.writef("emi(%.3f) ", a.emi);
			result.writef("spc(%.3f) ", a.spc);
			result.writef("power(%.2f)", a.power);
			if (a.tex != "") result.writef(" tex(\"%s\")", a.tex);
			if (a.aplane != "") result.writef(" aplane(\"%s\")", a.aplane);
			if (a.bump != "") result.writef(" bump(\"%s\")", a.bump);
			if (a.proj_type != 0)
			{
				result.writef(" proj_type(%d)", a.proj_type);
				result.writef(" proj_pos(%.3f %.3f %.3f)", a.proj_pos.x, a.proj_pos.y, a.proj_pos.z);
				result.writef(" proj_scale(%.3f %.3f %.3f)", a.proj_scale.x, a.proj_scale.y, a.proj_scale.z);
				result.writef(" proj_angle(%.3f %.3f %.3f)", a.proj_angle.x, a.proj_angle.y, a.proj_angle.z);
			}
			result.writeLine("");
		}
		result.writeLine("}");
		
		foreach (Object a; _object)
		{
			result.writefln("Object \"%s\" {", a.name);
			if (a.patch != 0) result.writefln("\tpatch %d", a.patch);
			if (a.patch == 3) result.writefln("\tsegment %d", a.segment);
			result.writefln("\tvisible %d", a.visible);
			result.writefln("\tlocking %d", a.locking);
			result.writefln("\tshading %d", a.shading);
			result.writefln("\tfacet %.1f", a.facet);
			result.writefln("\tcolor %.3f %.3f %.3f", a.color.x, a.color.y, a.color.z);
			result.writefln("\tcolor_type %d", a.color_type);
			if (a.mirror != 0) result.writefln("\tmirror %d", a.mirror);
			if (a.mirror != 0) result.writefln("\tmirror_axis %d", a.mirror_axis);
			if (a.mirror == 2 && a.mirror_dis != 0.0) result.writefln("\tmirror_dis %.3f", a.mirror_dis);
			if (a.lathe != 0)
			{
				result.writefln("\tlathe %d", a.lathe);
				result.writefln("\tlathe_axis %d", a.lathe_axis);
				result.writefln("\tlathe_seg %d", a.lathe_seg);
			}
			result.writefln("\tvertex %d {", a.vertex.length);
			foreach (Vector3 b; a.vertex)
			{
				result.writefln("\t\t%.4f %.4f %.4f", b.x, b.y, b.z);
			}
			result.writeLine("\t}");
			
			result.writefln("\tface %d {", a.face.length);
			foreach (Face b; a.face)
			{
				result.writef("\t\t%d ", b.vlength);
				if (b.vlength == 2) result.writef("V(%d %d)", b.V[0], b.V[1]);
				else if (b.vlength == 3) result.writef("V(%d %d %d)", b.V[0], b.V[1], b.V[2]);
				else if (b.vlength == 4) result.writef("V(%d %d %d %d)", b.V[0], b.V[1], b.V[2], b.V[3]);
				if (_material.length != 0)
				{
					result.writef(" M(%d)", b.M);
					if (b.UV.length != 0)
					{
						if (b.vlength == 3)
						{
							result.writef(" UV(%.5f %.5f %.5f %.5f %.5f %.5f)",
								b.UV[0].x, b.UV[0].y,
								b.UV[1].x, b.UV[1].y,
								b.UV[2].x, b.UV[2].y
								);
						}
						else if (b.vlength == 4)
						{
							result.writef(" UV(%.5f %.5f %.5f %.5f %.5f %.5f %.5f %.5f)",
								b.UV[0].x, b.UV[0].y,
								b.UV[1].x, b.UV[1].y,
								b.UV[2].x, b.UV[2].y,
								b.UV[3].x, b.UV[3].y
								);
						}
					}
				}
				result.writeLine("");
			}
			result.writeLine("\t}");
			result.writeLine("}");
		}
		
		if (_blob.length != 0) result.writefln("Blob blob1 %d {", _blob.length);
		foreach (Blob b; _blob)
		{
			result.writefln("\t%s {", b.name);
			if (b.name == "torus" || b.name == "tube")
			{
				result.writefln("\t\tweight %.3f", b.weight);
			}
			result.writefln("\t\tpos %.4f %.4f %.4f", b.pos.x, b.pos.y, b.pos.z);
			result.writefln("\t\tscale %.4f %.4f %.4f", b.scale.x, b.scale.y, b.scale.z);
			result.writefln("\t\tangle %.4f %.4f %.4f", b.angle.x, b.angle.y, b.angle.z);
			result.writefln("\t\tpower %.4f", b.power);
			result.writeLine("\t}");
		}
		if (_blob.length != 0) result.writeLine("}");
		
		result.writeLine("Eof");
		return result.toString();
	}
	
private:
	void read(Mtt a)
	{
		if (a != lex.read()) throwSyntaxError();
	}
	void read(char a) { read(cast(Mtt)a); }
	float readFloat()
	{
		if (Mtt.FLOAT != lex.read()) throwSyntaxError();
		return lex.floatBuffer;
	}
	Vector2 readVector2()
	{
		Vector2 result;
		result.x = readFloat(); result.y = readFloat();
		return result;
	}
	Vector3 readVector3()
	{
		Vector3 result;
		result.x = readFloat(); result.y = readFloat();
		result.z = readFloat();
		return result;
	}
	Vector4 readVector4()
	{
		Vector4 result;
		result.x = readFloat(); result.y = readFloat();
		result.z = readFloat(); result.w = readFloat();
		return result;
	}
	int readInt()
	{
		if (Mtt.INTEGER != lex.read()) throwSyntaxError();
		return lex.intBuffer;
	}
	char[] readString()
	{
		if (Mtt.STRING != lex.read()) throwSyntaxError();
		return lex.stringBuffer;
	}
	
	private void throwSyntaxError()
	{
		throw new Error(format("syntax error(%d): %s", lex.line, lex.stringBuffer));
	}
}
