﻿module skeleton;
private import std.file;
private import std.stream;
private import std.math;
private import std.string;
private import mqo;
private import mkm;
private import coneneko.math;
private import coneneko.sjis2utf8;

enum BoneRelationType
{
	PARENT_CHILD,
	PARENT_LINE_CHILD,
}
interface _Skeleton
{
	//Skeleton.this(char[] mqoFileName)
	//Skeleton.this(Mqo mqo)
	//Skeleton.this(Bone[] boneList, LineRelation[] lineRelationList)
	Bone[] boneList();
	BoneRelation[] boneRelationList();
	BoneRelationType getRelationType(Bone a, Bone b);
	Bone rootBone();
	Bone getParent(Bone bone);
	uint indexOf(Bone bone);
	float[] extractBone(); // float9[]
	ubyte[] extractRelation(); // ubyte2[]
	bit isRoot(uint boneIndex);
	uint getParent(uint boneIndex);
}
interface _Bone
{
	//Bone.this(Vector p0, Vector p1, Vector p2)
	//Bone.this(char[] name, Vector p0, Vector p1, Vector p2)
	Vector startPosition();
	Vector endPosition();
	Vector hPosition();
	char[] name();
	char[] materialName();
}
interface _BoneRelation
{
	//BoneRelation.this(Bone first, Bone second, BoneRelationType type)
	Bone first();
	Bone second();
	BoneRelationType type();
}
interface _LineRelation
{
	//LineRelation.this(Vector p0, Vector p1)
	Vector position0();
	Vector position1();
	bit isRelational(Vector a, Vector b);
}

class Skeleton : _Skeleton
{
	unittest
	{
		Bone[] boneList = new Bone[4];
		//   3      2
		// 2 1 -> 3 1
		//   0      0
		boneList[0] = new Bone(
			Vector.create(0, 0, 0),
			Vector.create(0, 2, 0),
			Vector.create(1, 0, 0)
			);
		boneList[1] = new Bone(
			Vector.create(0, 4, 0),
			Vector.create(0, 6, 0),
			Vector.create(1, 4, 0)
			);
		boneList[2] = new Bone(
			Vector.create(0, 2, 0),
			Vector.create(-2, 2, 0),
			Vector.create(0, 3, 0)
			);
		boneList[3] = new Bone(
			Vector.create(0, 6, 0),
			Vector.create(0, 8, 0),
			Vector.create(1, 6, 0)
			);
		LineRelation[] lineRelationList = new LineRelation[1];
		lineRelationList[0] = new LineRelation(
			Vector.create(0, 2, 0), Vector.create(0, 4, 0));
		Skeleton skeleton2 = new Skeleton(boneList, lineRelationList);
		assert(3 == skeleton2.boneRelationList.length);
		assert(boneList[0] == skeleton2.boneList[0]);
		assert(boneList[1] == skeleton2.boneList[1]);
		assert(boneList[3] == skeleton2.boneList[2]);
		assert(boneList[2] == skeleton2.boneList[3]);

		// 置き換え前の関連順と同じ
		assert(boneList[0] == skeleton2.boneRelationList[0].first);
		assert(boneList[1] == skeleton2.boneRelationList[0].second);
		assert(boneList[0] == skeleton2.boneRelationList[1].first);
		assert(boneList[2] == skeleton2.boneRelationList[1].second);
		assert(boneList[1] == skeleton2.boneRelationList[2].first);
		assert(boneList[3] == skeleton2.boneRelationList[2].second);
		
		assert(BoneRelationType.PARENT_LINE_CHILD == skeleton2.getRelationType(boneList[0], boneList[1]));
		assert(BoneRelationType.PARENT_CHILD == skeleton2.getRelationType(boneList[1], boneList[3]));
		assert(BoneRelationType.PARENT_CHILD == skeleton2.getRelationType(boneList[0], boneList[2]));
	}

	private final Bone[] _boneList;
	private final BoneRelation[] _boneRelationList;
	
	this(char[] mqoFileName)
	{
		char[] mqoSrc = sjisToUtf8(cast(char[])read(mqoFileName));
		this(new Mqo(new MemoryStream(mqoSrc)));
	}
	
	this(Mqo mqo)
	{
		Bone[] bl;
		LineRelation[] lrl;
		
		Mqo.Object boneObject;
		foreach (Mqo.Object a; mqo.object)
		{
			if (a.name.length >= 5 && "bone:" == a.name[0..5])
			{
				boneObject = a;
				break;
			}
		}
		
		foreach (Mqo.Face a; boneObject.face)
		{
			if (a.V.length != 3) continue;
			with (boneObject)
			{
				bl ~= new Bone(
					mqo.material[a.M].name,
					Vector.create(vertex[a.V[0]].x, vertex[a.V[0]].y, vertex[a.V[0]].z),
					Vector.create(vertex[a.V[1]].x, vertex[a.V[1]].y, vertex[a.V[1]].z),
					Vector.create(vertex[a.V[2]].x, vertex[a.V[2]].y, vertex[a.V[2]].z));
			}
		}
		
		foreach (Mqo.Face a; boneObject.face)
		{
			if (a.V.length != 2) continue;
			with (boneObject)
			{
				lrl ~= new LineRelation(
					Vector.create(vertex[a.V[0]].x, vertex[a.V[0]].y, vertex[a.V[0]].z),
					Vector.create(vertex[a.V[1]].x, vertex[a.V[1]].y, vertex[a.V[1]].z));
			}
		}
		
		assert(0 < bl.length);
		this(bl, lrl);
	}
	
	Bone[] boneList() { return _boneList; }
	BoneRelation[] boneRelationList() { return _boneRelationList; }
	
	this(Bone[] boneList, LineRelation[] lineRelationList)
	{
		this._boneList.length = boneList.length;
		this._boneList[] = boneList[];
		this._boneRelationList = createBoneRelationList(boneList, lineRelationList);
		relocate();
	}
	
	BoneRelationType getRelationType(Bone a, Bone b)
	{
		foreach (BoneRelation relation; _boneRelationList)
		{
			if ((relation.first == a && relation.second == b)
				|| (relation.first == b && relation.second == a))
			{
				return relation.type;
			}
		}
		assert(false);
	}
	
	// boneListを再帰的ツリーの順に並び替える
	private void relocate()
	{
		Bone[] result;

		void relocateR(inout Bone[] result, Bone currentBone) // 再帰
		{
			result ~= currentBone;
			foreach (BoneRelation a; _boneRelationList)
			{
				if (currentBone is a.first)
				{
					relocateR(result, a.second);
				}
			}
		}

		relocateR(result, rootBone);
		if (_boneList.length != result.length) throw new Error("bone relocate error");
		_boneList[] = result;
	}
	
	Bone rootBone() // rootBoneはひとつだけ
	{
		Bone result = _boneList[0];
		while (true)
		{
			Bone parent = getParent(result);
			if (parent is null) return result;
			result = parent;
		}
		
		throw new Error("a");
	}

	Bone getParent(Bone bone)
	{
		foreach (BoneRelation a; _boneRelationList)
		{
			if (a.second is bone) return a.first;
		}
		return null;
	}

	private static BoneRelation[] createBoneRelationList(Bone[] boneList, LineRelation[] lineRelationList)
	{
		BoneRelation[] result;
		foreach (Bone a; boneList)
		{
			foreach (Bone b; boneList)
			{
				if (a is b) continue;
				if (a.endPosition == b.startPosition)
				{
					result ~= new BoneRelation(a, b, BoneRelationType.PARENT_CHILD);
				}
				else
				{
					foreach (LineRelation c; lineRelationList)
					{
						if (c.isRelational(a.endPosition, b.startPosition))
						{
							result ~= new BoneRelation(a, b, BoneRelationType.PARENT_LINE_CHILD);
						}
					}
				}
			}
		}
		return result;
	}

	uint indexOf(Bone bone)
	{
		for (int i = 0; i < _boneList.length; i++)
		{
			if (bone is _boneList[i]) return i;
		}
		assert(false);
	}

	override char[] toString()
	{
		MemoryStream result = new MemoryStream();
		foreach (Bone a; _boneList) result.writeLine(a.toString());
		result.writeLine("");
		foreach (BoneRelation a; _boneRelationList)
		{
			result.writeLine(.toString(indexOf(a.first)) ~ " " ~ .toString(indexOf(a.second)));
		}
		return result.toString();
	}
	
	float[] extractBone()
	{
		MemoryStream result = new MemoryStream();
		foreach (Bone a; _boneList)
		{
			result.write(a.startPosition.x);
			result.write(a.startPosition.y);
			result.write(a.startPosition.z);
			result.write(a.endPosition.x);
			result.write(a.endPosition.y);
			result.write(a.endPosition.z);
			result.write(a.hPosition.x);
			result.write(a.hPosition.y);
			result.write(a.hPosition.z);
		}
		return cast(float[])result.data;
	}
	
	ubyte[] extractRelation()
	{
		MemoryStream result = new MemoryStream();
		foreach (BoneRelation a; boneRelationList)
		{
			result.write(cast(ubyte)indexOf(a.first));
			result.write(cast(ubyte)indexOf(a.second));
		}
		return cast(ubyte[])result.data;
	}
	
	this(float[] bones, ubyte[] relations)
	in
	{
		assert(0 == bones.length % 9);
		assert(0 == relations.length % 2);
	}
	body
	{
		Bone[] boneList;
		for (int i = 0; i < bones.length; i += 9)
		{
			Vector a = Vector.create(bones[i + 0], bones[i + 1], bones[i + 2]);
			Vector b = Vector.create(bones[i + 3], bones[i + 4], bones[i + 5]);
			Vector c = Vector.create(bones[i + 6], bones[i + 7], bones[i + 8]);
			boneList ~= new Bone(a, b, c);
		}
		
		LineRelation[] lineRelationList;
		for (int i = 0; i < relations.length; i += 2)
		{
			Vector a = boneList[relations[i + 0]].endPosition;
			Vector b = boneList[relations[i + 1]].startPosition;
			lineRelationList ~= new LineRelation(a, b);
		}
		
		this(boneList, lineRelationList);
	}
	
	bit isRoot(uint boneIndex) { return boneIndex == 0 ? true : false; }
	uint getParent(uint boneIndex) { return indexOf(getParent(boneList[boneIndex])); }
}

class Bone : _Bone
{
	unittest
	{
		Vector p0 = Vector.create(0, 0, 0);
		Vector p1 = Vector.create(0, 2, 0);
		Vector p2 = Vector.create(1, 0, 0);
		Bone a = new Bone(p0, p1, p2);
		assert(p0 == a.startPosition);
		assert(p1 == a.endPosition);
		assert(p2 == a.hPosition);
		Bone b = new Bone(p1, p2, p0);
		assert(p0 == b.startPosition);
		assert(p1 == b.endPosition);
		assert(p2 == b.hPosition);
		assert("0.000000 0.000000 0.000000  0.000000 2.000000 0.000000  1.000000 0.000000 0.000000" == a.toString());
		assert(a == b);
	}

	private final Vector _startPosition, _endPosition, _hPosition;

	this(Vector p0, Vector p1, Vector p2)
	{
		float d01 = distance(p0, p1);
		float d02 = distance(p0, p2);
		float d12 = distance(p1, p2);
		float dmax = max(max(d01, d02), d12);
		float dmin = min(min(d01, d02), d12);
		_startPosition = dmax == d01 ? p2 : dmax == d02 ? p1 : p0;
		_endPosition = dmin == d01 ? p2 : dmin == d02 ? p1 : p0;
		_hPosition = p0 != startPosition && p0 != endPosition ? p0 :
			p1 != startPosition && p1 != endPosition ? p1 : p2;
	}
	
	this(char[] materialName, Vector p0, Vector p1, Vector p2)
	{
		this._name = insertRorL(materialName, p0.x);
		this._materialName = materialName;
		this(p0, p1, p2);
	}
	
	this(float x0, float y0, float z0,
		float x1, float y1, float z1,
		float x2, float y2, float z2)
	{
		this(Vector.create(x0, y0, z0),
			Vector.create(x1, y1, z1),
			Vector.create(x2, y2, z2));
	}
	
	Vector startPosition() { return _startPosition; }
	Vector endPosition() { return _endPosition; }
	Vector hPosition() { return _hPosition; }

	private static float max(float a, float b)
	{
		return a >= b ? a : b;
	}

	private static float min(float a, float b)
	{
		return a <= b ? a : b;
	}

	public override char[] toString()
	{
		MemoryStream result = new MemoryStream();
		result.printf("%f %f %f  %f %f %f  %f %f %f",
			cast(double)startPosition.x, cast(double)startPosition.y, cast(double)startPosition.z,
			cast(double)endPosition.x, cast(double)endPosition.y, cast(double)endPosition.z,
			cast(double)hPosition.x, cast(double)hPosition.y, cast(double)hPosition.z
			);
		return result.toString();
	}

	override int opEquals(Object o)
	{
		return toString() == o.toString();
	}
	
	private final char[] _name;
	char[] name() { return _name; } // [] R, L付きの
	
	private final char[] _materialName;
	char[] materialName() { return _materialName; }
	
	void materialName(char[] a)
	{
		_name = insertRorL(a, _startPosition.x);
		_materialName = a;
	}
	
	private static char[] insertRorL(char[] a, float x) // []の中のR, Lを入れる
	{
		if (-1 == find(a, "[]")) return a;
		if (x >= 0) return replace(a, "[]", "[L]");
		else return replace(a, "[]", "[R]");
	}
	
	
	static float distanceFromCenter(Bone bone, Vector position)
	{
		return coneneko.math.distance((bone._startPosition + bone._endPosition) / 2.0, position);
	}
	
	static float distanceFromLineSegment(Bone bone, Vector position)
	{
		return coneneko.math.distanceOfPointAndLineSegment(
			position,
			bone._startPosition,
			bone._endPosition
		);
	}
	
	float height()
	{
		return distance(_startPosition, _hPosition);
	}
}

class BoneRelation : _BoneRelation
{
	unittest
	{
		BoneRelation br = new BoneRelation(
			new Bone(Vector.create(1, 2, 3), Vector.create(4, 5, 6), Vector.create(7, 8, 9)),
			new Bone(Vector.create(9, 8, 7), Vector.create(6, 5, 4), Vector.create(3, 2, 1)),
			BoneRelationType.PARENT_LINE_CHILD
			);
		assert("PARENT_LINE_CHILD  "
			~ "4.000000 5.000000 6.000000  7.000000 8.000000 9.000000  1.000000 2.000000 3.000000"
			~ "  ----  "
			~ "6.000000 5.000000 4.000000  3.000000 2.000000 1.000000  9.000000 8.000000 7.000000"
			== br.toString());
	}

	private final Bone _first, _second;
	private final BoneRelationType _type;

	this(Bone first, Bone second, BoneRelationType type)
	{
		this._first = first;
		this._second = second;
		this._type = type;
	}
	
	public Bone first() { return _first; }
	public Bone second() { return _second; }
	public BoneRelationType type() { return _type; }
	
	public override char[] toString()
	{
		MemoryStream result = new MemoryStream();
		result.writeString(type == BoneRelationType.PARENT_CHILD ? "PARENT_CHILD" : "PARENT_LINE_CHILD");
		result.writeString("  ");
		result.writeString(first.toString());
		result.writeString("  ----  ");
		result.writeString(second.toString());
		return result.toString();
	}
}

class LineRelation : _LineRelation
{
	unittest
	{
		Vector a = Vector.create(0, 0, 0);
		Vector b = Vector.create(1, 1, 1);
		Vector c = Vector.create(2, 2, 2);
		LineRelation ab = new LineRelation(a, b);
		assert(ab.isRelational(a, b));
		assert(ab.isRelational(b, a));
		assert(!ab.isRelational(a, c));
		assert(!ab.isRelational(b, c));
		assert(!ab.isRelational(c, a));
		assert(!ab.isRelational(c, b));
	}

	private final Vector _position0, _position1;
	
	this(Vector p0, Vector p1)
	{
		this._position0 = p0;
		this._position1 = p1;
	}
	
	Vector position0() { return _position0; }
	Vector position1() { return _position1; }

	bit isRelational(Vector a, Vector b)
	{
		return (position0 == a && position1 == b)
			|| (position0 == b && position1 == a) ? true : false;
	}
}
