/* $Id$ */
package saccubus.converter.filegetter;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import org.apache.commons.io.FilenameUtils;
import saccubus.ConvertStopFlag;
import saccubus.converter.profile.InputFileSetting;
import saccubus.net.TextProgressListener;

/**
 * ダウンロード処理を全く必要としない場合のファイルインスタンス化クラス.
 * @author yuki
 */
public class FileInstanciator {

    private final InstanciationType videoType;
    private final InstanciationType commentType;
    private final InstanciationType tcommType;
    private final String videoId;
    private FileGetter videoFileGetter;
    private FileGetter commentFileGetter;
    private FileGetter tcommFileGetter;

    protected void setVideoFileGetter(FileGetter getter) {
        this.videoFileGetter = getter;
    }

    protected void setCommentFileGetter(FileGetter getter) {
        this.commentFileGetter = getter;
    }

    protected void setTcommFileGetter(FileGetter getter) {
        this.tcommFileGetter = getter;
    }

    public static FileInstanciator create(
            ConvertStopFlag stopFlag,
            InstanciationType videoType,
            CommentInstanciationType commentType,
            InstanciationType tcommType,
            LoginInfo li,
            String tag, String time) throws
            IOException {
        FileInstanciator getter;
        if (videoType.isDoanload() || commentType.isDoanload() || tcommType.isDoanload()) {
            getter = new WebFileInstanciator(stopFlag, videoType, commentType, tcommType, li, tag, time);
        } else {
            getter = new FileInstanciator(videoType, commentType, tcommType, tag);
        }
        return getter;
    }

    protected FileInstanciator(
            InstanciationType videoType,
            InstanciationType commentType,
            InstanciationType tcommType,
            String videoId) {
        this.videoType = videoType;
        this.commentType = commentType;
        this.tcommType = tcommType;
        this.videoId = videoId;
        FileGetter getter = new FileGetter();
        setVideoFileGetter(getter);
        setCommentFileGetter(getter);
        setTcommFileGetter(getter);
    }

    /**
     * @return 動画のタイトル.
     * @throws FileNotFoundException ビデオタイトルの自動命名時、参考となるファイルが見つからなかった。
     */
    public String getVideoTitle() throws FileNotFoundException {
        String fileName = null;
        if (!videoType.isAutoFileName()) {
            // 動画ファイル名を直接指定している場合は、そのファイル名を基にタイトルを取得する.
            fileName = videoType.getInitFile().toString();
        } else {
            // 自動命名の場合は、ディレクトリ内にあるファイルからタイトル名を類推。
            String[] files = videoType.getInitFile().list();
            if (files != null) {
                for (String file : files) {
                    if (file.startsWith(getVideoIdWithBracket())) {
                        fileName = FilenameUtils.getBaseName(file);
                        break;
                    }
                }
            }
            if (fileName == null) {
                throw new FileNotFoundException(getVideoIdWithBracket() + "のファイルを特定できませんでした。");
            }
        }

        String baseName = FilenameUtils.getBaseName(fileName);
        int s = baseName.indexOf(getVideoIdWithBracket());
        return baseName.replace(getVideoIdWithBracket(), "");
    }

    /**
     * 動画ファイルを取得します.
     * @param listener 進捗通知を受け取るためのリスナ.
     * @return 動画ファイル.
     * @throws IOException 動画ファイルが存在しない, 取得に失敗した.
     */
    public final File getVideoFile(TextProgressListener listener) throws IOException {
        File file = new FileLocator(videoType.isAutoFileName(), videoType.getInitFile(), getVideoIdWithBracket(),
                getVideoTitle(), ".flv", ".mp4", ".swf").getFile();
        file = videoFileGetter.get(file, listener);
        return file;
    }

    /**
     * コメントファイルを取得します.
     * @param listener 進捗通知を受け取るためのリスナ.
     * @return コメントファイル.
     * @throws IOException コメントファイルが存在しない, 取得に失敗した.
     */
    public final File getCommentFile(TextProgressListener listener) throws IOException {
        File file = new FileLocator(commentType.isAutoFileName(), commentType.getInitFile(), getVideoIdWithBracket(),
                getVideoTitle(), ".xml").getFile();
        file = commentFileGetter.get(file, listener);
        return file;
    }

    /**
     * 投稿者コメントファイルを取得します.
     * @param listener 進捗通知を受け取るためのリスナ.
     * @return 投稿者コメントファイル.
     * @throws IOException コメントファイルが存在しない, 取得に失敗した.
     */
    public final File getTcommFile(TextProgressListener listener) throws IOException {
        File file = new FileLocator(tcommType.isAutoFileName(), tcommType.getInitFile(), getVideoIdWithBracket(),
                getVideoTitle(), ".xml").getFile();
        file = tcommFileGetter.get(file, listener);
        return file;
    }

    private String getVideoIdWithBracket() {
        return "[" + videoId + "]";
    }

    public static class InstanciationType {

        private final boolean download;
        private final boolean autoFileName;
        private final File initFile;

        /**
         * ファイルをインスタンス化する方法を指定するクラス.
         */
        public InstanciationType(InputFileSetting fileSetting) {
            this.download = fileSetting.isDownload();
            this.autoFileName = !fileSetting.getFile().isFile();
            this.initFile = fileSetting.getFile().getFile();
        }

        public boolean isDoanload() {
            return download;
        }

        public File getInitFile() {
            return initFile;
        }

        public boolean isAutoFileName() {
            return autoFileName;
        }
    }

    public static class CommentInstanciationType extends InstanciationType {

        private final boolean autoCommentNum;
        private final int backComment;

        public CommentInstanciationType(InputFileSetting fileSetting, boolean autoCommentNum, int backComment) {
            super(fileSetting);
            this.autoCommentNum = autoCommentNum;
            this.backComment = backComment;
        }

        public boolean isAutoCommentNum() {
            return autoCommentNum;
        }

        public int getBackComment() {
            return backComment;
        }
    }
}
