// -----------------------------------------------------
// Crowbar組込高機能グラフィックライブラリTomahawk
// by T.Shirai
// 開発開始 2012/6/7  -> Crowbar Ver.3.00.00 (2012/6/21)
// -----------------------------------------------------
// tomahawkMode : 0, 1, 2 はTomahawkのモードを切り替える非常に重要なパラメータで，実行中に変更するものではない．
// 0 : このCrowbarにはTomahawkが存在しない場合（tomahawkClass.pdeを削除し，crowbarClass.pde冒頭のコメントを外す）
//     Tomahawkは非常に負荷が大きいのでビューポートを利用する必要がない場合はこのモードの使用を勧めます．
// 1 : Tomahawkを利用するがCrowbarのテキスト表示を背景に描画する．ビューポートはその上のレイヤーとして上書きされる．
//     実行ウィンドウの一部を画像描画用のビューポートとして活用する場合などに適している．
//     テキスト専用レイヤーをブレンドしないため比較的負荷が少ない．
// 2 : Tomahawkを利用する．さらにCrowbarのテキスト出力を専用のレイヤーに描画する．
//     レイヤーは背景，ビューポート，テキスト表示専用レイヤーの順でブレンドされるためテキスト表示が隠れることが無い．
//     オーバーヘッドが最も重いモード．
// （テキスト専用レイヤー使用時：2の制約）：Processingの仕様に基づく
//   まず，フォントの見栄えが悪い．テキストの色も何故か指定した値よりも少し暗くなる．
//   レイヤーを重ねるのにblend()を使用している．テクニックとしては不透明度０（つまり透明）かつ黒い背景に不透明度255で
//   グラフィックス描画を行ない，blend(BLEND)すると不透明度255のPixelのみが合成されるという手法を取っています．
//   ところがtext()はbackground(#000000, 0)の背景色#000000を利用してくれるのですが，不透明度255で描画してしまいます．
//   そこで文字をtext()で描画する度にそのエリアをpixel単位でチェックして背景色#000000の場合は強制的にcolor(#00000, 0)
//   に変更します．したがって文字色として#000000を指定すると区別が付きませんので内部で勝手に#000002に文字色を変更して
//   います．この不透明度の変更の処理にも余計なオーバーヘッドが生じています．あまり大きなフォントサイズを指定すると，
//   blend()にも時間が掛かります．

// 現在の線色や座標などの情報を保持
class currentDrawingClass {
  // 描画関係
  boolean  isFill;       // 塗り潰す？
  int      fillColor;    // 塗り潰し色
  int      fillTrans;    // 塗り潰しの不透明度
  boolean  isStroke;     // 外形線描く？
  int      strokeColor;  // 外形線色
  int      strokeTrans;  // 外形線の不透明度
  float    strokeWeight; // 線幅
  int      rectMode;     // 四角形の座標指定のモード（CORNER, CORNERS, CENTER, RADUIS）
  // 文字関係
  int      textColor;    // テキスト文字の色
  int      textSize;     // フォントサイズ
  int      textAlignH;   // 水平方向割り当て
  int      textAlignV;   // 垂直方向割り当て
  int      textLeading;  // 改行幅
  // LP
  int      vpx, vpy;     // ビューポート内の最後に描画した座標
  float    wpx, wpy;     // ワールド座標における最後に描画した座標
  currentDrawingClass() {
    setDefault();
  }
  // 複製
  void clone(currentDrawingClass src) {
  // 描画関係
    isFill       = src.isFill;
    fillColor    = src.fillColor;
    fillTrans    = src.fillTrans;
    isStroke     = src.isStroke;
    strokeColor  = src.strokeColor;
    strokeTrans  = src.strokeTrans;
    strokeWeight = src.strokeWeight;
    rectMode     = src.rectMode;
  // 文字関係
    textColor    = src.textColor;
    textSize     = src.textSize;
    textAlignH   = src.textAlignH;
    textAlignV   = src.textAlignV;
    textLeading  = src.textLeading;
  // LP
    vpx          = src.vpx;
    vpy          = src.vpy;
    wpx          = src.wpx;
    wpy          = src.wpy;
  }
  // 初期値に設定
  void setDefault() {
  // 描画関係
    isFill       = true;
    fillColor    = #ffffff;
    fillTrans    = 255;
    isStroke     = true;
    strokeColor  = #000000;
    strokeTrans  = 255;
    strokeWeight = 1;
    rectMode     = CORNER;
  // 文字関係
    textColor    = _TEXTCOLOR;
    textSize     = _FONTSIZE;
    textAlignH   = LEFT;
    textAlignV   = TOP;
    textLeading  = int(textSize * 1.5);
  // LP
    vpx = vpy    = 0;
    wpx = wpy    = 0.0;
  }
}

// 各ビューポートのクラス
class viewportClass {
  protected PGraphics pg;                              // ビューポートグラフィックスのハンドル
  protected float     wx_min, wx_max, wy_min, wy_max;  // ビューポートのワールド座標
  protected int       vx, vy;                          // ビューポートの実行スクリーンに対する左上の座標
  protected int       vwidth, vheight;                 // ビューポートの横幅／縦幅
  boolean             monoRatio;                       // ピクセル比は縦も横も同一にするか
  float               pixelRatioX, pixelRatioY;        // ピクセル比（縦と横）ワールド座標→ビューポート内のPixel座標変換用
  protected int       bgColor;                         // ビューポートの背景色
  protected String    label;                           // ビューポートの説明（オプション）
  currentDrawingClass cdSet;                           // 最新の描画関係の情報
  protected int       serialNumber;                    // 作成順の通し番号（viewStack[[]の引数）
  // 見え方（見せ方）
  protected boolean   transparentView;                 // ビューポートの背景が透明なビュー
  protected int       transparency;                    // ビューポートの透明度（枠は除く）： 0～255（デフォルトは255）
  boolean             drawFrame;                       // 枠の表示を行なうか？
  int                 frameColor;                      // 枠の色
  int                 frameWidth;                      // 枠の幅（ピクセル）
  protected int       layerLevel;                      // 表示順（優先度）：数字が大きいほど後（上）に表示される
  float               magnification;                   // 画面の拡大率（ピクセル比にこの値を掛ける）
  float               offsetX, offsetY;                // ワールド座標系におけるオフセット量
  float               rotateAngle;                     // ビューポートの視点の回転角度（[rad]）
  float               pivotX, pivotY;                  // ピボット座標（拡大縮小，回転）
  protected boolean   visible;                         // ビューポートの表示をダイナミックにOn/Offできる
  // ピクセル関係
  int []              pixels;                          // loadPixels()でセットされる
  // テキスト関係
  private PFont       font;
  // コンストラクタ
  viewportClass(String name, int x, int y, int width, int height) {
    // デバッグ情報
    label     = name;
    // 座標関係の初期化
    vx         = x;
    vy         = y;
    vwidth     = width;
    vheight    = height;
    monoRatio  = true;
//  world(0.0, 0.0, vwidth, vheight);  // 初期値で設定
    worldOrigin(1.0);                  // 初期値で設定
    // 背景色，フレーム
    bgColor      = _BGCOLOR;
    transparentView = false;
    drawFrame    = true;
    transparency = 255;
    frameColor   = #000000;
    frameWidth   = 1;
    // 最新の描画関係の情報
    cdSet      = new currentDrawingClass();
    // ビューポートの見せ方
    viewPivotCenter();          // 拡大縮小，回転の中心座標（ビューポート中心に初期設定）
    magnification     = 1.0;    // 拡大縮小
    offsetX = offsetY = 0.0;    // 平行移動量
    rotateAngle       = 0.0;    // 回転角度（[rad]）
    serialNumber      = -1;     // 作成順の通し番号
    layerLevel        = -1;     // レイヤーの順番

    // ビューポート用のグラフィックスを確保
    pg        = createGraphics(vwidth, vheight, JAVA2D);    // P3Dで描画すると円に変な線が沢山入るので．
    pg.beginDraw();
//  pg.smooth();  // 一応，念のために
    pg.endDraw();
    visible   = true;
  }
  // プロパティの複製（属性関係のみ）：そっくりさん．splitViewH/V()で作成した際に用いる
  void cloneAttr(viewportClass src) {
    bgColor         = src.bgColor;
    transparentView = src.transparentView;
    drawFrame       = src.drawFrame;
    transparency    = src.transparency;
    frameColor      = src.frameColor;
    frameWidth      = src.frameWidth;
    cdSet.clone(src.cdSet);
    pivotX          = src.pivotX;
    pivotY          = src.pivotY;
    magnification   = src.magnification;
    offsetX         = src.offsetX;
    offsetY         = src.offsetY;
    rotateAngle     = src.rotateAngle;
    visible         = src.visible;
  }
  // プロパティの複製（全部）
  void clone(viewportClass src) {
    vx              = src.vx;
    vy              = src.vy;
    vwidth          = src.vwidth;
    vheight         = src.vheight;
    monoRatio       = src.monoRatio;
    cloneAttr(src);
    // ユニークな情報（複製注意）
    label           = src.label;
    serialNumber    = src.serialNumber;
    layerLevel      = src.layerLevel;
  }
  // 現在の設定値でグラフィックスを初期化する
  void restore() {
    pg.beginDraw();
    if (cdSet.isFill) pg.fill(cdSet.fillColor, cdSet.fillTrans);
      else            pg.noFill();
    if (cdSet.isStroke) pg.stroke(cdSet.strokeColor, cdSet.strokeTrans);
      else              pg.noStroke();
    pg.strokeWeight(cdSet.strokeWeight);
    pg.rectMode(cdSet.rectMode);
    pg.endDraw();
  }

  // ピクセル比を設定する
  void setPixelRatio(float x, float y) {
    setPixelRatioX(x);
    setPixelRatioY(y);
  }
  void setPixelRatioX(float x) {
    if (x == 0.0) {
      println("エラー：指定されたピクセル比がゼロです．");
      exit();
    }
    pixelRatioX = x;
  }
  void setPixelRatioY(float y) {
    if (y == 0.0) {
      println("エラー：指定されたピクセル比がゼロです．");
      exit();
    }
    pixelRatioY = y;
  }
  
  // ピクセル比の計算
  void calcPixelRatio() {
    float px, py;
    px = vwidth  / (wx_max - wx_min);
    py = vheight / (wy_max - wy_min);
    if (monoRatio) {
      // 縦と横のピクセル比を同一に
      if (px > py) {
        // 縦優先
        setPixelRatio(py, py);
      } else {
        // 横優先
        setPixelRatio(px, px);
      }
    } else {
      // 別々
      setPixelRatio(px, py);
    }
  }
  // --------------------
  // ワールド座標系の設定
  // --------------------
  // 【参照】
  float world_minX() { return wx_min; }
  float world_maxX() { return wx_max; }
  float world_minY() { return wy_min; }
  float world_maxY() { return wy_max; }
  // 【変更】
  // (a) 全指定
  void world(float minx, float miny, float maxx, float maxy) {
    if ((minx >= maxx) || (miny >= maxy)) {
      println("ワールド座標系の設定が正しくありません（最小値と最大値が逆です）");
      exit();
    }
    wx_min = minx;
    wy_min = miny;
    wx_max = maxx;
    wy_max = maxy;
    calcPixelRatio();
  }
  // (b) ｘ，ｙそれぞれの最小値／最大値を個別に変更
  float world_minX(float x) {
    float oldvalue = wx_min;
    world(x, wy_min, wx_max, wy_max);
    return oldvalue;
  }
  float world_minY(float y) {
    float oldvalue = wy_min;
    world(wx_min, y, wx_max, wy_max);
    return oldvalue;
  }
  float world_maxX(float x) {
    float oldvalue = wx_max;
    world(wx_min, wy_min, x, wy_max);
    return oldvalue;
  }
  float world_maxY(float y) {
    float oldvalue = wy_max;
    world(wx_min, wy_min, wx_max, y);
    return oldvalue;
  }
  // (c) ピクセル比のみを指定
  // (c-1) ビューポート中央を原点
  void worldOrigin() { worldOrigin(pixelRatioX); }
  void worldOrigin(float ratio) {
    setPixelRatio(ratio, ratio);
    float dwx, dwy;
    dwx = vwidth  / 2.0 / ratio;
    dwy = vheight / 2.0 / ratio;
    world(-dwx, -dwy, dwx, dwy);
  }
  // (c-2) ビューポート左上を原点
  void worldLU() { worldLU(pixelRatioX); }
  void worldLU(float ratio) {
    setPixelRatio(ratio, ratio);
    float dwx, dwy;
    dwx = vwidth  / ratio;
    dwy = vheight / ratio;
    world(0.0, -dwy, dwx, 0.0);
  }
  // (c-2) ビューポート左下を原点
  void worldLL() { worldLL(pixelRatioX); }
  void worldLL(float ratio) {
    setPixelRatio(ratio, ratio);
    float dwx, dwy;
    dwx = vwidth  / ratio;
    dwy = vheight / ratio;
    world(0.0, 0.0, dwx, dwy);
  }
  // (c-3) ビューポート右上を原点
  void worldRU() { worldRU(pixelRatioX); }
  void worldRU(float ratio) {
    setPixelRatio(ratio, ratio);
    float dwx, dwy;
    dwx = vwidth  / ratio;
    dwy = vheight / ratio;
    world(-dwx, -dwy, 0.0, 0.0);
  }
  // (c-4) ビューポート右下を原点
  void worldRL() { worldRL(pixelRatioX); }
  void worldRL(float ratio) {
    setPixelRatio(ratio, ratio);
    float dwx, dwy;
    dwx = vwidth  / ratio;
    dwy = vheight / ratio;
    world(-dwx, 0.0, 0.0, dwy);
  }
  // (d) ワールド座標の中心座標とピクセル比を指定
  // つまりこの指定した座標がビューポートの中央に表示されるようにマッピングされる
  void worldCenter(float x, float y) { worldCenter(x, y, pixelRatioX); }
  void worldCenter(float x, float y, float ratio) {
    setPixelRatio(ratio, ratio);
    worldOrigin();
    world(wx_min + x, wy_min + y, wx_max + x, wy_max + y);
  }

  // ---------------------------------------------------
  // ワールド座標系からビューポートのPixel座標系への変換
  // ---------------------------------------------------
  // a) 単にピクセル比を掛けたもの（半径や高さ／幅用）
  int _px(float x) { return int(x * magnification * pixelRatioX); }
  int _py(float y) { return int(y * magnification * pixelRatioY); }
  // b) 座標変換
  int px(float x) {
//  float center = (wx_max + wx_min) / 2.0 - offsetX;
//  return int((x - center) * magnification * pixelRatioX) + (vwidth / 2);
    float center = (wx_max + wx_min) / 2.0;
    return int((x - center) * pixelRatioX) + (vwidth / 2);
  }
  int py(float y) {
//  float center = (wy_max + wy_min) / 2.0 - offsetY;
//  return vheight - (int((y - center) * magnification * pixelRatioY) + (vheight / 2));
    float center = (wy_max + wy_min) / 2.0;
    return vheight - (int((y - center) * pixelRatioY) + (vheight / 2));
  }
  // ビューポートに描画する最終処理（拡大縮小・並進・回転）
  int [] baseConv(int [] p, float x, float y) {
    // 一旦、ピボット座標を原点とし、Y軸正が上方向の座標系に変換する
    float cx, cy;
    cx = pivotX;
    cy = pivotY;
    x -= cx;
    y -= cy;
    y  = -y;
    // 拡大縮小
    x *= magnification;
    y *= magnification;
    // 回転
    float rx, ry;
    rx = x * cos(rotateAngle) - y * sin(rotateAngle);
    ry = x * sin(rotateAngle) + y * cos(rotateAngle);
    // 元の座標に戻す
    rx += cx;
    ry  = -ry;
    ry += cy;
    // オフセット
    rx += offsetX;
    ry += offsetY;
    p[0] = int(rx);
    p[1] = int(ry);
    return p;
  }
 
  // --------------------------------------------------
  // ビューポートの表示／非表示のダイナミックな切り替え
  // --------------------------------------------------
  // 状態
  boolean isViewVisible() { return visible; }
  // (a) 表示
  void viewVisible()   { visible = true;  }
  // (b) 非表示
  void viewUnvisible() { visible = false; }
  // (c) 引数による指定
  void viewVisible(boolean flag) { if (flag) viewVisible(); else viewUnvisible(); }
  // (d) トグル動作
  boolean toggleViewVisible() {
    if (isViewVisible()) {
      viewUnvisible();
      return false;
    } else {
      viewVisible();
      return true;
    }
  }
  // ビューポートの移動
  void moveView(int x, int y)      { vx  = x;  vy  = y;  }
  void moveViewRel(int dx, int dy) { vx += dx; vy += dy; }

  // ----------------------------------
  // ビューポートの枠線や不透明度の設定
  // ----------------------------------
  // フレームカラー
  // (a-1) フレームカラーを取得（現在のフレームのみ）
  int frameColor() { return frameColor; }
  // (a-2) フレームカラーの設定（＋変更前の値）
  int frameColor(int c) {
    int oldcolor = frameColor();
    frameColor = c & #ffffff;
    return oldcolor;
  }
  // フレームの表示／非表示
  // (b) フレームの表示／非表示
  void drawFrame(boolean flag) { drawFrame = flag; }
  // フレーム幅
  // (c-1) フレーム幅を取得
  int frameWidth() { return frameWidth; }
  // (c-2) フレーム幅の設定（＋変更前の値）
  int frameWidth(int width) {
    int oldwidth = frameWidth();
    frameWidth = width;
    return oldwidth;
  }

  // (a) 不透明度の取得
  int transparency() { return transparency; }
  // (b) 不透明度の設定（＋変更前の値）
  int transparency(int trans) {
    int oldtrans = transparency();
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    transparency = cdSet.fillTrans = cdSet.strokeTrans = trans;  // 塗り潰しや線の不透明度も変更する
    return oldtrans;
  }

  // 背景色の設定
  // (a) 色を取得（不透明度は無視）
  int viewBgColor() { return bgColor; }
  // (b) 色を設定（不透明度は255）
  int viewBgColor(int c) {
    int oldcolor = viewBgColor();
    bgColor = c & #ffffff;
    return oldcolor;
  }

  // 背景を透明なビューポートにする／しない
  void transparentView(boolean flag) {
    if (flag) drawFrame = false;  // 透明なビューを指定されたら一緒に枠も消してしまう（お節介？）
    transparentView = flag;
  }

  // ----------------------------------------------
  // ピボット（拡大縮小および回転の中心座標）の設定
  // ----------------------------------------------
  // (a) 現在のピボット座標を返す
  PVector viewPivot() {
    PVector vp = new PVector(pivotX, pivotY);
    return vp;
  }
  // (b) ピボット座標の設定（＋現在の値）
  // (b-1) 自由に指定
  PVector viewPivot(float x, float y) {
    PVector vp = viewPivot();
    pivotX = x; 
    pivotY = y;
    return vp;
  }
  // (b-2) ビューポート中心
  PVector viewPivotCenter() { return viewPivot(vwidth / 2, vheight / 2); }
  // (b-3) 左上，左下，右上，右下は必要か？  

  // --------
  // 倍率関係
  // --------
  // (a) 現在の倍率を返す
  float viewScale() { return magnification; }
  // (b) 倍率の値を変更する（＋変更前の値を返す）
  float viewScale(float s) {
    float oldscale;
    if (s <= 0.0) s = 1.0;  // 不正な値
    oldscale = viewScale();
    magnification = s;
    return oldscale;
  }
  // ----------------------------
  // ビューポートの視点の回転角度
  // ----------------------------
  // (a) 現在の角度を返す
  float viewRotate() { return rotateAngle; }
  // (b) 視点の角度を設定する（＋変更前の値を返す）
  float viewRotate(float a) {
    float oldangle = viewRotate();
    if (a > TWO_PI) a -= TWO_PI; else if (a < -TWO_PI) a += TWO_PI;  // % を使うと誤差が生じる？という根拠のない小心な推測
    rotateAngle = a;
    return oldangle;
  }
  // (c) 視点の角度を増減する（＋変更前の値を返す）
  float viewRotateRel(float a) { return viewRotate(viewRotate() + a); }
  // ----------------
  // オフセット量関係
  // ----------------
  // (a) 現在のビューポートのオフセット量を返す
  // (a-1) ｘ方向
  float viewOffsetX() { return offsetX; }
  // (a-2) ｙ方向
  float viewOffsetY() { return offsetY; }
  // (b) オフセット量を変更する（＋変更前の値を返す）：現在のビューポート
  // (b-1) ｘ方向
  float viewOffsetX(float offset) {
    float oldoffset = viewOffsetX();
    offsetX = offset;
    return oldoffset;
  }
  // (b-2) ｙ方向
  float viewOffsetY(float offset) {
    float oldoffset = viewOffsetY();
    offsetY = offset;
    return oldoffset;
  }
  // (b-3) ｘ方向／ｙ方向同時：常にtrueを返す
  boolean viewOffset(float offx, float offy) {
    viewOffsetX(offx);
    viewOffsetY(offy);
    return true;
  }

  // ------
  // 色関係
  // ------
  // (a) fill()
  // (a-1) 現在の塗り潰し色を取得
  int fill() { return cdSet.fillColor; }
  // (a-2) 塗り潰し色を変更（＋現在の値を返す）
  int fill(int c) { return this.fill(c, cdSet.fillTrans); }
  // (a-3) 塗り潰し色を変更（不透明度対応＋現在の値を返す）
  int fill(int c, int trans) {
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    int oldcolor;
    oldcolor = this.fill();
    cdSet.fillColor = c & #ffffff;
    cdSet.fillTrans = trans;
    pg.beginDraw();
    pg.fill(cdSet.fillColor, cdSet.fillTrans);
    pg.endDraw();
    cdSet.isFill = false;
    return oldcolor;
  }
  // (a-4) 塗り潰し無し
  void noFill() {
    cdSet.isFill = false;
    pg.beginDraw();
    pg.noFill();
    pg.endDraw();
  }
  // (a-5) 再び塗り潰し
  void reFill() {
    this.fill(cdSet.fillColor, cdSet.fillTrans);
  }
  // (b) stroke()
  // (b-1) 現在の線色を取得
  int stroke() { return cdSet.strokeColor; }
  // (b-2) 線色を変更（＋現在の値を返す）
  int stroke(int c) { return this.stroke(c, cdSet.fillTrans); }
  // (b-3) 線色を変更（不透明度対応＋現在の値を返す）
  int stroke(int c, int trans) {
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    int oldcolor;
    oldcolor = this.stroke();
    cdSet.strokeColor = c & #ffffff;
    cdSet.strokeTrans = trans;
    pg.stroke(cdSet.strokeColor, cdSet.strokeTrans);
    return oldcolor;
  }
  // (b-4) 線無し
  void noStroke() {
    cdSet.isStroke = false;
    pg.beginDraw();
    pg.noStroke();
    pg.endDraw();
  }
  // (b-5) 再び線有り
  void reStroke() {
    this.stroke(cdSet.strokeColor, cdSet.strokeTrans);
  }
  // (c) 背景色
  int getBgColor() {
    return bgColor;
  }
  // 線幅
  // (a) 現在値読み出し
  float strokeWeight() { return cdSet.strokeWeight; }
  // (b) 設定（＋現在の値）
  float strokeWeight(float width) {
    float oldwidth = strokeWeight();
    pg.beginDraw();
    pg.strokeWeight(width);
    pg.endDraw();
    cdSet.strokeWeight = width;
    return oldwidth;
  }

  // 現在のビューポートを画面消去する
  void clrView() {
    pg.beginDraw();
    if (transparentView) pg.background(bgColor, 0);
      else               pg.background(bgColor, transparency);
    pg.endDraw();
  }

  // --------------
  // 座標を移動する
  // --------------
  void _moveTo(int x, int y) {
    cdSet.vpx = x;
    cdSet.vpy = y;
  }
  void moveTo(float x, float y) {
    cdSet.wpx = x;
    cdSet.wpy = y;
  }
  void _moveToRel(int dx, int dy) {
    cdSet.vpx += dx;
    cdSet.vpy += dy;
  }
  void moveToRel(float dx, float dy) {
    cdSet.wpx += dx;
    cdSet.wpy += dy;
  }
  // 実際に描画する関数
  // 共有する作業用変数
  int [] tmp1 = new int[2];
  int [] tmp2 = new int[2];
  int [] tmp3 = new int[2];
  int [] tmp4 = new int[2];
//  int [] tmp1, tmp2, tmp3, tmp4;
  // ------------
  // 直線描画関係
  // ------------
  // 直線を描く（ピクセル座標）
  void _line(int x1, int y1, int x2, int y2) {
    tmp1 = baseConv(tmp1, x1, y1);
    tmp2 = baseConv(tmp2, x2, y2);
    pg.beginDraw();
    pg.line(tmp1[0], tmp1[1], tmp2[0], tmp2[1]);
    pg.endDraw();
    _moveTo(x2, y2);
  }
  // 直線を描く（ワールド）
  void line(float x1, float y1, float x2, float y2) {
    _line(px(x1), py(y1), px(x2), py(y2));
    moveTo(x2, y2);
  }
  // 始点省略（絶対座標）
  void _lineTo(int x, int y)    { _line(cdSet.vpx, cdSet.vpy, x, y); }
  void lineTo(float x, float y) { this.line(cdSet.wpx, cdSet.wpy, x, y); }
  // 始点省略（相対座標）
  void _lineRel(int dx, int dy)    { _line(cdSet.vpx, cdSet.vpy, cdSet.vpx + dx, cdSet.vpy + dy); }
  void lineRel(float dx, float dy) { this.line(cdSet.wpx, cdSet.wpy, cdSet.wpx + dx, cdSet.wpy + dy); }
  // ----------
  // 点線を描く
  // ----------
  // (a) 絶対座標指定
  // (a-1) ピクセル座標
  void _dashedLine(int x0, int y0, int x1, int y1, int step) {
    float  px0, py0, px1, py1, normv;
    float  unitx, unity;
    float  dx, dy;
    int    countmax, count;
    boolean stroke = true;

    countmax = int(norm(x0, y0, x1, y1) / step);
    if (countmax <= 0) {
      _line(x0, y0, x1, y1);
      return;
    }
    px0 = x0;
    py0 = y0;
    for (count = 0; count < countmax; count++) {
      dx = x1 - px0;
      dy = y1 - py0;
      normv = sqrt(dx * dx + dy * dy);
      if (normv < step) {
        _line(int(px0), int(py0), x1, y1);
        return;
      }
      unitx = dx / normv;
      unity = dy / normv;
      px1 = px0 + unitx * step;
      py1 = py0 + unity * step;
      if (stroke) _line(int(px0), int(py0), int(px1), int(py1));
      stroke = !stroke;
      px0 = px1;
      py0 = py1;
    }
    _moveTo(x1, y1);
  }
  // (a-2) ワールド座標系
  void dashedLine(float x1, float y1, float x2, float y2, float step) {
    _dashedLine(px(x1), py(y1), px(x2), py(y2), int((_px(step) + _py(step)) / 2.0));
    moveTo(x2, y2);
  }
  void dashedLine(float x1, float y1, float x2, float y2, int step) {
    _dashedLine(px(x1), py(y1), px(x2), py(y2), step);
    moveTo(x2, y2);
  }
  // (b) 始点省略（絶対座標）
  void _dashedLineTo(int x, int y, int step)      { _dashedLine(cdSet.vpx, cdSet.vpy, x, y, step); }
  void dashedLineTo(float x, float y, float step) { dashedLine(cdSet.wpx, cdSet.wpy, x, y, step);  }
  // (c) 始点省略（相対座標）
  void _dashedLineRel(int dx, int dy, int step)      { _dashedLine(cdSet.vpx, cdSet.vpy, cdSet.vpx + dx, cdSet.vpy + dy, step); }
  void dashedLineRel(float dx, float dy, float step) { dashedLine(cdSet.wpx, cdSet.wpy, cdSet.wpx + dx, cdSet.wpy + dy, step);  }

  // --------
  // 円を描く
  // --------
  // (a) 楕円を描く
  void _ellipse(int x, int y, int w, int h) {
    tmp1 = baseConv(tmp1, x, y);
    pg.beginDraw();
    pg.ellipse(tmp1[0], tmp1[1], w, h);
    pg.endDraw();
    _moveTo(x, y);
  }
  void ellipse(float x, float y, float w, float h) {
    _ellipse(px(x), py(y), _px(w), _py(h));
    moveTo(x, y);
  }
  void ellipse(float x, float y, int w, int h) {
    _ellipse(px(x), py(y), w, h);
    moveTo(x, y);
  }
  // (b) 真円を描く
  void _ellipse(int x, int y, int r)      { _ellipse(x, y, r, r); }
  void ellipse(float x, float y, float r) { this.ellipse(x, y, r, r); }
  void ellipse(float x, float y, int r)   { this.ellipse(x, y, r, r); }
  // (c) 真円を描く（半径のみ指定）
  void _ellipse(int  r) { _ellipse(cdSet.vpx, cdSet.vpy, r); }
  void ellipse(float r) { ellipse(cdSet.wpx, cdSet.wpy,  r); }
  void ellipse(int   r) { ellipse(cdSet.wpx, cdSet.wpy,  r); }

  // -----------
  // 三角形を描く
  // -----------
  // (a) 三点指定
  void _triangle(int x1, int y1, int x2, int y2, int x3, int y3) {
    tmp1 = baseConv(tmp1, x1, y1);
    tmp2 = baseConv(tmp2, x2, y2);
    tmp3 = baseConv(tmp3, x3, y3);
    pg.beginDraw();
    pg.triangle(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1]);
    pg.endDraw();
    _moveTo(int((x1 + x2 + x3) / 3.0), int((y1 + y2 + y3) / 3.0));  // 中央，だと思う
  }
  void triangle(float x1, float y1, float x2, float y2, float x3, float y3) {
    _triangle(px(x1), py(y1), px(x2), py(y2), px(x3), py(y3));
    moveTo((x1 + x2 + x3) / 3.0, (y1 + y2 + y3) / 3.0);
  }
  // 中心座標を指定（回転対応）
  void _triangle(int x, int y, int s, float angle) {
    int   dx1, dy1, dx2, dy2, dx3, dy3;
    float da;
    da  = TWO_PI / 3.0;
    dx1 = 0;
    dy1 = s;
    dx2 = int(rotate2Dx(dx1, dy1, angle + da));
    dy2 = int(rotate2Dy(dx1, dy1, angle + da));
    dx3 = int(rotate2Dx(dx1, dy1, angle - da));
    dy3 = int(rotate2Dy(dx1, dy1, angle - da));
    dx1 = int(rotate2Dx(dx1, dy1, angle));
    dy1 = int(rotate2Dy(dx1, dy1, angle));
    _triangle(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3);
    _moveTo(x, y);
  }
  void triangle(float x, float y, float s, float angle) {
    float dx1, dy1, dx2, dy2, dx3, dy3, da;
    da  = TWO_PI / 3.0;
    dx1 = 0;
    dy1 = s;
    dx2 = rotate2Dx(dx1, dy1, angle + da);
    dy2 = rotate2Dy(dx1, dy1, angle + da);
    dx3 = rotate2Dx(dx1, dy1, angle - da);
    dy3 = rotate2Dy(dx1, dy1, angle - da);
    dx1 = rotate2Dx(dx1, dy1, angle);
    dy1 = rotate2Dy(dx1, dy1, angle);
    this.triangle(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3);
    moveTo(x, y);
  }
  // 中心座標を指定（回転なし）
  void _triangle(int x, int y, int s)         { _triangle(x, y, s, 0.0); }
  void  triangle(float x, float y, float s)   {  triangle(x, y, s, 0.0); }
  // サイズのみ指定（回転対応）
  void _triangle(int   s, float angle) { _triangle(cdSet.vpx, cdSet.vpy, s, angle); }
  void  triangle(float s, float angle) {  triangle(cdSet.wpx, cdSet.wpy, s, angle); } 
  // サイズのみ指定（回転なし）
  void _triangle(int   s) { _triangle(s, 0.0); }
  void  triangle(float s) {  triangle(s, 0.0); }
  
  // ------------
  // 四辺形を描く
  // ------------
  void _quad(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
    tmp1 = baseConv(tmp1, x1, y1);
    tmp2 = baseConv(tmp2, x2, y2);
    tmp3 = baseConv(tmp3, x3, y3);
    tmp4 = baseConv(tmp4, x4, y4);
    pg.beginDraw();
    pg.quad(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1], tmp4[0], tmp4[1]);
    pg.endDraw();
    float [][] pt = new float[4][2];
    pt[0][0] = x1;
    pt[0][1] = y1;
    pt[1][0] = x2;
    pt[1][1] = y1;
    pt[2][0] = x3;
    pt[2][1] = y3;
    pt[3][0] = x4;
    pt[3][1] = y4;
    float [] xyG = new float[2];
    xyG = polygonG(pt);
    _moveTo(int(xyG[0]), int(xyG[1]));
  }
  void quad(float x1, float y1, float x2, float y2, float x3, float y3, float x4, float y4) {
    _quad(px(x1), py(y1), px(x2), py(y2), px(x3), py(y3), px(x4), py(y4));
    float [][] pt = new float[4][2];
    pt[0][0] = x1;
    pt[0][1] = y1;
    pt[1][0] = x2;
    pt[1][1] = y1;
    pt[2][0] = x3;
    pt[2][1] = y3;
    pt[3][0] = x4;
    pt[3][1] = y4;
    float [] xyG = new float[2];
    xyG = polygonG(pt);
    moveTo(xyG[0], xyG[1]);
  }
  // ------------
  // 長方形を描く
  // ------------
  // (a) 左上始点
  void _rect(int x, int y, int w, int h) {
/*
    tmp1 = baseConv(tmp1, x, y);  // 回転に正しく対応できていない
    pg.beginDraw();
    pg.rect(tmp1[0], tmp1[1], w, h);
    pg.endDraw();
*/
    switch (cdSet.rectMode) {
      case CORNERS :
        _quad(x, y, x, h, w, h, w, y);
        break;
      case CENTER :
        int dx = w / 2;
        int dy = h / 2;
        _quad(x - dx, y - dy, x - dx, y + dy, x + dx, y + dy, x + dx, y - dy);
        break;
      case RADIUS :
        _quad(x - w,  y - h,  x - w,  y + h,  x + w,  y + h,  x + w,  y - h );
        break;
      default :
        _quad(x, y, x, y + h, x + w, y + h, x + w, y);
        break;
    }
    _moveTo(x, y);
  }
  void rect(float x, float y, float w, float h) {
    _rect(px(x), py(y), _px(w), _py(h));
    moveTo(x, y);
  }
  // (b) それ以外の指定方法はrectMode()で指定すれば良いので省略
  // CORNER（デフォルト）, CORNERS, CENTER, RADIUSが使える
  void rectMode(int mode) {
    pg.beginDraw();
    pg.rectMode(mode);
    pg.endDraw();
    cdSet.rectMode = mode;
  }
  // ---------
  // 円弧を描く
  // ---------
  void _arc(int x, int y, int w, int h, float start, float stop) {
    tmp1 = baseConv(tmp1, x, y);  // 回転に正しく対応できていない
    pg.beginDraw();
    pg.arc(tmp1[0], tmp1[1], w, h, start, stop);
    pg.endDraw();
    _moveTo(x, y);
  }
  void arc(float x, float y, float w, float h, float start, float stop) {
    _arc(px(x), py(y), _px(w), _py(h), start, stop);
    moveTo(x, y);
  }
  // ------------
  // 点を描画する
  // ------------
  void _point(int x, int y) {
    tmp1 = baseConv(tmp1, x, y);
    pg.beginDraw();
    pg.point(tmp1[0], tmp1[1]);
    pg.endDraw();
    _moveTo(x, y);
  }
  void point(float x, float y) {
    _point(px(x), py(y));
    moveTo(x, y);
  }
  // --------
  // 頂点関係
  // --------
  void beginShape(int mode) {
    pg.beginDraw();
    pg.beginShape(mode);
    pg.endDraw();
  }
  void beginShape() {
    pg.beginDraw();
    pg.beginShape();
    pg.endDraw();
  }
  void endShape() {
    pg.beginDraw();
    pg.endShape();
    pg.endDraw();
  }
  void _vertex(int   x, int y) {
    tmp1 = baseConv(tmp1, x, y);  // 回転に正しく対応できていない
    pg.beginDraw();
    pg.vertex(tmp1[0], tmp1[1]);
    pg.endDraw();
    _moveTo(x, y);
  }
  void  vertex(float x, float y) {
    _vertex(px(x), py(y));
    moveTo(x, y);
  }
  void _curveVertex(int x, int y) {
    tmp1 = baseConv(tmp1, x, y);
    pg.beginDraw();
    pg.curveVertex(tmp1[0], tmp1[1]);
    pg.endDraw();
    _moveTo(x, y);
  }
  void curveVertex(float x, float y) {
    _curveVertex(px(x), py(y));
    moveTo(x, y);
  }
  void _bezierVertex(int cx1, int cy1, int cx2, int cy2, int x, int y) {
    tmp1 = baseConv(tmp1, cx1, cy1);
    tmp2 = baseConv(tmp2, cx2, cy2);
    tmp3 = baseConv(tmp3, x,   y  );
    pg.beginDraw();
    pg.bezierVertex(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1]);
    pg.endDraw();
    _moveTo(x, y);
  }
  void bezierVertex(float cx1, float cy1, float cx2, float cy2, float x, float y) {
    _bezierVertex(px(cx1), py(cy1), px(cx2), py(cy2), px(x), py(y));
    moveTo(x, y);
  }
  void texture(PImage img) {
    pg.beginDraw();
    pg.texture(img);
    pg.endDraw();
  }
  void textureMode(int mode) {
    pg.beginDraw();
    pg.textureMode(mode);
    pg.endDraw();
  }

  // --------
  // 曲線関係
  // --------
  void _curve(int a1x, int a1y, int x1, int y1, int x2, int y2, int a2x, int a2y) {
    tmp1 = baseConv(tmp1, a1x, a1y);
    tmp2 = baseConv(tmp2, x1,  y1 );
    tmp3 = baseConv(tmp3, x2,  y2 );
    tmp4 = baseConv(tmp4, a2x, a2y);
    pg.beginDraw();
    pg.curve(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1], tmp4[0], tmp4[1]);
    pg.endDraw();
    _moveTo(x2, y2);
  }
  void curve(float a1x, float a1y, float x1, float y1, float x2, float y2, float a2x, float a2y) {
    curve(px(a1x), py(a1y), px(x1), py(y1), px(x2), py(y2), px(a2x), py(a2y));
    moveTo(x2, y2);
  }
  void _curveTo(int a1x, int a1y, int x2, int y2, int a2x, int a2y) { _curve(a1x, a1y, cdSet.vpx, cdSet.vpy, x2, y2, a2x, a2y); }
  void curveTo(float a1x, float a1y, float x2, float y2, float a2x, float a2y) { curve(a1x, a1y, cdSet.wpx, cdSet.wpy, x2, y2, a2x, a2y); }
  void curveTightness(float squishy) {
    pg.beginDraw();
    pg.curveTightness(squishy);
    pg.endDraw();
  }
  void curveDetail(int detail) {
    pg.beginDraw();
    pg.curveDetail(detail);
    pg.endDraw();
  }

  // ----------------
  // テキスト表示関係
  // ----------------
  void prepareFont() {
    prepareFont(cdSet.textSize);
  }
  void prepareFont(int fontsize) {
    prepareFont(_FONTNAME, fontsize);
  }
  void prepareFont(String fontname, int fontsize) {
    if (crowbar.status != 0) {
      println("エラー: フォントの宣言はOptions()かSetup()で行って下さい，");
      return;
    }
    pg.beginDraw();
    font = createFont(fontname, fontsize, false);  // フォントを変換
    pg.textFont(font);         // フォントを設定
//  pg.textMode(SCREEN);       // SCREENに設定すると表示されない
    pg.textSize(fontsize);     // フォントサイズを設定（不要？）
    pg.textAlign(cdSet.textAlignH, cdSet.textAlignV);   // 配置
//  pg.textLeading(cdSet.textLeading);
    pg.endDraw();
    this.textLeading(int((this.textAscent() + this.textDescent()) * 1.2));
    cdSet.textSize = fontsize;
  }
  // フォントサイズ
  int textSize() { return cdSet.textSize; }
  int textSize(int size) {
    int oldsize = cdSet.textSize;
    pg.beginDraw();
    pg.textSize(size);
    pg.endDraw();
    cdSet.textSize = size;
    return oldsize;
  }
  // 割り当て
  int textAlign()  { return cdSet.textAlignH; }
  int textAlignH() { return cdSet.textAlignH; }
  int textAlignV() { return cdSet.textAlignV; }
  int textAlign(int align) {
    int oldalign = cdSet.textAlignH;
    pg.beginDraw();
    pg.textAlign(align);
    pg.endDraw();
    cdSet.textAlignH = align;
    return oldalign;
  }
  void textAlign(int align, int yalign) {
    pg.beginDraw();
    pg.textAlign(align, yalign);
    pg.endDraw();
    cdSet.textAlignH = align;
    cdSet.textAlignV = yalign;
  }
  // 行間
  int textLeading() { return cdSet.textLeading; }
  int textLeading(int dist) {
    int oldleading = cdSet.textLeading;
    pg.beginDraw();
    pg.textLeading(dist);
    pg.endDraw();
    cdSet.textLeading = dist;
    return oldleading;
  }
  // ベースラインからの距離
  float textAscent() {
    float h;
    pg.beginDraw();
    h = pg.textAscent();
    pg.endDraw();
    return h;
  }
  float textDescent() {
    float h;
    pg.beginDraw();
    h = pg.textDescent();
    pg.endDraw();
    return h;
  }
  float textWidth(String str) {
    float width;
    pg.beginDraw();
    width = pg.textWidth(str);
    pg.endDraw();
    return width;
  }
  // 実際に描画する
  float _text(String str, int x, int y) {
    tmp1 = baseConv(tmp1, x, y);  // 回転に正しく対応できていない
    pg.beginDraw();
    pg.fill(cdSet.textColor);
    pg.text(str, tmp1[0], tmp1[1]);
    pg.fill(cdSet.fillColor);
    _moveTo(x + int(pg.textWidth(str)), y);
    pg.endDraw();
    return this.textWidth(str);
  }
  float _text(String str) {
    return _text(str, cdSet.vpx, cdSet.vpy);
  }
  float text(String str, float x, float y) {
    float width;
    width = _text(str, px(x), py(y)) / pixelRatioX;
    moveTo(x + width, y);
    return width;
  }
  float text(String str) {
    return _text(str);
  }
  // 文字色関係
  // (a) 文字色の取得
  int viewTextColor() {
    return cdSet.textColor;
  }
  // (b) 文字色の設定（＋変更前の値を返す）
  int viewTextColor(int c) {
    int oldcolor = this.viewTextColor();
    cdSet.textColor = c & #ffffff;
    return oldcolor;
  }

  // --------
  // 画像関係
  // --------
  // 読み込み（loadImage()はPGraphicsを使う必要は無い．なお，グローバルで実行してもダメ）
  // モード (CORNER, CORNERS)
  void imageMode(int MODE) {
    pg.beginDraw();
    pg.imageMode(MODE);
    pg.endDraw();
  }
  // 出力
  // (a) ピクセル座標
  void _image(PImage img, float x, float y) {
    pg.beginDraw();
    pg.image(img, x, y);
    pg.endDraw();
  }
  void _image(PImage img, float x, float y, float width, float height) {
    pg.beginDraw();
    pg.image(img, x, y, width, height);
    pg.endDraw();
  }
  // (b) ワールド座標
  void image(PImage img, float x, float y) {
    _image(img, px(x), py(y));
  }
  void image(PImage img, float x, float y, float width, float height) {
    _image(img, px(x), py(y), _px(width), _py(height));
  }
  void tint(float gray) {
    pg.beginDraw();
    pg.tint(gray);
    pg.endDraw();
  }
  void tint(float gray, float alpha) {
    pg.beginDraw();
    pg.tint(gray, alpha);
    pg.endDraw();
  }
  void tint(float value1, float value2, float value3) {
    pg.beginDraw();
    pg.tint(value1, value2, value3);
    pg.endDraw();
  }
  void tint(color c) {
    pg.beginDraw();
    pg.tint(c);
    pg.endDraw();
  }
  void tint(color c, float alpha) {
    pg.beginDraw();
    pg.tint(c, alpha);
    pg.endDraw();
  }
/*
  void tint(int hex) {  // void tint(color c) と重複
    pg.beginDraw();
    pg.tint(hex);
    pg.endDraw();
  }
  void tint(int hex, float alpha) {  // void tint(color c, float alpha) と重複
    pg.beginDraw();
    pg.tint(hex, alpha);
    pg.endDraw();
  }
*/
  void noTint() {
    pg.beginDraw();
    pg.noTint();
    pg.endDraw();
  }

  // ピクセル関連
  PImage _createImage(int w, int h, int mode) {  // 必要は無いが統一性の維持のためだけに用意
    return createImage(w, h, mode);  // mode : RGB, ARGB, ALPHA(grayscale alpha channel)
  }
  PImage createImage(float w, float h, int mode) {
    return createImage(_px(w), _py(h), mode);
  }
  void loadPixels() {
    pg.beginDraw();
    pg.loadPixels();
    pg.endDraw();
    pixels = pg.pixels;
  }
  void updatePixels() {
    pg.beginDraw();
    pg.updatePixels();
    pg.endDraw();
  }
  PImage _get() {
    PImage img;
    pg.beginDraw();
    img = pg.get();
    pg.endDraw();
    return img;
  }
  color _get(int x, int y) {
    color pixel;
    pg.beginDraw();
    pixel = pg.get(x, y);
    pg.endDraw();
    return pixel;
  }
  PImage _get(int x, int y, int w, int h) {
    PImage img;
    pg.beginDraw();
    img = pg.get(x, y, w, h);
    pg.endDraw();
    return img;
  }
  PImage get() { return _get(); }
  color  get(float x, float y) { return _get(px(x), py(y)); }
  PImage get(float x, float y, float w, float h) { return _get(px(x), py(y), _px(w), _py(h)); }
  void _set(int x, int y, color c) {
    pg.beginDraw();
    pg.set(x, y, c);
    pg.endDraw();
  }
  void _set(int x, int y, PImage img) {
    pg.beginDraw();
    pg.set(x, y, img);
    pg.endDraw();
  }
  void set(float x, float y, color c)    { _set(px(x), py(y), c); }
  void set(float x, float y, PImage img) { _set(px(x), py(y), img); }
  void filter(int mode) {
    pg.beginDraw();
    pg.filter(mode);
    pg.endDraw();
  }
  void filter(int mode, float level) {
    pg.beginDraw();
    pg.filter(mode, level);
    pg.endDraw();
  }
  // copy() : ビューポート間をまたぐコピーは考えていない．自力で頑張れ．コピー先のサイズ指定省略版は用意した．
  void _copy(int x, int y, int w, int h, int dx, int dy, int dw, int dh) {
    pg.beginDraw();
    pg.copy(x, y, w, h, dx, dy, dw, dh);
    pg.endDraw();
  }
  void _copy(int x, int y, int w, int h, int dx, int dy) { _copy(x, y, w, h, dx, dy, w, h); }
  void _copy(PImage img, int x, int y, int w, int h, int dx, int dy, int dw, int dh) {
    pg.beginDraw();
    pg.copy(img, x, y, w, h, dx, dy, dw, dh);
    pg.endDraw();
  }
  void _copy(PImage img, int x, int y, int w, int h, int dx, int dy) { _copy(img, x, y, w, h, dx, dy, w, h); }
  void copy(float x, float y, float w, float h, float dx, float dy, float dw, float dh) {
    pg.beginDraw();
    pg.copy(px(x), py(y), _px(w), _py(h), px(dx), py(dy), _px(dw), _py(dh));
    pg.endDraw();
  }
  void copy(float x, float y, float w, float h, float dx, float dy) { copy(x, y, w, h, dx, dy, w, h); }
  void copy(PImage img, float x, float y, float w, float h, float dx, float dy, float dw, float dh) {
    pg.beginDraw();
    pg.copy(img, px(x), py(y), _px(w), _py(h), px(dx), py(dy), _px(dw), _py(dh));
    pg.endDraw();
  }
  void copy(PImage img, float x, float y, float w, float h, float dx, float dy) { copy(img, x, y, w, h, dx, dy, w, h); }

  // ビューポートのキャプチャ（動作未確認）：もしかしたらプロセスコントロールが必要かも知れない
  void save(String fname) {
    pg.beginDraw();
    pg.save(fname);
    pg.endDraw();
  }
/*
  void saveFrame() {    // saveFrame()はPGraphicsに存在しないようだ
    pg.beginDraw();
    pg.saveFrame();
    pg.endDraw();
  }
  void saveFrame(String format) {  // saveFrame()はPGraphicsに存在しないようだ
    pg.beginDraw();
    pg.saveFrame(format);
    pg.endDraw();
  }
*/
}

// Tomahawk本体クラス（Crowbarに継承されます）
class tomahawkClass {
  protected int               viewMax;       // 宣言済みのビューポートの数
  protected int               cvNumber;      // 現在注目しているビューポートの番号
  protected viewportClass     cv;            // 現在注目しているビューポートクラス
  protected viewportClass []  viewStack;     // 宣言済みビューポート
  protected viewportClass []  cloneStack;    // Options(), Setup()で宣言した状態をバックアップ＆リストアするための領域
  protected int []            layerStack;    // ビューポートの表示順のスタック(viewLayerPush(), viewLayerPop()専用）
  protected int               tomahawkMode;  // mode == 0: noTomahawk, 1: Basic, 2: Advanced
  tomahawkClass(int mode) {
//  background(_BGCOLOR);  // Crowbarで指定する背景色で初期化
    viewStack    = (viewportClass [])new viewportClass[1];  // 配列を一つだけ確保する（あとで宣言される度に拡張する）
    viewStack[0] = null;
    cloneStack    = (viewportClass [])new viewportClass[1];  // 配列を一つだけ確保する（あとで宣言される度に拡張する）
    cloneStack[0] = null;
    cvNumber     = 0;
    cv           = null;
    viewMax      = 0;
    layerStack   = null;
    tomahawkMode = mode;
  }
  // 現在のビューポートの状態をバックアップ
  void backupViewports() {
    int i;
    if (cloneStack == null) return;
    for (i = 0; i < viewMax; i++) cloneStack[i].clone(viewStack[i]);
  }
  // バックアップされたビューポートの状態をリストアする
  void restoreViewports() {
    int i;
    if (cloneStack == null) return;
    for (i = 0; i < viewMax; i++) {
      viewStack[i].clone(cloneStack[i]);
      viewStack[i].restore();
    }
    layerStack = null;  // 初期化（つまりPushされていない状態）
  }
  // デバッグ用（全ビューの情報を表示する）
  void displayAllViewData() {
    int  i;
    viewportClass vp;
    for (i = 0; i < viewMax; i++) {
      vp = viewStack[i];
      print(str(i) + ":[" + str(vp.serialNumber) + "]" + vp.label + "/(" + str(vp.vx) + "," + str(vp.vy) + "," + str(vp.vwidth) + "," + str(vp.vheight) + ")");
      print("-(" + str(vp.wx_min) + "," + str(vp.wy_min) + "," + str(vp.wx_max) + "," + str(vp.wy_max) + ")");
      print(" Color=#" + hex(vp.viewBgColor()) + ":Ratio =[" + str(vp.pixelRatioX) + "," + str(vp.pixelRatioY) + "]:Scale=" + str(vp.viewScale()));
      print(" Offset=" + str(vp.viewOffsetX()) + "," + str(vp.viewOffsetY()) + "Visible:" + str(vp.visible));
      println();
    }
  }
  
  // ----------------
  // ビューポート関連
  // ----------------
  // ビューポートの新規作成（成功した場合はビュー番号を返す．失敗した場合は -1 を返す）
  int createView(String label, int x, int y, int width, int height) {
    // エラーチェック
    if (crowbar.status != 0) {
      println("エラー：ビューポートの作成はOptions()かSetup()内で行なって下さい．");
      exit();
    }
    if (getView(label) != null) {
      println("エラー：指定されたラベルと同じビューポートが既に作成済みです（重複）／" + label);
      exit();
    }
    // 領域確保
    if (viewMax > 0) {
      viewStack  = (viewportClass [])expand(viewStack,  viewMax + 1);  // 最初だけは配列数１で確保されているため
      cloneStack = (viewportClass [])expand(cloneStack, viewMax + 1);  // 最初だけは配列数１で確保されているため
    }
    if (viewStack.length != viewMax + 1) return -1;  // 正しく配列の拡大ができなかった場合
    // 追加処理
    viewMax++;
    cvNumber = viewMax - 1;
    if ((cv = viewStack[cvNumber] = new viewportClass(label, x, y, width, height)) == null) return -1;  // 実際に確保する
    cloneStack[cvNumber] = new viewportClass(label, x, y, width, height);                               // バックアップ領域を確保
    viewStack[cvNumber].serialNumber = cvNumber;  // 作成順のビューポート番号
    viewStack[cvNumber].layerLevel   = cvNumber;  // 表示順
    // 色情報などのデフォルト設定
    cv.fill(#ffffff,   cv.transparency); // 透明度に対応．初期色
    cv.stroke(#000000, cv.transparency); // 透明度に対応．初期色
    cv.world(0, 0, width, height);
    return cvNumber;
  }
  // 全画面指定のビューポートを作成する
  int createView(String label) {
    return createView(label, 0, 0, width, height);
  }
  // ビューポートを分割して新しく確保したビューポート番号を返す．失敗した場合は-1を返す．
  // (a) 水平方向
  // (a-1) ビューポート番号
  int splitViewH(int v, String label, float rate) {
    // エラーチェック
    if ((rate <= 0) || (rate >= 1.0)) {
      println("エラー：分割割合（第二引数）は 0 < rate < 1.0の範囲でして下さい．");
      return -1;
    }
    if (isValidViewNumber(v) == false) { // 不正なビュー番号
      println("エラー：指定されたビューポート番号（第一引数）は不正です／" + str(v));
      return -1;
    }
    // 分割
    viewportClass vp = getView(v);
    int xL, yL, wL, hL;
    int xR, yR, wR, hR;
    // 先に左のビューポートを縮小する
    xL = vp.vx;
    yL = vp.vy;
    wL = vp.vwidth;
    hL = vp.vheight;

    vp.vwidth = int(wL * rate);  // 左のビューポートの変更後の幅
    cv.world(0, 0, vp.vwidth, vp.vheight);
    // 終わったら新規作成
    xR = xL + vp.vwidth;            // 右のビューポートの座標
    yR = yL;
    wR = wL - vp.vwidth;
    hR = hL;
    
    int newV = createView(label, xR, yR, wR, hR);
    getView(newV).cloneAttr(vp);  // 属性のコピー
    return newV;
  }
  // (a-2) ビューポートラベル
  int splitViewH(String target, String label, float rate) {
    viewportClass vp = getView(target);
    if (vp == null) {
      println("エラー：指定されたビューポートラベルは不正です／" + target);
      return -1;
    }
    return splitViewH(vp.serialNumber, label, rate);
  }
  // (a-3) 現在のビューポート
  int splitViewH(String label, float rate) {
    return splitViewH(cvNumber, label, rate);
  }
  // (b) 垂直方向
  // (b-1) 指定したビューポート番号
  int splitViewV(int v, String label, float rate) {
    // エラーチェック
    if ((rate <= 0) || (rate >= 1.0)) {
      println("エラー：分割割合（第二引数）は 0 < rate < 1.0の範囲でして下さい．");
      return -1;
    }
    if (isValidViewNumber(v) == false) { // 不正なビュー番号
      println("エラー：指定されたビューポート番号（第一引数）は不正です／" + str(v));
      return -1;
    }
    // 分割
    viewportClass vp = getView(v);
    int xU, yU, wU, hU;
    int xL, yL, wL, hL;
    // 先に上のビューポートを縮小する
    xU = vp.vx;
    yU = vp.vy;
    wU = vp.vwidth;
    hU = vp.vheight;

    vp.vheight = int(hU * rate);  // 上のビューポートの変更後の高さ
    cv.world(0, 0, vp.vwidth, vp.vheight);
    // 終わったら新規作成
    xL = xU;
    yL = yU + vp.vheight;
    wL = wU;
    hL = hU - vp.vheight;

    int newV = createView(label, xL, yL, wL, hL);
    getView(newV).cloneAttr(vp);  // 属性のコピー
    return newV;
  }
  // (b-2) ビューポートラベル
  int splitViewV(String target, String label, float rate) {
    viewportClass vp = getView(target);
    if (vp == null) {
      println("エラー：指定されたビューポートラベルは不正です／" + target);
      return -1;
    }
    return splitViewV(vp.serialNumber, label, rate);
  }
  // (b-3) 現在のビューポート
  int splitViewV(String label, float rate) {
    return splitViewV(cvNumber, label, rate);
  }
  // 全く同一のビューポートを複製する
  // (1) 指定したビューポート番号
  int cloneView(int v, String label) {
    if (isValidViewNumber(v) == false) { // 不正なビュー番号
      println("エラー：指定されたビューポート番号（第一引数）は不正です／" + str(v));
      return -1;
    }
    viewportClass vp = getView(v);
    int newV = createView(label, vp.vx, vp.vy, vp.vwidth, vp.vheight);
    getView(newV).cloneAttr(vp);  // 属性のコピー
    return newV;
  }
  // (2) 指定したビューポートラベル
  int cloneView(String target, String label) {
    viewportClass vp = getView(target);
    if (vp == null) {
      println("エラー：指定されたビューポートラベルは不正です／" + target);
      return -1;
    }
    return cloneView(vp.serialNumber, label);
  }
  // (3) 現在のビューポート
  int cloneView(String label) {
    return cloneView(cvNumber, label);
  }

  // 指定されたビュー番は正しいか？
  boolean isValidViewNumber(int n) {
    if (n >= viewMax)         return false;
    if (n < 0)                return false;
    if (n > viewStack.length) return false;
    return true;
  }
  
  // ----------------------------------
  // 指定されたビューポートに切り替える
  // ----------------------------------
  // (a) 現在のビューポート番号を返す
  int view() { return cvNumber; }
  // (b) ビューポートを選択（成功した場合はtrue）
  // (b-1) 指定されたビューポート番号のビューポート
  boolean view(int n) {
    if (isValidViewNumber(n) == false) return false;
    cvNumber = n;
    cv = viewStack[cvNumber];
    return true;
  }
  // (b-2) 指定されたラベルのビューポート
  boolean view(String label) {
    viewportClass vp = getView(label);
    if (vp == null) return false;
    cvNumber = vp.serialNumber;
    cv = vp;
    return true;
  }
  // (c) 相対的/絶対的な移動
  // (c-1) 一つ次のビューポートへ（行き過ぎた場合は-1を返す）
  int nextView() {
    if (++cvNumber >= viewMax) return - 1;
    view(cvNumber);
    return cvNumber;
  }
  // (c-2) 一つ前のビューポートへ（戻り過ぎた場合は-1を返す）
  int prevView() {
    if (--cvNumber < 0) return -1;
    view(cvNumber);
    return cvNumber;
  }
  // (c-3) 最初に確保されたビューポートへ
  void firstView() { view(0);           }
  // (c-4) 最後に確保されたビューポートへ
  void lastView()  { view(viewMax - 1); }

  // ------------------------------------
  // 指定されたビューポートのクラスを返す
  // ------------------------------------
  // (a) 指定された番号のビューポートクラスを返す
  viewportClass getView(int no) {
    if (isValidViewNumber(no) == false) return null;
    return viewStack[no];
  }
  // (b) 指定されたラベルのビューポートクラスを返す（大文字小文字は区別しない）
  viewportClass getView(String label) {
    int v;
    for (v = 0; v < viewMax; v++) {
      if (viewStack[v].label.toUpperCase().equals(label.toUpperCase()) == true) return viewStack[v];
    }
    return null;
  }
  // (c) 指定されたビューポートラベルのビューポート番号を返す（見付からない場合は -1）
  int  getViewNumber(String label) {
    viewportClass vp = getView(label);
    if (vp == null) return -1;
    return vp.serialNumber;
  }
  int getViewNumber() { return cvNumber; }
 
  // ----------------------------
  // ビューポートの上下関係の移動
  // ----------------------------
  // ビューポートの表示順を表わす行列を返す
  int [] getViewOrderList() {
    int [] order = new int[viewMax];
    int i;
    for (i = 0; i < viewMax; i++) order[viewStack[i].layerLevel] = i;
    return order;
  }
  // ビューポートの表示順リストを実際のビューポートのデータに書き戻す
  void updateViewOrder(int [] order) {
    int i;
    for (i = 0; i < viewMax; i++) viewStack[order[i]].layerLevel = i;
  }
  // --------------
  // 指定したビューポートの表示順を一番上に移す
  void viewMoveToTop(int v) {
    if (viewMax <= 1) return;  // 不要なので
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    int i;
    int [] order = getViewOrderList();
    for (i = viewStack[v].layerLevel; i < viewMax - 1; i++) order[i] = order[i + 1];
    order[i] = v;
    updateViewOrder(order);
  }
  // 現在のビューポートの表示順を一番上に移す
  void viewMoveToTop()    { viewMoveToTop(cvNumber); }
  // 指定したビューポートの表示順を一番下に移す
  void viewMoveToBottom(int v) {
    if (viewMax <= 1) return;  // 不要なので
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    int i;
    int [] order = getViewOrderList();
    for (i = viewStack[v].layerLevel; i > 0; i--) order[i] = order[i - 1];
    order[0] = v;
    updateViewOrder(order);
  }
  // 現在のビューポートの表示順を一番下に移す
  void viewMoveToBottom() { viewMoveToBottom(cvNumber); }
  // 指定したビューポートを指定したビューポートの上に移す．
  void viewMoveToAbove(int vt, int vr) {
    if (viewMax <= 1) return;                            // ビューポートが一つの場合は無効
    if ((isValidViewNumber(vt) == false) || (isValidViewNumber(vr) == false)) return;  // 不正なビュー番号
    int i, it, ir;
    it = viewStack[vt].layerLevel;
    ir = viewStack[vr].layerLevel;
    if (it == viewMax - 1) return; // 既に一番上
    if (it > ir) return;           // 既に上にいる
    int [] order = getViewOrderList();
    for (i = it; i < ir; i++) order[i] = order[i + 1];
    order[i] = vt;
    updateViewOrder(order);
  }
  void viewMoveToAbove(int vr) { viewMoveToAbove(cvNumber, vr); }
  void viewMoveToAbove(String label1, String label2) {
    int v1, v2;
    v1 = getViewNumber(label1);
    v2 = getViewNumber(label2);
    if ((v1 < 0) || (v2 < 0))  {
      println("エラー：指定されたビューポートラベルは不正です／" + label1 + " or " + label2);
      return;
    }
    viewMoveToAbove(v1, v2);
  }
  void viewMoveToAbove(String label) {
    int v = getViewNumber(label);
    if (v < 0) {
      println("エラー：指定されたビューポートラベルは不正です／" + label);
      return;
    }
    viewMoveToAbove(cvNumber, v);
  }
  // 指定したビューポートを指定したビューポートの下に移す．
  void viewMoveToBelow(int vt, int vr) {
    if (viewMax <= 1) return;                            // ビューポートが一つの場合は無効
    if ((isValidViewNumber(vt) == false) || (isValidViewNumber(vr) == false)) return;  // 不正なビュー番号
    int i, it, ir;
    it = viewStack[vt].layerLevel;
    ir = viewStack[vr].layerLevel;
    if (it == 0) return;   // 既に一番下
    if (it < ir) return;   // 既に下にいる
    int [] order = getViewOrderList();
    for (i = it; i > ir; i--) order[i] = order[i - 1];
    order[i] = vt;
    updateViewOrder(order);
  }
  void viewMoveToBelow(int vr) { viewMoveToBelow(cvNumber, vr); }
  void viewMoveToBelow(String label1, String label2) {
    int v1, v2;
    v1 = getViewNumber(label1);
    v2 = getViewNumber(label2);
    if ((v1 < 0) || (v2 < 0))  {
      println("エラー：指定されたビューポートラベルは不正です／" + label1 + " or " + label2);
      return;
    }
    viewMoveToBelow(v1, v2);
  }
  void viewMoveToBelow(String label) {
    int v = getViewNumber(label);
    if (v < 0) {
      println("エラー：指定されたビューポートラベルは不正です／" + label);
      return;
    }
    viewMoveToBelow(cvNumber, v);
  }
  // 指定されたビューポートを指定された数だけ上下に移す
  void viewMoveToRel(int v, int num) {
    if (viewMax <= 1) return;                   // ビューポートが一つの場合は無効
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    if (num == 0) return;
    if ((v + num) <= 0) {
      viewMoveToTop(v);      // 一番上に
    } else if ((v + num) >= viewMax - 1) {
      viewMoveToBottom(v);   // 一番下に
    } else {
      int i, it;
      int [] order = getViewOrderList();
      it = viewStack[v].layerLevel;
      if (num < 0) {
        // 下に
        for (i = it; i > it + num; i--) order[i] = order[i - 1];
      } else {
        // 上に
        for (i = it; i < it + num; i++) order[i] = order[i + 1];
      }
      order[i] = v;
      updateViewOrder(order);
    }
  }
  void viewMoveToRel(String label, int num) {
    int v = getViewNumber(label);
    if (v < 0) {
      println("エラー：指定されたビューポートラベルは不正です／" + label);
      return;
    }
    viewMoveToRel(v, num);
  }
  void viewMoveToRel(int num) { viewMoveToRel(cvNumber, num);  }
  // 指定されたビューポートを指定された数だけ上に移す
  void viewMoveToUpper(int v, int num)        { viewMoveToRel(v, num);     }
  void viewMoveToUpper(String label, int num) { viewMoveToRel(label, num); }
  void viewMoveToUpper(int num)               { viewMoveToRel(num);        }
  // 指定されたビューポートを一つ上に移す
  void viewMoveToUp(int v)         { viewMoveToUpper(v, 1);     }
  void viewMoveToUp(String label)  { viewMoveToUpper(label, 1); }
  void viewMoveToUp()              { viewMoveToUpper(1);        }
  // 指定されたビューポートを指定された数だけ下に移す
  void viewMoveToLower(int v, int num)        { viewMoveToRel(v, -num);     }
  void viewMoveToLower(String label, int num) { viewMoveToRel(label, -num); }
  void viewMoveToLower(int num)               { viewMoveToRel(-num);        }
  // 指定されたビューポートを一つ下に移す
  void viewMoveToDown(int v)         { viewMoveToLower(v, 1);     }
  void viewMoveToDown(String label)  { viewMoveToLower(label, 1); }
  void viewMoveToDown()              { viewMoveToLower(1);        }
  // ビューポートの並び順を初期状態（作成順）に戻す
  void viewLayerReset() {
    int i;
    if (viewMax <= 1) return;                   // ビューポートが一つの場合は無効
    int [] order = getViewOrderList();
    for (i = 0; i < viewMax; i++) order[i] = i;
    updateViewOrder(order);
  }
  // ビューポートの並び順を記録する
  void viewLyaerPush() {
    if (viewMax <= 1) return;                   // ビューポートが一つの場合は無効
    layerStack = getViewOrderList();
  }
  // ビューポートの並び順を復元する
  void viewLayerPop() {
    if (viewMax <= 1) return;                   // ビューポートが一つの場合は無効
    if (layerStack == null) {
      println("エラー：ビューポートの並び順の記録前にviewLayerPop()が呼ばれました．");
      exit();
    }
    int [] order = getViewOrderList();
    if (layerStack.length != viewMax) {
      println("エラー：ビューポートの並び順を復元しようとしましたがスタックのサイズが異なります．");
      exit();
    }
    int i;
    for (i = 0; i < viewMax; i++) order[i] = layerStack[i];
    updateViewOrder(order);
  }

  // ----------------------------------
  // ビューポートの枠線や不透明度の設定
  // ----------------------------------
  // (a) フレームカラーを取得
  int frameColor(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).frameColor;
  }
  // (b) フレームカラーの設定（＋変更前の値）
  // (b-2) 指定したビューポート
  int frameColor(int v, int c) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    int oldcolor = frameColor(v);
    getView(v).frameColor(c);
    return oldcolor;
  }
  // (c) フレームの表示／非表示
  void viewDrawFrame(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).drawFrame(flag);
  }
  // (a) 不透明度の取得
  int transparency(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).transparency();
  }
  // (b) 不透明度の設定（＋変更前の値）
  int transparency(int v, int trans) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).transparency(trans);
  }
  // 指定されたビューの背景色を変更する
  // (a) 背景色を取得
  int viewBgColor(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).viewBgColor();
  }
  // (b) 背景色を指定
  int viewBgColor(int v, int c) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).viewBgColor(c);
  }
  // ビューポートの背景を透明にするか？（trueで透明）
  void transparentView(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).transparentView(flag);    
  }

  // ------------------------
  // ビューポートの位置の変更
  // ------------------------
  // (a) 指定した位置に移動
  // (a-1) 現在のビューポート
  // (a-2) 指定したビューポート
  void moveView(int v, int x, int y) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).moveView(x, y);
  }
  void moveViewRel(int v, int dx, int dy) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).moveViewRel(dx, dy);
  }

  // 座標変換（現在のビュー）
  // 座標変換（ワールド→ビュー：大きさのみ）
  private int _px(float x) { return cv._px(x); }
  private int _py(float y) { return cv._py(y); }
  // 座標変換（ワールド→ビュー）
  private int px(float x) { return cv.px(x); }
  private int py(float y) { return cv.py(y); }

  // -----------------------------------
  // 描画コマンド関係（ユーザによる利用）
  // -----------------------------------
  // 全てのビューポートを画面消去する（現在のビューは変更しない）
  void clrAllView() {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) clrView(i);
    view(cv);
  }
  // 指定されたビューポートを画面消去する（現在のビューポートをそのビューポートに移す）
  void clrView(int i) {
    if (view(i) != true) return;
    view(i);
    cv.clrView();
  }

  // --------
  // 倍率関係
  // --------
  // (a) 現在の倍率を返す：不正なビュー番号の場合は1.0を返す
  float viewScale(int v) {
    if (isValidViewNumber(v) == false) return 1.0;  // 不正なビュー番号
    return getView(v).viewScale();
  }
  // (b) 倍率の値を変更する（＋変更前の値を返す）
  float viewScale(int v, float s) {
    float oldscale;    
    if (isValidViewNumber(v) == false) return 1.0;  // 不正なビュー番号
    oldscale = getView(v).viewScale();
    getView(v).viewScale(s);
    return oldscale;
  }
  // (c) 全てのビューポートの倍率を指定した倍率に変更する
  void viewScaleAll(float s) {
    int i;
    for (i = 0; i < viewMax; i++) scale(i, s);
  }

  // ----------------
  // オフセット量関係
  // ----------------
  // (a) 指定したビューポートのオフセット量を返す
  // (a-1) ｘ方向
  float viewOffsetX(int v) {
      viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    return vp.viewOffsetX();
  }
  // (a-2) ｙ方向
  float viewOffsetY(int v) {
      viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    return vp.viewOffsetY();
  }
  // (b) オフセット量を変更する（＋変更前の値を返す）：指定したビューポート
  // (b-1) ｘ方向
  float viewOffsetX(int v, float offset) {
    viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    float oldoffset = vp.viewOffsetX();
    vp.viewOffsetX(offset);
    return oldoffset;
  }
  // (b-2) ｙ方向
  float viewOffsetY(int v, float offset) {
    viewportClass vp = getView(v);
    if (vp == null) return 0.0;  // 不正なビュー
    float oldoffset = vp.viewOffsetY();
    vp.viewOffsetY(offset);
    return oldoffset;
  }
  // (b-3) ｘ方向／ｙ方向同時：成功した場合はtrueを返す
  boolean viewOffset(int v, float offx, float offy) {
    float oldoffset;
    viewportClass vp = getView(v);
    if (vp == null) return false;  // 不正なビュー
    vp.viewOffsetX(offx);
    vp.viewOffsetY(offy);
    return true;
  }
  // 全てのビューポートの設定を変更する
  void viewOffsetXAll(float offset) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) viewOffsetX(i, offset);
    view(cv);
  }
  void viewOffsetYAll(float offset) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) viewOffsetY(i, offset);
    view(cv);
  }
  void viewOffsetAll(float offx, float offy) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) viewOffset(i, offx, offy);
    view(cv);
  }

  // --------------------
  // ワールド座標系の設定
  // --------------------
  // 【参照】
  // (a) ワールド座標系の設定値の参照：指定したビューポート：不正なビューポート番号が指定された場合は0.0を返す
  float world_minX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wx_min;
  }
  float world_maxX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wx_max;
  }
  float world_minY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wy_min;
  }
  float world_maxY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wy_max;
  }
  // 【変更】
  // (a) ワールド座標系の変更：指定したビューポート：成功した場合はtrueを返す
  boolean world(int v, float x1, float y1, float x2, float y2) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).world(x1, y1, x2, y2);
    return true;
  }
  boolean world(String label, float x1, float y1, float x2, float y2) {
    int v = getViewNumber(label);
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).world(x1, y1, x2, y2);
    return true;
  }
  boolean world(float x1, float y1, float x2, float y2) {
    if (viewMax < 1) return false;
    cv.world(x1, y1, x2, y2);
    return true;
  }
  // (b) ｘ，ｙそれぞれの最小値／最大値を個別に変更
  // 指定したビューポートの値を変更：成功した場合はtrueを返す
  boolean world_minX(int v, float x) { return world(v, x, world_maxX(v), world_minY(v), world_maxY(v)); }
  boolean world_maxX(int v, float x) { return world(v, world_minX(v), x, world_minY(v), world_maxY(v)); }
  boolean world_minY(int v, float y) { return world(v, world_minX(v), world_maxX(v), y, world_maxY(v)); }
  boolean world_maxY(int v, float y) { return world(v, world_minX(v), world_maxX(v), world_minY(v), y); }
  // (c) ピクセル比のみを指定（引数なし＝ピクセル比は変えない）
  // (c-1) ビューポート中央を原点
  boolean worldOrigin(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldOrigin();
    return true;
  }
  boolean worldOrigin(int v, float ratio) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldOrigin(ratio);
    return true;
  }
  // (c-2) ビューポート左上を原点
  boolean worldLU(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldLU();
    return true;
  }
  boolean worldLU(int v, float ratio) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldLU(ratio);
    return true;
  }
  // (c-2) ビューポート左下を原点
  boolean worldLL(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldLL();
    return true;
  }
  boolean worldLL(int v, float ratio) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldLL(ratio);
    return true;
  }
  // (c-3) ビューポート右上を原点
  boolean worldRU(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldRU();
    return true;
  }
  boolean worldRU(int v, float ratio) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldRU(ratio);
    return true;
  }
  // (c-4) ビューポート右下を原点
  boolean worldRL(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldRL();
    return true;
  }
  boolean worldRL(int v, float ratio) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldRL(ratio);
    return true;
  }
  // (d) ワールド座標のオフセット量とピクセル比を指定
  // つまりこの指定した座標がビューポートの中央に表示されるようにマッピングされる
  boolean worldCenter(int v, float x, float y) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldCenter(x, y);
    return true;
  }
  boolean worldCenter(int v, float x, float y, float ratio) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    getView(v).worldCenter(x, y, ratio);
    return true;
  }

  // --------------------------------------------------
  // ビューポートの表示／非表示のダイナミックな切り替え
  // --------------------------------------------------
  // 状態
  boolean isViewVisible(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).isViewVisible();
  }
  // (a) 表示
  void viewVisible(int v) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).viewVisible();
  }
  // (b) 非表示
  void viewUnvisible(int v) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    getView(v).viewUnvisible();
  }
  // (c) 引数による指定
  void viewVisible(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return;  // 不正なビュー番号
    if (flag) viewVisible(v); else viewUnvisible(v);
  }
  // (d) トグル動作
  boolean toggleViewVisible(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).toggleViewVisible();
  }

  // 与えられた座標（マウスクリックなど）はどのビューポートを指しているのか？
  // （どのビューポートも含まれないならば-1）
  // trans == true の場合は透明なビューポートを無視する
  int detectActiveView(int x, int y) { return detectActiveView(x, y, true); }
  int detectActiveView(int x, int y, boolean trans) {
    int [] order = getViewOrderList();
    int i, x1, y1, x2, y2;
    viewportClass vp;
    for (i = viewMax - 1; i >= 0; i--) {
      vp = viewStack[order[i]];
      if (vp.isViewVisible() == false) continue;  // 非表示は無視
      if ((trans == true) && (vp.transparentView == true)) continue;
      if (vp.transparency == 0) {           // ビューポートが透明背景
        if (trans == true) continue;
        int col;
        vp.pg.beginDraw();
        col = vp.pg.get(x - vp.vx, y - vp.vy);
        vp.pg.endDraw();
        if ((col & 0x0ff000000) == 0) continue;  // もし描画されていないならば無視
      }
      x1 = vp.vx;
      y1 = vp.vy;
      x2 = x1 + vp.vwidth;
      y2 = y1 + vp.vheight;
      if ((x1 <= x) && (x <= x2) && (y1 <= y) && (y <= y2)) return order[i];
    }
    return -1;
  }

// =========================================================================================================

  // -----------------------------------------------
  // ビューの合成（crowbarClass.pdeのdraw()にて使用）
  // -----------------------------------------------
  void refreshView() {
    if (tomahawkMode == 0) return;
//  if (tomahawkMode == 2) background(_BGCOLOR);
    if (crowbar.status == 10) {
      // ビューの合成
      int i;
      int [] order = getViewOrderList();
      // 表示順のリスト作成（きちんと欠番，重複なく並んでいることは上下関係の操作時に確認済みという前提）
      viewportClass vp;
      for (i = 0; i < viewMax; i++) {
        vp = viewStack[order[i]];
        if (vp.isViewVisible() == false) continue;
        if (vp.drawFrame) {
          vp.pg.beginDraw();
          vp.pg.noFill();
          vp.pg.stroke(vp.frameColor, 255);
          vp.pg.strokeWeight(vp.frameWidth);
          vp.pg.rect(vp.frameWidth / 2, vp.frameWidth / 2, vp.vwidth - vp.frameWidth, vp.vheight - vp.frameWidth);
//        vp.pg.rect(0, 0, vp.vwidth - 1, vp.vheight - 1);
          vp.pg.stroke(vp.cdSet.strokeColor, vp.cdSet.strokeTrans);
          vp.pg.strokeWeight(vp.cdSet.strokeWeight);
          vp.pg.endDraw();
        }
        blend(vp.pg, 0, 0, vp.vwidth, vp.vheight, vp.vx, vp.vy, vp.vwidth, vp.vheight, BLEND);
      }
    }
    if (tomahawkMode == 2) blend(crowbar.screenSetting.pgText, 0, 0, crowbar.screenSetting.gwx, crowbar.screenSetting.gwy, 0, 0, crowbar.screenSetting.gwx, crowbar.screenSetting.gwy, BLEND);
  }
}

