// ---------------------------------------------------------
// Crowbar組込高機能グラフィックライブラリTomahawk
// by T.Shirai
// 開発開始 2012/06/07  -> Crowbar Ver.3.00.00 (2012/06/21)
// α版     2012/06/21  ->         Ver.3.01.00 (2012/07/02)
// α版     2012/07/02  ->         Ver.3.02.00 (2012/07/05)
// α版     2012/07/05  ->         Ver.3.03.00 (2012/07/11)
// α版     2012/07/11  ->         Ver.3.04.00 (2012/07/16)
// β版     2012/07/16  ->
// ---------------------------------------------------------
// tomahawkMode : 0, 1, 2 はTomahawkのモードを切り替える非常に重要なパラメータで，実行中に変更するものではない．
// 0 : このCrowbarにはTomahawkが存在しない場合（tomahawkClass.pdeを削除し，crowbarClass.pde冒頭のコメントを外す）
//     Tomahawkは非常に負荷が大きいのでビューポートを利用する必要がない場合はこのモードの使用を勧めます．
// 1 : Tomahawkを利用するがCrowbarのテキスト表示を背景に描画する．ビューポートはその上のレイヤーとして上書きされる．
//     実行ウィンドウの一部を画像描画用のビューポートとして活用する場合などに適している．
//     テキスト専用レイヤーをブレンドしないため比較的負荷が少ない．
// 2 : Tomahawkを利用する．さらにCrowbarのテキスト出力を専用のレイヤーに描画する．
//     レイヤーは背景，ビューポート，テキスト表示専用レイヤーの順でブレンドされるためテキスト表示が隠れることが無い．
//     オーバーヘッドが最も重いモード．
// （テキスト専用レイヤー使用時：2の制約）：Processingの仕様に基づく
//   まず，フォントの見栄えが悪い．テキストの色も何故か指定した値よりも少し暗くなる．
//   レイヤーを重ねるのにblend()を使用している．テクニックとしては不透明度０（つまり透明）かつ黒い背景に不透明度255で
//   グラフィックス描画を行ない，blend(BLEND)すると不透明度255のPixelのみが合成されるという手法を取っています．
//   ところがtext()はbackground(#000000, 0)の背景色#000000を利用してくれるのですが，不透明度255で描画してしまいます．
//   そこで文字をtext()で描画する度にそのエリアをpixel単位でチェックして背景色#000000の場合は強制的にcolor(#00000, 0)
//   に変更します．したがって文字色として#000000を指定すると区別が付きませんので内部で勝手に#000002に文字色を変更して
//   います．この不透明度の変更の処理にも余計なオーバーヘッドが生じています．あまり大きなフォントサイズを指定すると，
//   blend()にも時間が掛かります．

// 角度を扱うクラス（ネイティブは度で，デフォルトも度）
class angleClass {
  private float degree;
  private float radian;
  // コンストラクタ
  angleClass(float angle) {
    set(angle);
  }
  // 複製
  void clone(angleClass src) {
    degree = src.degree;
    radian = src.radian;
  }
  // deg/radで設定
//void set(float angle) { setDegree(angle); }
  void set(float a) {
    if (!crowbar.degreeBase) setRadian(a);
      else                   setDegree(a);
  }
  angleClass set(float angle, boolean deg) {
    if (deg) setDegree(angle);
      else   setRadian(angle);
    return this;
  }
  angleClass setDegree(float degV) {
    degree = degV;
    radian = radians(degV);
    return this;
  }
  angleClass setRadian(float radV) {
    radian = radV;
    degree = degrees(radV);
    return this;
  }
  // deg/radで返す
//float get()            { return degree; }
  float get() {
    if (!crowbar.degreeBase) return getRadian();
    return getDegree();
  }
  float get(boolean deg) { return (deg ? degree : radian); }
  float getDegree()      { return degree; }
  float getRadian()      { return radian; }
  // ±π，±180度に正規化，じゃなくて
  // -2π ≦θ＜～2π，-360≦θ＜360に正規化
  angleClass normalize() {
    radian = radians(degree);
/*
    while (radian >  PI)    radian -= TWO_PI;
    while (radian < -PI)    radian += TWO_PI;
    while (degree > 180.0)  degree -= 360.0;
    while (degree < -180.0) degree += 360.0;
*/
    while (radian >=  TWO_PI) radian -= TWO_PI;
    while (radian <  -TWO_PI) radian += TWO_PI;
    while (degree >=  360.0) degree -= 360.0;
    while (degree <  -360.0) degree += 360.0;
    return this;
  }
  // 足し算
  angleClass add(angleClass src) {
    degree += src.degree;
    return this.normalize();
  }
  angleClass add(float deg) {
    degree += deg;
    return this.normalize();
  }
  // 引き算
  angleClass sub(angleClass src) {
    degree -= src.degree;
    return this.normalize();
  }
  angleClass sub(float deg) {
    degree -= deg;
    return this.normalize();
  }
  // スカラを掛ける
  angleClass mult(float a) {
    degree *= a;
    return this.normalize();
  }
  // スカラで割る
  angleClass divide(float a) {
    degree /= a;
    return this.normalize();
  }
}

// 現在の線色や座標などの情報を保持
class currentDrawingClass {
  // 描画関係
  boolean  isFill;       // 塗り潰す？
  int      fillColor;    // 塗り潰し色
  int      fillTrans;    // 塗り潰しの不透明度
  boolean  isStroke;     // 外形線描く？
  int      strokeColor;  // 外形線色
  int      strokeTrans;  // 外形線の不透明度
  float    strokeWeight; // 線幅
  int      rectMode;     // 四角形の座標指定のモード（CORNER, CORNERS, CENTER, RADUIS）
  int      ellipseMode;  // 楕円と円弧の座標指定のモード（CENTER, RADIUS, CORNER)
  int      textureMode;  // テクスチャマッピングのモード（vertex()４引数 (IMAGE, NORMALIZED)
  int      imageMode;    // image(), copy()の座標の取り方 (CENTER, CORNER)
  float    curveTightness; // 曲線（curve）の形状を修正するパラメータ
  int      curveDetail;    // 曲線の曲線解像度
  // 文字関係
  int      textColor;    // テキスト文字の色
  int      textSize;     // フォントサイズ
  int      textAlignH;   // 水平方向割り当て
  int      textAlignV;   // 垂直方向割り当て
  int      textLeading;  // 改行幅
  // LP
  int      vpx, vpy;     // ビューポート内の最後に描画した座標
  float    wpx, wpy;     // ワールド座標における最後に描画した座標
  currentDrawingClass() {
    setDefault();
  }
  // 複製
  void clone(currentDrawingClass src) {
  // 描画関係
    isFill         = src.isFill;
    fillColor      = src.fillColor;
    fillTrans      = src.fillTrans;
    isStroke       = src.isStroke;
    strokeColor    = src.strokeColor;
    strokeTrans    = src.strokeTrans;
    strokeWeight   = src.strokeWeight;
    rectMode       = src.rectMode;
    ellipseMode    = src.ellipseMode;
    textureMode    = src.textureMode;
    imageMode      = src.imageMode;
    curveTightness = src.curveTightness;
    curveDetail    = src.curveDetail;
  // 文字関係
    textColor    = src.textColor;
    textSize     = src.textSize;
    textAlignH   = src.textAlignH;
    textAlignV   = src.textAlignV;
    textLeading  = src.textLeading;
  // LP
    vpx          = src.vpx;
    vpy          = src.vpy;
    wpx          = src.wpx;
    wpy          = src.wpy;
  }
  // 初期値に設定
  void setDefault() {
  // 描画関係
    isFill         = true;
    fillColor      = #ffffff;
    fillTrans      = 255;
    isStroke       = true;
    strokeColor    = #000000;
    strokeTrans    = 255;
    strokeWeight   = 1;
    rectMode       = CORNER;
    ellipseMode    = CENTER;
    textureMode    = IMAGE;
    imageMode      = CORNER;
    curveTightness = 0.0;
    curveDetail    = 20;
  // 文字関係
    textColor    = _TEXTCOLOR;
    textSize     = _FONTSIZE;
    textAlignH   = LEFT;
    textAlignV   = TOP;
    textLeading  = int(textSize * 1.5);
  // LP
    vpx = vpy    = 0;
    wpx = wpy    = 0.0;
  }
}

class curveClass {
  float  x1a,  y1a,  x2a,  y2a,  x3a,  y3a,  x4a,  y4a;
  int   _x1a, _y1a, _x2a, _y2a, _x3a, _y3a, _x4a, _y4a;
  int   status, _status;
  curveClass () {
    status  = 0;
    _status = 0;
  }
  // ワールド座標系
  void start(float x, float y) { x1a = x; y1a = y; x2a = x; y2a = y; status = 1; }
  void addSub(float x, float y, boolean relative) {
    if (relative) {
      switch (status) {
        case 0  : x += x1a; y += y1a; break;
        case 1  : x += x2a; y += y2a; break;
        case 2  : x += x3a; y += y3a; break;
        default : x += x4a; y += y4a; break;
      }
    }
    switch (status) {
      case 0 : println("エラー：曲線の始点が指定されていません．");
               x1a = x; y1a = y;x2a = x; y2a = y; status++; break;  // 停止すべき？
      case 1 : x3a = x; y3a = y; status++; break;
      case 2 : x4a = x; y4a = y; status++; break;
      case 3 : status++;
      default :
        x1a = x2a; x2a = x3a; x3a = x4a; x4a = x;
        y1a = y2a; y2a = y3a; y3a = y4a; y4a = y;
    }
  }
  void addRel(float dx, float dy) { addSub(dx, dy, true); }
  void add(float x, float y)      { addSub( x,  y, false); }
  void end() {
    if (status == 3) {
      x1a = x2a; x2a = x3a; x3a = x4a;
      y1a = y2a; y2a = y3a; y3a = y4a;
      x4a = x3a;
      y4a = y3a;
      status++;
    } else {
      x1a = x2a; x2a = x3a; x3a = x4a;
      y1a = y2a; y2a = y3a; y3a = y4a;
    }
    if (status <= 3) println("エラー：曲線に指定された点数が足りません．");
    status = 0;
  }
  // ピクセル座標系
  void _start(int x, int y) { _x1a = x; _y1a = y; _x2a = x; _y2a = y; _status = 1; }
  void _addSub(int x, int y, boolean relative) {
    if (relative) {
      switch (_status) {
        case 0  : x += _x1a; y += _y1a; break;
        case 1  : x += _x2a; y += _y2a; break;
        case 2  : x += _x3a; y += _y3a; break;
        default : x += _x4a; y += _y4a; break;
      }
    }
    switch (_status) {
      case 0 : println("エラー：曲線の始点が指定されていません．");
               _x1a = x; _y1a = y; _x2a = x; _y2a = y; _status++; break;  // 停止すべき？
      case 1 : _x3a = x; _y3a = y; _status++; break;
      case 2 : _x4a = x; _y4a = y; _status++; break;
      case 3 : _status++;
      default :
        _x1a = _x2a; _x2a = _x3a; _x3a = _x4a; _x4a = x;
        _y1a = _y2a; _y2a = _y3a; _y3a = _y4a; _y4a = y;
    }
  }
  void _addRel(int dx, int dy) { _addSub(dx, dy, true); }
  void _add(int x, int y)      { _addSub( x,  y, false); }
  void _end() {
    if (_status == 3) {
      _x1a = _x2a; _x2a = _x3a; _x3a = _x4a;
      _y1a = _y2a; _y2a = _y3a; _y3a = _y4a;
      _x4a = _x3a;
      _y4a = _y3a;
      _status++;
    } else {
      _x1a = _x2a; _x2a = _x3a; _x3a = _x4a;
      _y1a = _y2a; _y2a = _y3a; _y3a = _y4a;
    }
    if (_status <= 3) println("エラー：曲線に指定された点数が足りません．");
    _status = 0;
  }
}

// 各ビューポートのクラス
class viewportClass {
  protected PGraphics pg;                              // ビューポートグラフィックスのハンドル
  protected float     wx_min, wx_max, wy_min, wy_max;  // ビューポートのワールド座標
  protected int       vx, vy;                          // ビューポートの実行スクリーンに対する左上の座標
  protected int       width, height;                   // ビューポートの横幅／縦幅
  boolean             monoRatio;                       // ピクセル比は縦も横も同一にするか
  float               pixelRatioX, pixelRatioY;        // ピクセル比（縦と横）ワールド座標→ビューポート内のPixel座標変換用
  protected int       bgColor;                         // ビューポートの背景色
  protected String    label;                           // ビューポートの説明（オプション）
  currentDrawingClass cdSet;                           // 最新の描画関係の情報
  protected int       serialNumber;                    // 作成順の通し番号（viewStack[[]の引数）
  // 見え方（見せ方）
  protected boolean   transparentView;                 // ビューポートの背景が透明なビュー
  protected int       transparency;                    // ビューポートの透明度（枠は除く）： 0～255（デフォルトは255）
  boolean             drawFrame;                       // 枠の表示を行なうか？
  int                 frameColor;                      // 枠の色
  int                 frameWidth;                      // 枠の幅（ピクセル）
  protected int       layerLevel;                      // 表示順（優先度）：数字が大きいほど後（上）に表示される
  float               magnification;                   // 画面の拡大率（ピクセル比にこの値を掛ける）
  float               offsetX, offsetY;                // ワールド座標系におけるオフセット量
  angleClass          rotateAngle;                     // ビューポートの視点の回転角度（[rad]）
  float               pivotX, pivotY;                  // ピボット座標（拡大縮小，回転）
  protected boolean   visible;                         // ビューポートの表示をダイナミックにOn/Offできる
  // 曲線を簡便に描くためのクラス
  curveClass          cc;
  // ピクセル関係
  int []              pixels;                          // loadPixels()でセットされる
  // テキスト関係
  private PFont       font;
  // コンストラクタ
  viewportClass(String name, int x, int y, int w, int h) {
    // デバッグ情報
    label     = name;
    // 座標関係の初期化
    vx          = x;
    vy          = y;
    this.width  = w;
    this.height = h;
    monoRatio   = true;
//  world(0.0, 0.0, this.width, this.height);  // 初期値で設定
    worldOrigin(1.0);   // 初期値で設定
    // 背景色，フレーム
    bgColor      = _BGCOLOR;
    transparentView = false;
    drawFrame    = true;
    transparency = 255;
    frameColor   = #000000;
    frameWidth   = 1;
    // 最新の描画関係の情報
    cdSet      = new currentDrawingClass();
    // ビューポートの見せ方
    viewPivotCenter();          // 拡大縮小，回転の中心座標（ビューポート中心に初期設定）
    magnification     = 1.0;    // 拡大縮小
    offsetX = offsetY = 0.0;    // 平行移動量
    rotateAngle       = new angleClass(0.0);   // 回転角度（[rad]）
    serialNumber      = -1;     // 作成順の通し番号
    layerLevel        = -1;     // レイヤーの順番

    // ビューポート用のグラフィックスを確保
    pg = createGraphics(this.width, this.height, JAVA2D);    // P3Dで描画すると円に変な線が沢山入るので．
    pg.beginDraw();
//  pg.smooth();  // 一応，念のために
    pg.endDraw();
    visible   = true;
    // 曲線を描くクラス
    cc = new curveClass();
  }
  // プロパティの複製（属性関係のみ）：そっくりさん．splitViewH/V()で作成した際に用いる
  void cloneAttr(viewportClass src) {
    bgColor         = src.bgColor;
    transparentView = src.transparentView;
    drawFrame       = src.drawFrame;
    transparency    = src.transparency;
    frameColor      = src.frameColor;
    frameWidth      = src.frameWidth;
    cdSet.clone(src.cdSet);
    pivotX          = src.pivotX;
    pivotY          = src.pivotY;
    magnification   = src.magnification;
    offsetX         = src.offsetX;
    offsetY         = src.offsetY;
    rotateAngle.clone(src.rotateAngle);
    visible         = src.visible;
  }
  // プロパティの複製（全部）
  void clone(viewportClass src) {
    vx              = src.vx;
    vy              = src.vy;
    this.width      = src.width;
    this.height     = src.height;
    monoRatio       = src.monoRatio;
    cloneAttr(src);
    // ユニークな情報（複製注意）
    label           = src.label;
    serialNumber    = src.serialNumber;
    layerLevel      = src.layerLevel;
  }
  // ワールド座標系のコピー（cloneView()で使用）
  void cloneWorld(viewportClass src) {
    wx_min      = src.wx_min;
    wx_max      = src.wx_max;
    wy_min      = src.wy_min;
    wy_max      = src.wy_max;
    monoRatio   = src.monoRatio;
    pixelRatioX = src.pixelRatioX;
    pixelRatioY = src.pixelRatioY;
  }

  // 現在の設定値でグラフィックスを初期化する
  void restore() {
    pg.beginDraw();
    if (cdSet.isFill) pg.fill(cdSet.fillColor, cdSet.fillTrans);
      else            pg.noFill();
    if (cdSet.isStroke) pg.stroke(cdSet.strokeColor, cdSet.strokeTrans);
      else              pg.noStroke();
    pg.strokeWeight(cdSet.strokeWeight);
    pg.rectMode(cdSet.rectMode);
    pg.ellipseMode(cdSet.ellipseMode);
    pg.textureMode(cdSet.textureMode);
    pg.imageMode(cdSet.imageMode);
    pg.curveTightness(cdSet.curveTightness);
    pg.curveDetail(cdSet.curveDetail);
    pg.endDraw();
  }

  // ピクセル比を設定する
  void setPixelRatio(float x, float y) {
    setPixelRatioX(x);
    setPixelRatioY(y);
  }
  void setPixelRatioX(float x) {
    if (x == 0.0) {
      println("エラー：指定されたピクセル比がゼロです．");
      crowbar.halt();
      return;
    }
    pixelRatioX = x;
  }
  void setPixelRatioY(float y) {
    if (y == 0.0) {
      println("エラー：指定されたピクセル比がゼロです．");
      crowbar.halt();
      return;
    }
    pixelRatioY = y;
  }
  
  // ピクセル比の計算
  void calcPixelRatio() {
    float px, py;
    px = this.width  / (wx_max - wx_min);
    py = this.height / (wy_max - wy_min);
    if (monoRatio) {
      // 縦と横のピクセル比を同一に
      if (px > py) {
        // 縦優先
        setPixelRatio(py, py);
      } else {
        // 横優先
        setPixelRatio(px, px);
      }
    } else {
      // 別々
      setPixelRatio(px, py);
    }
  }
  // --------------------
  // ワールド座標系の設定
  // --------------------
  // 【参照】
  float world_minX() { return wx_min; }
  float world_maxX() { return wx_max; }
  float world_minY() { return wy_min; }
  float world_maxY() { return wy_max; }
  // 【変更】
  // (a) 全指定
  viewportClass world(float minx, float miny, float maxx, float maxy) {
    if ((minx >= maxx) || (miny >= maxy)) {
      println("ワールド座標系の設定が正しくありません（最小値と最大値が逆です）");
      crowbar.halt();
      return this;
    }
    wx_min = minx;
    wy_min = miny;
    wx_max = maxx;
    wy_max = maxy;
    calcPixelRatio();
    return this;
  }
  viewportClass worldX(float minx, float maxx) { return world(minx, wy_min, maxx, wy_max); }
  viewportClass worldY(float miny, float maxy) { return world(wx_min, miny, wx_max, maxy); }
  // (b) ｘ，ｙそれぞれの最小値／最大値を個別に変更
  float world_minX(float x) {
    float oldvalue = wx_min;
    world(x, wy_min, wx_max, wy_max);
    return oldvalue;
  }
  float world_minY(float y) {
    float oldvalue = wy_min;
    world(wx_min, y, wx_max, wy_max);
    return oldvalue;
  }
  float world_maxX(float x) {
    float oldvalue = wx_max;
    world(wx_min, wy_min, x, wy_max);
    return oldvalue;
  }
  float world_maxY(float y) {
    float oldvalue = wy_max;
    world(wx_min, wy_min, wx_max, y);
    return oldvalue;
  }
  // (c) ピクセル比のみを指定
  // 以下八方位（0:中央, 1:左上, 2:左下, 3:右上, 4: 右下, 5:中央上, 6:中央下, 7:左中央, 8:右中央
  viewportClass _world(int pos, float ratio) {
    float dwx, dwy, dwx2, dwy2;
    setPixelRatio(ratio, ratio);
    dwx = this.width  / ratio;
    dwy = this.height / ratio;
    dwx2 = dwx / 2.0;
    dwy2 = dwy / 2.0;
    switch (pos) {
      case 0 : world(-dwx2, -dwy2, dwx2, dwy2); break;
      case 1 : world( 0.0,  -dwy,  dwx,  0.0);  break;
      case 2 : world( 0.0,   0.0,  dwx,  dwy);  break;
      case 3 : world(-dwx,  -dwy,  0.0,  0.0);  break;
      case 4 : world(-dwx,   0.0,  0.0,  dwy);  break;
      case 5 : world(-dwx2, -dwy,  dwx2, 0.0);  break;
      case 6 : world(-dwx2,  0.0,  dwx2, dwy);  break;
      case 7 : world( 0.0,  -dwy2, dwx,  dwy2); break;
      case 8 : world(-dwx,  -dwy2, 0.0,  dwy2); break;
    }
    return this; 
  }
  // (c-1) ビューポート中央を原点
  viewportClass worldOrigin() { return worldOrigin(pixelRatioX); }
  viewportClass worldOrigin(float ratio) { return _world(0, ratio); }
  // (c-2) ビューポート左上を原点
  viewportClass worldLT() { return worldLT(pixelRatioX); }
  viewportClass worldLT(float ratio) { return _world(1, ratio); }
  // (c-3) ビューポート左下を原点
  viewportClass worldLB() { return worldLB(pixelRatioX); }
  viewportClass worldLB(float ratio) { return _world(2, ratio); }
  // (c-4) ビューポート右上を原点
  viewportClass worldRT() { return worldRT(pixelRatioX); }
  viewportClass worldRT(float ratio) { return _world(3, ratio); }
  // (c-5) ビューポート右下を原点
  viewportClass worldRB() { return worldRB(pixelRatioX); }
  viewportClass worldRB(float ratio) { return _world(4, ratio); }
  // (c-6) ビューポート中央上を原点
  viewportClass worldCT() { return worldCT(pixelRatioX); }
  viewportClass worldCT(float ratio) { return _world(5, ratio); }
  // (c-7) ビューポート中央下を原点
  viewportClass worldCB() { return worldCB(pixelRatioX); }
  viewportClass worldCB(float ratio) { return _world(6, ratio); }
  // (c-8) ビューポート左中央を原点
  viewportClass worldLC() { return worldLC(pixelRatioX); }
  viewportClass worldLC(float ratio) { return _world(7, ratio); }
  // (c-9) ビューポート右中央を原点
  viewportClass worldRC() { return worldRC(pixelRatioX); }
  viewportClass worldRC(float ratio) { return _world(8, ratio); }
  
  // (d) ワールド座標の中心座標とピクセル比を指定
  // つまりこの指定した座標がビューポートの中央に表示されるようにマッピングされる
  viewportClass worldCenter(float x, float y) { return worldCenter(x, y, pixelRatioX); }
  viewportClass worldCenter(float x, float y, float ratio) {
    setPixelRatio(ratio, ratio);
    worldOrigin();
    world(wx_min + x, wy_min + y, wx_max + x, wy_max + y);
    return this;
  }
  
  // (e) ワールド座標系をｘｙ軸方向に移動する
  viewportClass worldMove(float dx, float dy) { return world(wx_min + dx, wy_min + dy, wx_max + dx, wy_max + dy); }
  viewportClass worldMoveX(float dx) { return worldMove(dx, 0.0); }
  viewportClass worldMoveY(float dy) { return worldMove(0.0, dy); }

  // ---------------------------------------------------
  // ワールド座標系からビューポートのPixel座標系への変換
  // ---------------------------------------------------
  // a) 単にピクセル比を掛けたもの（半径や高さ／幅用）
  int _px(float x) { return int(x * magnification * pixelRatioX); }
  int _py(float y) { return int(y * magnification * pixelRatioY); }
  // b) 座標変換（ワールド座標→ピクセル座標）
  int px(float x) {
    float center = (wx_max + wx_min) / 2.0;
    return int((x - center) * pixelRatioX) + (this.width / 2);
  }
  int py(float y) {
    float center = (wy_max + wy_min) / 2.0;
    return this.height - (int((y - center) * pixelRatioY) + (this.height / 2));
  }
  // c) 逆座標変換（ピクセル座標→ワールド座標）
  float _rpx(int x) { return x / magnification / pixelRatioX; }
  float _rpy(int y) { return y / magnification / pixelRatioY; }

  float rpx(int x) {
    float center = (wx_max + wx_min) / 2.0;
//  return  ((x - offsetX) - (this.width / 2)) / pixelRatioX + center;
    return  (x - (this.width / 2)) / pixelRatioX + center;
  }
  float rpy(int y) {
    float center = (wy_max + wy_min) / 2.0;
//  return ((this.height - (y + offsetY)) - (this.height / 2)) / pixelRatioY + center;
    return ((this.height - y) - (this.height / 2)) / pixelRatioY + center;
  }
  // ビューポートに描画する最終処理（拡大縮小・並進・回転）
  int [] baseConv(int [] p, float x, float y) {
/*
    // 以下の処理はrotate(), translate(), scale()を使わずにビューポートの拡大縮小回転に対応した際のもの．
    // 一旦、ピボット座標を原点とし、Y軸正が上方向の座標系に変換する
    float cx, cy;
    cx = pivotX;
    cy = pivotY;
    x -= cx;
    y -= cy;
    y  = -y;
    // 拡大縮小
    x *= magnification;
    y *= magnification;
    // 回転
    float rx, ry;
    float angle = rotateAngle.getRadian();
    rx = x * cos(angle) - y * sin(angle);
    ry = x * sin(angle) + y * cos(angle);
    // 元の座標に戻す
    rx += cx;
    ry  = -ry;
    ry += cy;
    // オフセット
    rx +=  offsetX;
    ry += -offsetY;
    p[0] = int(rx);
    p[1] = int(ry);
*/
    // 後日，この関数自体が何か便利な役に立つかも知れないので一応，残しておくがいつまでたっても使い道が無いならば消す．
    p[0] = int(x);
    p[1] = int(y);
    return p;
  }
 
  // ビューポートに描画する最終処理（拡大縮小・並進・回転）の反転
  int [] reverseBaseConv(int [] p, float x, float y) {
    // オフセットを引く
    x -=  offsetX;
    y -= -offsetY;
    // 一旦、ピボット座標を原点とし、Y軸正が上方向の座標系に変換する
    float cx, cy;
    cx = pivotX;
    cy = pivotY;
    x -= cx;
    y -= cy;
    y  = -y;
    // 拡大縮小を元に戻す
    x /= magnification;
    y /= magnification;
    // 逆回転
    float rx, ry;
    float angle = rotateAngle.getRadian();
    rx = x * cos(-angle) - y * sin(-angle);
    ry = x * sin(-angle) + y * cos(-angle);
    // 元の座標に戻す
    ry  = -ry;
    rx += cx;
    ry += cy;
    p[0] = int(rx);
    p[1] = int(ry);
    return p;
  }

  // --------------------------------------------------
  // ビューポートの表示／非表示のダイナミックな切り替え
  // --------------------------------------------------
  // 状態
  boolean isViewVisible() { return visible; }
  // (a) 表示
  viewportClass viewVisible()   { visible = true; return this; }
  // (b) 非表示
  viewportClass viewUnvisible() { visible = false; return this; }
  // (c) 引数による指定
  viewportClass viewVisible(boolean flag) { if (flag) viewVisible(); else viewUnvisible(); return this; }
  // (d) トグル動作（変更後の状態を返す）
  boolean toggleViewVisible() {
    if (isViewVisible()) {
      viewUnvisible();
      return false;
    } else {
      viewVisible();
      return true;
    }
  }
  viewportClass setViewVisible(boolean flag) {
    viewVisible(flag);
    return this;
  }
  // ビューポートの移動
  viewportClass moveView(int x, int y)      { vx  = x;  vy  = y;  return this; }
  viewportClass moveViewRel(int dx, int dy) { vx += dx; vy += dy; return this; }

  // ----------------------------------
  // ビューポートの枠線や不透明度の設定
  // ----------------------------------
  // フレームカラー
  // (a-1) フレームカラーを取得（現在のフレームのみ）
  int frameColor() { return frameColor; }
  // (a-2) フレームカラーの設定（＋変更前の値）
  int frameColor(int c) {
    int oldcolor = frameColor();
    frameColor = c & #ffffff;
    return oldcolor;
  }
  viewportClass setFrameColor(int c) {
    frameColor(c);
    return this;
  }
  // フレームの表示／非表示
  // (b) フレームの表示／非表示
  boolean drawFrame() { return drawFrame; }
  boolean drawFrame(boolean flag) {
    boolean oldstatus = drawFrame;
    drawFrame = flag;
    return oldstatus;
  }
  viewportClass setDrawFrame(boolean flag) {
    drawFrame(flag);
    return this;
  }
  // フレーム幅
  // (c-1) フレーム幅を取得
  int frameWidth() { return frameWidth; }
  // (c-2) フレーム幅の設定（＋変更前の値）
  int frameWidth(int w) {
    int oldwidth = frameWidth();
    frameWidth = w;
    return oldwidth;
  }
  viewportClass setFrameWidth(int w) {
    frameWidth(w);
    return this;
  }

  // (a) 不透明度の取得
  int transparency() { return transparency; }
  // (b) 不透明度の設定（＋変更前の値）
  int transparency(int trans) {
    int oldtrans = transparency();
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    transparency = cdSet.fillTrans = cdSet.strokeTrans = trans;  // 塗り潰しや線の不透明度も変更する
    return oldtrans;
  }
  viewportClass setTransparency(int trans) {
    transparency(trans);
    return this;
  }

  // 背景色の設定
  // (a) 色を取得（不透明度は無視）
  int viewBgColor() { return bgColor; }
  // (b) 色を設定（不透明度は255）
  int viewBgColor(int c) {
    int oldcolor = viewBgColor();
    bgColor = c & #ffffff;
    return oldcolor;
  }
  viewportClass setViewBgColor(int c) {
    viewBgColor(c);
    return this;
  }

  // 背景を透明なビューポートにする／しない
  // (a) 状態を返す
  boolean transparentView() { return transparentView; }
  boolean transparentView(boolean flag) {
    if (flag) drawFrame = false;  // 透明なビューを指定されたら一緒に枠も消してしまう（お節介？）
    boolean oldstatus = transparentView();
    transparentView = flag;
    return oldstatus;
  }
  viewportClass setTransparentView(boolean flag) {
    transparentView(flag);
    return this;
  }

  // ----------------------------------------------
  // ピボット（拡大縮小および回転の中心座標）の設定
  // ----------------------------------------------
  // (a) 現在のピボット座標を返す
  PVector viewPivot() {
    PVector vp = new PVector(pivotX, pivotY);
    return vp;
  }
  // (b) ピボット座標の設定（＋変更前の値）
  // (b-1) 自由に指定
  PVector viewPivot(float x, float y) {
    PVector vp = viewPivot();
    pivotX = x; 
    pivotY = y;
    return vp;
  }
  viewportClass setViewPivot(float x, float y) {
    pivotX = x;  pivotY = y;
    return this;
  }
  // (b-2) ビューポート中心
  PVector viewPivotCenter() { return viewPivot(this.width / 2, this.height / 2);  }
  viewportClass setViewPivotCenter() { viewPivotCenter(); return this; }
  // (b-3) 左上，左下，右上，右下
  PVector viewPivotLT()  { return viewPivot(0.0,              0.0);               }
  PVector viewPivotLB()  { return viewPivot(0.0,              this.height - 1);   }
  PVector viewPivotRT()  { return viewPivot(this.width - 1,   0.0);               }
  PVector viewPivotRB()  { return viewPivot(this.width - 1,   this.height - 1);   }
  PVector viewPivotCT()  { return viewPivot(this.width / 2.0, 0.0);               }
  PVector viewPivotCB()  { return viewPivot(this.width / 2.0, this.height - 1);   }
  PVector viewPivotLC()  { return viewPivot(0.0,              this.height / 2.0); }
  PVector viewPivotRC()  { return viewPivot(this.width - 1,   this.height / 2.0); }

  viewportClass setViewPivotLT()  { viewPivotLT(); return this; }
  viewportClass setViewPivotLB()  { viewPivotLB(); return this; }
  viewportClass setViewPivotRT()  { viewPivotRT(); return this; }
  viewportClass setViewPivotRB()  { viewPivotRB(); return this; }
  viewportClass setViewPivotCT()  { viewPivotCT(); return this; }
  viewportClass setViewPivotCB()  { viewPivotCB(); return this; }
  viewportClass setViewPivotLC()  { viewPivotLC(); return this; }
  viewportClass setViewPivotRC()  { viewPivotRC(); return this; }

  // --------
  // 倍率関係
  // --------
  // (a) 現在の倍率を返す
  float viewScale() { return magnification; }
  // (b) 倍率の値を変更する（＋変更前の値を返す）
  float viewScale(float s) {
    float oldscale;
    if (s <= 0.0) s = 1.0;  // 不正な値
    oldscale = viewScale();
    magnification = s;
    return oldscale;
  }
  viewportClass setViewScale(float s) { viewScale(s); return this; }
  // ----------------------------
  // ビューポートの視点の回転角度
  // ----------------------------
  // (a) 現在の角度を返す
  float viewRotate() { return rotateAngle.get(); }
  // (b) 視点の角度を設定する（＋変更前の値を返す）
  float viewRotate(float a) {
    float oldangle = viewRotate();
//    if (a > TWO_PI) a -= TWO_PI; else if (a < -TWO_PI) a += TWO_PI;  // % を使うと誤差が生じる？という根拠のない小心な推測
    rotateAngle.set(a);
    return oldangle;
  }
  viewportClass setViewRotate(float a) { viewRotate(a); return this; }
  // (c) 視点の角度を増減する（＋変更前の値を返す）
  float viewRotateRel(float a) { return viewRotate(viewRotate() + a); }
  viewportClass setViewRotateRel(float a) { viewRotateRel(a); return this; }
  // ----------------
  // オフセット量関係
  // ----------------
  // (a) 現在のビューポートのオフセット量を返す
  // (a-1) ｘ方向
  float viewOffsetX() { return offsetX; }
  // (a-2) ｙ方向（上を正としている点に注意）
  float viewOffsetY() { return offsetY; }
  // (b) オフセット量を変更する（＋変更前の値を返す）：現在のビューポート
  // (b-1) ｘ方向
  float viewOffsetX(float offset) {
    float oldoffset = viewOffsetX();
    offsetX = offset;
    return oldoffset;
  }
  viewportClass setViewOffsetX(float offset) { viewOffsetX(offset); return this; }
  // (b-2) ｙ方向（上を正としている点に注意）
  float viewOffsetY(float offset) {
    float oldoffset = viewOffsetY();
    offsetY = offset;
    return oldoffset;
  }
  viewportClass setViewOffsetY(float offset) { viewOffsetY(offset); return this; }
  // (b-3) ｘ方向／ｙ方向同時：常にtrueを返す
  boolean viewOffset(float offx, float offy) {
    viewOffsetX(offx);
    viewOffsetY(offy);
    return true;
  }
  viewportClass setViewOffset(float offx, float offy) { viewOffset(offx, offy); return this; }

  // ビューポートに新座標系を設定する
  void resetView() {
    float pax, pay, pbx, pby, pcx, pcy;
    pg.endDraw();
    pg.beginDraw();
    pg.rotate( - rotateAngle.getRadian());
    pg.scale(magnification);
    pax = pivotX + offsetX;
    pay = pivotY - offsetY;
    pbx = rotate2Dsx(pax, pay, -rotateAngle.getRadian());
    pby = rotate2Dsy(pax, pay, -rotateAngle.getRadian());
    pcx = pbx - magnification * pivotX;
    pcy = pby - magnification * pivotY;
    pg.translate(pcx / magnification, pcy / magnification);
  }

  // ------
  // 色関係
  // ------
  // (a) fill()
  // (a-1) 現在の塗り潰し色を取得
  int fill() { return cdSet.fillColor; }
  // (a-2) 塗り潰し色を変更（＋現在の値を返す）
//int fill(int c) { return this.fill(c, cdSet.fillTrans); }
  int fill(int c) { cdSet.fillTrans = 255; return this.fill(c, cdSet.fillTrans); }
  // (a-3) 塗り潰し色を変更（不透明度対応＋現在の値を返す）
  int fill(int c, int trans) {
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    int oldcolor;
    oldcolor = this.fill();
    cdSet.fillColor = c & #ffffff;
    cdSet.fillTrans = trans;
    pg.fill(cdSet.fillColor, cdSet.fillTrans);
    cdSet.isFill = false;
    return oldcolor;
  }
  viewportClass setFill(int c, int trans) { this.fill(c, trans); return this; }
  viewportClass setFill(int c)            { this.fill(c);        return this; }
  // (a-4) 塗り潰し無し
  viewportClass noFill() {
    cdSet.isFill = false;
    pg.noFill();
    return this;
  }
  // (a-5) 再び塗り潰し
  viewportClass reFill() {
    this.fill(cdSet.fillColor, cdSet.fillTrans);
    return this;
  }
  // (b) stroke()
  // (b-1) 現在の線色を取得
  int stroke() { return cdSet.strokeColor; }
  // (b-2) 線色を変更（＋現在の値を返す）
  int stroke(int c) { return this.stroke(c, cdSet.fillTrans); }
  // (b-3) 線色を変更（不透明度対応＋現在の値を返す）
  int stroke(int c, int trans) {
    if (trans < 0)   trans = 0;
    if (trans > 255) trans = 255;
    int oldcolor;
    oldcolor = this.stroke();
    cdSet.strokeColor = c & #ffffff;
    cdSet.strokeTrans = trans;
    pg.stroke(cdSet.strokeColor, cdSet.strokeTrans);
    return oldcolor;
  }
  viewportClass setStroke(int c, int trans) { this.stroke(c, trans); return this; }
  viewportClass setStroke(int c)            { this.stroke(c);        return this; }
  // (b-4) 線無し
  viewportClass noStroke() {
    cdSet.isStroke = false;
    pg.noStroke();
    return this;
  }
  // (b-5) 再び線有り
  viewportClass reStroke() {
    this.stroke(cdSet.strokeColor, cdSet.strokeTrans);
    return this;
  }
  // 線幅
  // (a) 現在値読み出し
  float strokeWeight() { return cdSet.strokeWeight; }
  // (b) 設定（＋現在の値）
  float strokeWeight(float w) {
    float oldwidth = strokeWeight();
    pg.strokeWeight(w);
    cdSet.strokeWeight = w;
    return oldwidth;
  }
  viewportClass setStrokeWeight(float w) { this.strokeWeight(w); return this; }

  // 現在のビューポートを画面消去する
  // flagがtrueの場合は背景色が透明に設定されているビューポートの消去は行なわない
  viewportClass clrView(boolean flag) {
    if (transparentView) {
      if (!flag) pg.background(bgColor, 0);
    } else       pg.background(bgColor, transparency);
    return this;
  }
  viewportClass clrView() { return clrView(false); }

  // ------------------------------------------
  // 最終描画点（カレントポジション）を移動する
  // ------------------------------------------
  viewportClass _moveTo(int x, int y) {
    cdSet.vpx = x;
    cdSet.vpy = y;
    return this;
  }
  viewportClass moveTo(float x, float y) {
    cdSet.wpx = x;
    cdSet.wpy = y;
    return this;
  }
  viewportClass _moveToRel(int dx, int dy) {
    cdSet.vpx += dx;
    cdSet.vpy += dy;
    return this;
  }
  viewportClass moveToRel(float dx, float dy) {
    cdSet.wpx += dx;
    cdSet.wpy += dy;
    return this;
  }
  int  _getCpX() { return cdSet.vpx; }
  int  _getCpY() { return cdSet.vpy; }
  float getCpX() { return cdSet.wpx; }
  float getCpY() { return cdSet.wpy; }

  // 実際に描画する関数
  // 共有する作業用変数
  int [] tmp1 = new int[2];
  int [] tmp2 = new int[2];
  int [] tmp3 = new int[2];
  int [] tmp4 = new int[2];
//  int [] tmp1, tmp2, tmp3, tmp4;
  // ------------
  // 直線描画関係
  // ------------
  // 直線を描く（ピクセル座標）
  viewportClass _line(int x1, int y1, int x2, int y2) {
    tmp1 = baseConv(tmp1, x1, y1);
    tmp2 = baseConv(tmp2, x2, y2);
    pg.line(tmp1[0], tmp1[1], tmp2[0], tmp2[1]);
    _moveTo(x2, y2);
    return this;
  }
  // 直線を描く（ワールド）
  viewportClass line(float x1, float y1, float x2, float y2) {
    _line(px(x1), py(y1), px(x2), py(y2));
    moveTo(x2, y2);
    return this;
  }
  // 始点省略（絶対座標）
  viewportClass _lineTo(int x, int y)    { return _line(cdSet.vpx, cdSet.vpy, x, y); }
  viewportClass lineTo(float x, float y) { return this.line(cdSet.wpx, cdSet.wpy, x, y); }
  // 始点省略（相対座標）
  viewportClass _lineRel(int dx, int dy)    { return _line(cdSet.vpx, cdSet.vpy, cdSet.vpx + dx, cdSet.vpy + dy); }
  viewportClass lineRel(float dx, float dy) { return this.line(cdSet.wpx, cdSet.wpy, cdSet.wpx + dx, cdSet.wpy + dy); }
  // ----------
  // 点線を描く
  // ----------
  // (a) 絶対座標指定
  // (a-1) ピクセル座標
  viewportClass _dashedLine(int x0, int y0, int x1, int y1, int step) {
    float  px0, py0, px1, py1, normv;
    float  unitx, unity;
    float  dx, dy;
    int    countmax, count;
    boolean stroke = true;

    countmax = int(norm(x0, y0, x1, y1) / step);
    if (countmax <= 0) {
      _line(x0, y0, x1, y1);
      return this;
    }
    px0 = x0;
    py0 = y0;
    for (count = 0; count < countmax; count++) {
      dx = x1 - px0;
      dy = y1 - py0;
      normv = sqrt(dx * dx + dy * dy);
      if (normv < step) {
        _line(int(px0), int(py0), x1, y1);
        return this;
      }
      unitx = dx / normv;
      unity = dy / normv;
      px1 = px0 + unitx * step;
      py1 = py0 + unity * step;
      if (stroke) _line(int(px0), int(py0), int(px1), int(py1));
      stroke = !stroke;
      px0 = px1;
      py0 = py1;
    }
    _moveTo(x1, y1);
    return this;
  }
  // (a-2) ワールド座標系
// 混乱の元になるのでワールド座標系による点線間隔の指定は無しにする
/*
  viewportClass dashedLine(float x1, float y1, float x2, float y2, float step) {
    _dashedLine(px(x1), py(y1), px(x2), py(y2), int((_px(step) + _py(step)) / 2.0));
    moveTo(x2, y2);
    return this;
  }
*/
  viewportClass dashedLine(float x1, float y1, float x2, float y2, int step) {
    _dashedLine(px(x1), py(y1), px(x2), py(y2), step);
    moveTo(x2, y2);
    return this;
  }
  // (b) 始点省略（絶対座標）
  viewportClass _dashedLineTo(int x, int y, int step)      { return _dashedLine(cdSet.vpx, cdSet.vpy, x, y, step); }
  viewportClass dashedLineTo(float x, float y, int step)   { return  dashedLine(cdSet.wpx, cdSet.wpy, x, y, step); }
  // (c) 始点省略（相対座標）
  viewportClass _dashedLineRel(int dx, int dy, int step)      { return _dashedLine(cdSet.vpx, cdSet.vpy, cdSet.vpx + dx, cdSet.vpy + dy, step); }
  viewportClass dashedLineRel(float dx, float dy, int step)   { return  dashedLine(cdSet.wpx, cdSet.wpy, cdSet.wpx + dx, cdSet.wpy + dy, step); }

  // --------
  // 円を描く
  // --------
  // (a) 楕円を描く
  viewportClass _ellipse(int x, int y, int w, int h) {
    tmp1 = baseConv(tmp1, x, y);
/*
    if (cdSet.ellipseMode == CORNERS) {
      tmp2 = baseConv(tmp2, w, h);
      w = tmp2[0];
      h = tmp2[1];
    }
*/
    pg.ellipse(tmp1[0], tmp1[1], w, h);
    _moveTo(x, y);
    return this;
  }
  viewportClass ellipse(float x, float y, float w, float h) {
    if (cdSet.ellipseMode == CORNERS) _ellipse(px(x), py(y), px(w), py(h));
      else                            _ellipse(px(x), py(y), _px(w), _py(h));
    moveTo(x, y);
    return this;
  }
// 表示位置はワールド座標，大きさはピクセル座標．便利そうだがリスクの方が大きいので止める！
//  →　plot()を追加し，これで代用
/*
  void ellipse(float x, float y, int w, int h) {
    _ellipse(px(x), py(y), w, h);
    moveTo(x, y);
  }
*/
  // (b) 真円を描く
  viewportClass _ellipse(int x, int y, int r)      { return _ellipse(x, y, r, r); }
  viewportClass ellipse(float x, float y, float r) { return this.ellipse(x, y, r, r); }
//void ellipse(float x, float y, int r)   { this.ellipse(x, y, r, r); }  // plot()に変更
  viewportClass plot(float x, float y, int r) {
    int mode = this.ellipseMode(RADIUS);
    ellipse(x, y, _rpx(r));
    this.ellipseMode(mode);
    return this;
  }
  // (c) 真円を描く（半径のみ指定）
  viewportClass _ellipse(int  r) { return _ellipse(cdSet.vpx, cdSet.vpy, r); }
  viewportClass ellipse(float r) { return ellipse(cdSet.wpx, cdSet.wpy,  r); }
//void ellipse(int   r) { ellipse(cdSet.wpx, cdSet.wpy,  r); }  // plot()に変更
  viewportClass plot(int r)      { return plot(cdSet.wpx, cdSet.wpy, r); }

  // 楕円（ellipse）と円弧（arc）の座標指定のモード
  // CENTER（デフォルト）, RADIUS, CORNER, CORNERSが使える
  int ellipseMode() { return cdSet.ellipseMode; }
  int ellipseMode(int mode) {
    int oldmode = ellipseMode();
    pg.ellipseMode(mode);
    cdSet.ellipseMode = mode;
    return oldmode;
  }
  viewportClass setEllipseMode(int mode) { this.ellipseMode(mode); return this; }

  // -----------
  // 三角形を描く
  // -----------
  // (a) 三点指定
  viewportClass _triangle(int x1, int y1, int x2, int y2, int x3, int y3) {
    tmp1 = baseConv(tmp1, x1, y1);
    tmp2 = baseConv(tmp2, x2, y2);
    tmp3 = baseConv(tmp3, x3, y3);
    pg.triangle(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1]);
    _moveTo(int((x1 + x2 + x3) / 3.0), int((y1 + y2 + y3) / 3.0));  // 中央，だと思う
    return this;
  }
  viewportClass triangle(float x1, float y1, float x2, float y2, float x3, float y3) {
    _triangle(px(x1), py(y1), px(x2), py(y2), px(x3), py(y3));
    moveTo((x1 + x2 + x3) / 3.0, (y1 + y2 + y3) / 3.0);
    return this;
  }
  // 中心座標を指定（回転対応）
  viewportClass _triangle(int x, int y, int s, float angle) {
    int   dx1, dy1, dx2, dy2, dx3, dy3;
    float da;
    angleClass ang = new angleClass(angle);
    angle = ang.getRadian();
    da  = TWO_PI / 3.0;
    dx1 = 0;
    dy1 = s;
    dx2 = int(rotate2Dx(dx1, dy1, angle + da));
    dy2 = int(rotate2Dy(dx1, dy1, angle + da));
    dx3 = int(rotate2Dx(dx1, dy1, angle - da));
    dy3 = int(rotate2Dy(dx1, dy1, angle - da));
    dx1 = int(rotate2Dx(dx1, dy1, angle));
    dy1 = int(rotate2Dy(dx1, dy1, angle));
    _triangle(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3);
    _moveTo(x, y);
    return this;
  }
  viewportClass triangle(float x, float y, float s, float angle) {
    float dx1, dy1, dx2, dy2, dx3, dy3, da;
    angleClass ang = new angleClass(angle);
    angle = ang.get();
    da  = TWO_PI / 3.0;
    dx1 = 0;
    dy1 = s;
    dx2 = rotate2Dx(dx1, dy1, angle + da);
    dy2 = rotate2Dy(dx1, dy1, angle + da);
    dx3 = rotate2Dx(dx1, dy1, angle - da);
    dy3 = rotate2Dy(dx1, dy1, angle - da);
    dx1 = rotate2Dx(dx1, dy1, angle);
    dy1 = rotate2Dy(dx1, dy1, angle);
    this.triangle(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3);
    moveTo(x, y);
    return this;
  }
  // 中心座標を指定（回転なし）
  viewportClass _triangle(int x, int y, int s)         { return _triangle(x, y, s, 0.0); }
  viewportClass  triangle(float x, float y, float s)   { return  triangle(x, y, s, 0.0); }
  // サイズのみ指定（回転対応）
  viewportClass _triangle(int   s, float angle) { return _triangle(cdSet.vpx, cdSet.vpy, s, angle); }
  viewportClass  triangle(float s, float angle) { return  triangle(cdSet.wpx, cdSet.wpy, s, angle); } 
  // サイズのみ指定（回転なし）
  viewportClass _triangle(int   s) { return _triangle(s, 0.0); }
  viewportClass  triangle(float s) { return  triangle(s, 0.0); }
  
  // ------------
  // 四辺形を描く
  // ------------
  viewportClass _quad(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
    tmp1 = baseConv(tmp1, x1, y1);
    tmp2 = baseConv(tmp2, x2, y2);
    tmp3 = baseConv(tmp3, x3, y3);
    tmp4 = baseConv(tmp4, x4, y4);
    pg.quad(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1], tmp4[0], tmp4[1]);
    float [][] pt = new float[4][2];
    pt[0][0] = x1;
    pt[0][1] = y1;
    pt[1][0] = x2;
    pt[1][1] = y1;
    pt[2][0] = x3;
    pt[2][1] = y3;
    pt[3][0] = x4;
    pt[3][1] = y4;
    float [] xyG = new float[2];
    xyG = polygonG(pt);
    _moveTo(int(xyG[0]), int(xyG[1]));
    return this;
  }
  viewportClass quad(float x1, float y1, float x2, float y2, float x3, float y3, float x4, float y4) {
    _quad(px(x1), py(y1), px(x2), py(y2), px(x3), py(y3), px(x4), py(y4));
    float [][] pt = new float[4][2];
    pt[0][0] = x1;
    pt[0][1] = y1;
    pt[1][0] = x2;
    pt[1][1] = y1;
    pt[2][0] = x3;
    pt[2][1] = y3;
    pt[3][0] = x4;
    pt[3][1] = y4;
    float [] xyG = new float[2];
    xyG = polygonG(pt);
    moveTo(xyG[0], xyG[1]);
    return this;
  }
  viewportClass _quad(int dx1, int dy1, int dx2, int dy2, int dx3, int dy3)             { return _quad(cdSet.vpx, cdSet.vpy, dx1, dy1, dx2, dy2, dx3, dy3); }
  viewportClass  quad(float dx1, float dy1, float dx2, float dy2, float dx3, float dy3) { return  quad(cdSet.wpx, cdSet.wpy, dx1, dy1, dx2, dy2, dx3, dy3); }
  viewportClass _quadRel(int x, int y, int dx1, int dy1, int dx2, int dy2, int dx3, int dy3, int dx4, int dy4) {
   return _quad(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3, x + dx4, y + dy4);
  }
  viewportClass quadRel(float x, float y, float dx1, float dy1, float dx2, float dy2, float dx3, float dy3, float dx4, float dy4) {
   return quad(x + dx1, y + dy1, x + dx2, y + dy2, x + dx3, y + dy3, x + dx4, y + dy4);
  }
  viewportClass _quadRel(int dx1, int dy1, int dx2, int dy2, int dx3, int dy3, int dx4, int dy4) {
    return _quad(cdSet.vpx + dx1, cdSet.vpy + dy1, cdSet.vpx + dx2, cdSet.vpy + dy2, cdSet.vpx + dx3, cdSet.vpy + dy3, cdSet.vpx + dx4, cdSet.vpy + dy4);
  }
  viewportClass quadRel(float dx1, float dy1, float dx2, float dy2, float dx3, float dy3, float dx4, float dy4) {
    return quad(cdSet.wpx + dx1, cdSet.wpy + dy1, cdSet.wpx + dx2, cdSet.wpy + dy2, cdSet.wpx + dx3, cdSet.wpy + dy3, cdSet.wpx + dx4, cdSet.wpy + dy4);
  }
  // --------------------------------------------------------
  // 長方形を描く
  // ビューポートの視点の回転に対応するためにquad()で描画する
  // --------------------------------------------------------
  // (a) 左上始点
  viewportClass _rect(int x, int y, int w, int h) {
    switch (cdSet.rectMode) {
      case CORNERS :
        _quad(x, y, x, h, w, h, w, y);
        break;
      case CENTER :
        int dx = w / 2;
        int dy = h / 2;
        _quad(x - dx, y - dy, x - dx, y + dy, x + dx, y + dy, x + dx, y - dy);
        break;
      case RADIUS :
        _quad(x - w,  y - h,  x - w,  y + h,  x + w,  y + h,  x + w,  y - h );
        break;
      default :  // CORNER（デフォルト）
        _quad(x, y, x, y + h, x + w, y + h, x + w, y);
        break;
    }
    _moveTo(x, y);
    return this;
  }
  viewportClass rect(float x, float y, float w, float h) {
    if (cdSet.rectMode == CORNERS) _rect(px(x), py(y),  px(w),  py(h));
      else                         _rect(px(x), py(y), _px(w), _py(h));
    moveTo(x, y);
    return this;
  }
  viewportClass _rect(int w, int h)     { return _rect(cdSet.vpx, cdSet.vpy, w, h); }
  viewportClass  rect(float w, float h) { return  rect(cdSet.wpx, cdSet.wpy, w, h); }
  // (b) それ以外の指定方法はrectMode()で指定すれば良いので省略
  // CORNER（デフォルト）, CORNERS, CENTER, RADIUSが使える
  int rectMode() { return cdSet.rectMode; }
  int rectMode(int mode) {
    int oldmode = rectMode();
    pg.rectMode(mode);
    cdSet.rectMode = mode;
    return oldmode;
  }
  viewportClass setRectMode(int mode) { this.rectMode(mode); return this; }
  // ---------
  // 円弧を描く
  // ---------
  viewportClass _arc(int x, int y, int w, int h, float start, float stop) {
    tmp1 = baseConv(tmp1, x, y);
    angleClass startAngle, stopAngle;
    startAngle = new angleClass(start);
    stopAngle  = new angleClass(stop);
    pg.arc(tmp1[0], tmp1[1], w, h, startAngle.getRadian(), stopAngle.getRadian());
    _moveTo(x, y);
    return this;
  }
  viewportClass _arc(int x, int y, int r, float start, float stop) { return _arc(x, y, r, r, start, stop); }
  viewportClass _arc(int w, int h, float start, float stop)        { return _arc(cdSet.vpx, cdSet.vpy, w, h, start, stop); }
  viewportClass _arc(int r, float start, float stop)               { return _arc(r, r, start, stop); }
  viewportClass arc(float x, float y, float w, float h, float start, float stop) {
    float temp;
    temp  = start;
    start = -stop;
    stop  = -temp;
    _arc(px(x), py(y), _px(w), _py(h), start, stop);
    moveTo(x, y);
    return this;
  }
  viewportClass arc(float x, float y, float r, float start, float stop) { return arc(x, y, r, r, start, stop); }
  viewportClass arc(float w, float h, float start, float stop)          { return arc(cdSet.wpx, cdSet.wpy, w, h, start, stop); }
  viewportClass arc(float r, float start, float stop)                   { return arc(r, r, start, stop); } 
  // ------------
  // 点を描画する
  // ------------
  viewportClass _point(int x, int y) {
    tmp1 = baseConv(tmp1, x, y);
    pg.point(tmp1[0], tmp1[1]);
    _moveTo(x, y);
    return this;
  }
  viewportClass _point() { return _point(cdSet.vpx, cdSet.vpy); }
  viewportClass point(float x, float y) {
    _point(px(x), py(y));
    moveTo(x, y);
    return this;
  }
  viewportClass point() { return point(cdSet.wpx, cdSet.wpy); }
  // --------
  // 頂点関係
  // --------
  // modeの選択肢は
  // LINES，LINE_STRIP，LINE_LOOP，TRIANGLES，TRIANGLE_FAN，TRIANGLE_STRIP，QUADS，QUAD_STRIP，POLYGON
  viewportClass beginShape(int mode) {
    pg.beginShape(mode);
    return this;
  }
  // mode省略時はPOLYGON
  viewportClass beginShape() {
    pg.beginShape();
    return this;
  }
  viewportClass endShape() {
    pg.endShape();
    return this;
  }
  // 頂点
  viewportClass _vertex(int x, int y) {
    tmp1 = baseConv(tmp1, x, y);  // 回転に正しく対応できていない
    pg.vertex(tmp1[0], tmp1[1]);
    _moveTo(x, y);
    return this;
  }
  viewportClass _vertex() { return _vertexRel(0, 0); }
  viewportClass  vertex(float x, float y) {
    _vertex(px(x), py(y));
    moveTo(x, y);
    return this;
  }
  // 頂点（テクスチャ用）：視点の回転には対応できないと思う
  viewportClass _vertex(float x, float y, float u, float v) { // 少々理解不足かも．x,yを省略すると頂点と区別が付かなくなる
    tmp1 = baseConv(tmp1, x, y);
    if (textureMode() == NORMALIZED) pg.vertex(x, y, u, v);
      else                           pg.vertex(tmp1[0], tmp1[1], u, v);
    return this;
  }
  // 頂点
  viewportClass vertex() { return vertexRel(0.0, 0.0); }
  viewportClass _vertexRel(int   dx, int   dy) { return _vertex(cdSet.vpx + dx, cdSet.vpy + dy); }
  viewportClass  vertexRel(float dx, float dy) { return  vertex(cdSet.wpx + dx, cdSet.wpy + dy); }
  // 頂点（テクスチャ用）：視点の回転には対応できないと思う
  viewportClass vertex(float x, float y, float u, float v) { // 少々理解不足かも．x,yを省略すると頂点と区別が付かなくなる
    if (textureMode() == NORMALIZED) _vertex(x, y, u, v);
      else                           _vertex(px(x), py(y), u, v);
    pg.endDraw();
    return this;
  }
  // 頂点（曲線）
  viewportClass _curveVertex(int x, int y) {
    tmp1 = baseConv(tmp1, x, y);
    pg.curveVertex(tmp1[0], tmp1[1]);
    _moveTo(x, y);
    return this;
  }
  viewportClass _curveVertex() { return _curveVertex(cdSet.vpx, cdSet.vpy); }
  viewportClass curveVertex(float x, float y) {
    _curveVertex(px(x), py(y));
    moveTo(x, y);
    return this;
  }
  viewportClass curveVertex() { return curveVertex(cdSet.wpx, cdSet.wpy); }
  viewportClass _curveVertexRel(int   dx, int   dy) { return _curveVertex(cdSet.vpx + dx, cdSet.vpy + dy); }
  viewportClass  curveVertexRel(float dx, float dy) { return  curveVertex(cdSet.wpx + dx, cdSet.wpy + dy); }
  viewportClass _bezierVertex(int cx1, int cy1, int cx2, int cy2, int x, int y) {
    tmp1 = baseConv(tmp1, cx1, cy1);
    tmp2 = baseConv(tmp2, cx2, cy2);
    tmp3 = baseConv(tmp3, x,   y  );
    pg.bezierVertex(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1]);
    _moveTo(x, y);
    return this;
  }
  viewportClass _bezierVertex(int cx1, int cy1, int cx2, int cy2) {
    _bezierVertex(cx1, cy1, cx2, cy2, cdSet.vpx, cdSet.vpy);
    return this;
  }
  viewportClass bezierVertex(float cx1, float cy1, float cx2, float cy2, float x, float y) {
    _bezierVertex(px(cx1), py(cy1), px(cx2), py(cy2), px(x), py(y));
    moveTo(x, y);
    return this;
  }
  viewportClass bezierVertex(float cx1, float cy1, float cx2, float cy2) {
    bezierVertex(cx1, cy1, cx2, cy2, cdSet.wpx, cdSet.wpy);
    return this;
  }
  viewportClass _bezierVertexRel(int dcx1, int dcy1, int dcx2, int dcy2, int dx, int dy) {
    _bezierVertexRel(cdSet.vpx + dcx1, cdSet.vpy + dcy1, cdSet.vpx + dcx2, cdSet.vpy + dcy2, cdSet.vpx + dx, cdSet.vpy + dy);
    return this;
  }
  viewportClass bezierVertexRel(float dcx1, float dcy1, float dcx2, float dcy2, float dx, float dy) {
    bezierVertexRel(cdSet.wpx + dcx1, cdSet.wpy + dcy1, cdSet.wpx + dcx2, cdSet.wpy + dcy2, cdSet.wpx + dx, cdSet.wpy + dy);
    return this;
  }
  viewportClass texture(PImage img) {
    pg.texture(img);
    return this;
  }
  // テクスチャーモード(IMAGE, NORMALIZED)
  int textureMode() { return cdSet.textureMode; }
  int textureMode(int mode) {
    int oldmode = textureMode();
    pg.textureMode(mode);
    cdSet.textureMode = mode;
    return oldmode;
  }
  viewportClass setTextureMode(int mode) { this.textureMode(mode); return this; }
  // --------
  // 曲線関係
  // --------
  // (1) curve
  viewportClass _curve(int a1x, int a1y, int x1, int y1, int x2, int y2, int a2x, int a2y) {
    tmp1 = baseConv(tmp1, a1x, a1y);
    tmp2 = baseConv(tmp2, x1,  y1 );
    tmp3 = baseConv(tmp3, x2,  y2 );
    tmp4 = baseConv(tmp4, a2x, a2y);
    pg.curve(tmp1[0], tmp1[1], tmp2[0], tmp2[1], tmp3[0], tmp3[1], tmp4[0], tmp4[1]);
    _moveTo(x2, y2);
    return this;
  }
  viewportClass curve(float a1x, float a1y, float x1, float y1, float x2, float y2, float a2x, float a2y) {
    _curve(px(a1x), py(a1y), px(x1), py(y1), px(x2), py(y2), px(a2x), py(a2y));
    moveTo(x2, y2);
    return this;
  }
  viewportClass _curveTo(int a1x, int a1y, int x2, int y2, int a2x, int a2y)            { return _curve(a1x, a1y, cdSet.vpx, cdSet.vpy, x2, y2, a2x, a2y); }
  viewportClass curveTo(float a1x, float a1y, float x2, float y2, float a2x, float a2y) { return  curve(a1x, a1y, cdSet.wpx, cdSet.wpy, x2, y2, a2x, a2y); }
  // (2) パラメータ
  float curveTightness() { return cdSet.curveTightness; }
  float curveTightness(float squishy) {
    float oldparam = curveTightness();
    pg.curveTightness(squishy);
    cdSet.curveTightness = squishy;
    return oldparam;
  }
  int curveDetail() { return cdSet.curveDetail; }
  int curveDetail(int detail) {
    int oldparam = curveDetail();
    pg.curveDetail(detail);
    cdSet.curveDetail = detail;
    return oldparam;
  }
  // (3) 簡略に曲線を描くための拡張
  // ワールド座標系
  void  drawCurve() {  curve(cc.x1a,  cc.y1a,  cc.x2a,  cc.y2a,  cc.x3a,  cc.y3a,  cc.x4a,  cc.y4a);  }
  void _drawCurve() { _curve(cc._x1a, cc._y1a, cc._x2a, cc._y2a, cc._x3a, cc._y3a, cc._x4a, cc._y4a); }
  viewportClass curveStart(float x, float y) { cc.start(x, y);                 return this; }
  viewportClass curveStart()                 { cc.start(cdSet.wpx, cdSet.wpy); return this; }
  viewportClass curveAdd(float x,  float y)  { cc.add(x, y);      if (cc.status >= 3) drawCurve(); return this; }
  viewportClass curveRel(float dx, float dy) { cc.addRel(dx, dy); if (cc.status >= 3) drawCurve(); return this; }
  viewportClass curveEnd()                   { cc.end();          if (cc.status == 0) drawCurve(); return this; }
  // ピクセル座標
  viewportClass _curveStart(int x, int y)  { cc._start(x, y);                 return this; }
  viewportClass _curveStart()              { cc._start(cdSet.vpx, cdSet.vpy); return this; }
  viewportClass _curveAdd(int x,   int y)  { cc._add(x, y);      if (cc._status >= 3) _drawCurve(); return this; }
  viewportClass _curveRel(int dx,  int dy) { cc._addRel(dx, dy); if (cc._status >= 3) _drawCurve(); return this; }
  viewportClass _curveEnd()                { cc._end();          if (cc._status == 0) _drawCurve(); return this; }

  // ----------------
  // テキスト表示関係
  // ----------------
  viewportClass prepareFont()             { return prepareFont(cdSet.textSize);  }
  viewportClass prepareFont(int fontsize) { return prepareFont(_FONTNAME, fontsize); }
  viewportClass prepareFont(String fontname, int fontsize) {
    if (crowbar.status != 0) {
      println("エラー: フォントの宣言はOptions()かSetup()で行って下さい，");
      return this;
    }
    font = createFont(fontname, fontsize, false);  // フォントを変換
    pg.textFont(font);         // フォントを設定
//  pg.textMode(SCREEN);       // SCREENに設定すると表示されない
    pg.textSize(fontsize);     // フォントサイズを設定（不要？）
    pg.textAlign(cdSet.textAlignH, cdSet.textAlignV);   // 配置
//  pg.textLeading(cdSet.textLeading);
    this.textLeading(int((this.textAscent() + this.textDescent()) * 1.2));
    cdSet.textSize = fontsize;
    return this;
  }
  // フォントサイズ
  int textSize() { return cdSet.textSize; }
  int textSize(int size) {
    int oldsize = cdSet.textSize;
    pg.textSize(size);
    cdSet.textSize = size;
    return oldsize;
  }
  viewportClass setTextSize(int size) { this.textSize(size); return this; }
  // 割り当て
  // LEFT, CENTER, RIGHT : 水平方向（デフォルトはLEFT）
  // TOP, BOTTOM, CENTER, BASELINE : 垂直方向（デフォルトはBASELINE）
  int textAlign()  { return cdSet.textAlignH; }
  int textAlignH() { return cdSet.textAlignH; }
  int textAlignV() { return cdSet.textAlignV; }
  int textAlign(int align) {
    int oldalign = cdSet.textAlignH;
    pg.textAlign(align);
    cdSet.textAlignH = align;
    return oldalign;
  }
  viewportClass textAlign(int align, int yalign) {
    pg.textAlign(align, yalign);
    cdSet.textAlignH = align;
    cdSet.textAlignV = yalign;
    return this;
  }
  // 行間
  int textLeading() { return cdSet.textLeading; }
  int textLeading(int dist) {
    int oldleading = cdSet.textLeading;
    pg.textLeading(dist);
    cdSet.textLeading = dist;
    return oldleading;
  }
  // ベースラインからの距離
  float textAscent() {
    float h;
    h = pg.textAscent();
    return h;
  }
  float textDescent() {
    float h;
    h = pg.textDescent();
    return h;
  }
  float textWidth(String str) {
    float w;
    w = pg.textWidth(str);
    return w;
  }
  // 実際に描画する
  float _text(String str, int x, int y) {
    tmp1 = baseConv(tmp1, x, y);  // 回転に正しく対応できていない
    pg.fill(cdSet.textColor);
    pg.text(str, tmp1[0], tmp1[1]);
    pg.fill(cdSet.fillColor);
    _moveTo(x + int(pg.textWidth(str)), y);
    return this.textWidth(str);
  }
  float _text(String str) {
    return _text(str, cdSet.vpx, cdSet.vpy);
  }
  float text(String str, float x, float y) {
    float w;
    w = _text(str, px(x), py(y)) / pixelRatioX;
    moveTo(x + w, y);
    return w;
  }
  float text(String str) {
    return _text(str);
  }

  viewportClass _setText(String str, int x, int y)    { _text(str, x, y); return this; }
  viewportClass _setText(String str)                  { _text(str);       return this; }
  viewportClass setText(String str, float x, float y) {  text(str, x, y); return this; }
  viewportClass setText(String str)                   {  text(str);       return this; }

  // 文字色関係
  // (a) 文字色の取得
  int viewTextColor() {
    return cdSet.textColor;
  }
  // (b) 文字色の設定（＋変更前の値を返す）
  int viewTextColor(int c) {
    int oldcolor = viewTextColor();
    cdSet.textColor = c & #ffffff;
    return oldcolor;
  }
  viewportClass setViewTextColor(int c) { viewTextColor(c); return this; }

  // --------
  // 画像関係
  // --------
  // 読み込み（loadImage()はPGraphicsを使う必要は無い．なお，グローバルで実行してもダメ）
  // モード (CORNER, CORNERS)
  // image(), copy()に影響を与える
  int imageMode() { return cdSet.imageMode; }
  int imageMode(int mode) {
    int oldmode = this.imageMode();
    pg.imageMode(mode);
    cdSet.imageMode = mode;
    return oldmode;
  }
  viewportClass setImageMode(int mode) {
    this.imageMode(mode);
    return this;
  }
  // 出力
  // (a) ピクセル座標
  viewportClass _image(PImage img, int x, int y, int w, int h) {
    pg.image(img, x, y, w, h);
    return this;
  }
  viewportClass _image(PImage img, int x, int y) { pg.image(img, x, y);  return this; }
  viewportClass _image(PImage img) { _image(img, cdSet.vpx, cdSet.vpy);  return this; }
  // (b) ワールド座標
  viewportClass image(PImage img, float x, float y, float w, float h) {
    _image(img, px(x), py(y), _px(w), _py(h));
    return this;
  }
  viewportClass image(PImage img, float x, float y) { _image(img, px(x), py(y)); return this; }
  viewportClass image(PImage img) { _image(img, px(cdSet.wpx), py(cdSet.wpy));   return this; }
  // 色や不透明度
  viewportClass tint(float gray)                               { pg.tint(gray);                   return this; }
  viewportClass tint(float gray, float alpha)                  { pg.tint(gray, alpha);            return this; }
  viewportClass tint(float value1, float value2, float value3) { pg.tint(value1, value2, value3); return this; }
  viewportClass tint(color c)                                  { pg.tint(c);                      return this; }
  viewportClass tint(color c, float alpha)                     { pg.tint(c, alpha);               return this; }
  // 取り消し
  viewportClass noTint() { pg.noTint(); return this; }

  // ピクセル関連
  // バッファの確保
  PImage createImageRGB(float w, float h)   { return createImage(_px(w), _py(h), RGB); }
  PImage createImageARGB(float w, float h)  { return createImage(_px(w), _py(h), ARGB); }
  PImage createImageALPHA(float w, float h) { return createImage(_px(w), _py(h), ALPHA); }
  // 読み込み
  viewportClass loadPixels() {
    pg.loadPixels();
    pixels = pg.pixels;
    return this;
  }
  // 更新
  viewportClass updatePixels() { pg.updatePixels(); return this; }

  PImage _get()                           { PImage img = pg.get();           return img;   }
  color  _get(int x, int y)               { color pixel = pg.get(x, y);      return pixel; }
  PImage _get(int x, int y, int w, int h) { PImage img = pg.get(x, y, w, h); return img;   }
  PImage get()                                   { return _get(); }
  color  get(float x, float y)                   { return _get(px(x), py(y)); }
  PImage get(float x, float y, float w, float h) { return _get(px(x), py(y), _px(w), _py(h)); }

  viewportClass _set(int x, int y, color c)       { pg.set(x, y, c);   return this; }
  viewportClass _set(int x, int y, PImage img)    { pg.set(x, y, img); return this; }
  viewportClass _set(PImage img)                  { pg.set(0, 0, img); return this; }
  viewportClass set(float x, float y, color c)    { return _set(px(x), py(y), c);   }
  viewportClass set(float x, float y, PImage img) { return _set(px(x), py(y), img); }
  viewportClass set(PImage img)                   { return _set(img);               }
  viewportClass filter(int mode)              { pg.filter(mode);        return this; }
  viewportClass filter(int mode, float level) { pg.filter(mode, level); return this; }
  // copy() : ビューポート間をまたぐコピーは考えていない．自力で頑張れ．コピー先のサイズ指定省略版は用意した．
  viewportClass _copy(int x, int y, int w, int h, int dx, int dy, int dw, int dh) {
    pg.copy(x, y, w, h, dx, dy, dw, dh);
    return this;
  }
  viewportClass _copy(int x, int y, int w, int h, int dx, int dy) { return _copy(x, y, w, h, dx, dy, w, h); }
  viewportClass _copy(PImage img, int x, int y, int w, int h, int dx, int dy, int dw, int dh) {
    pg.copy(img, x, y, w, h, dx, dy, dw, dh);
    return this;
  }
  viewportClass _copy(PImage img, int x, int y, int w, int h, int dx, int dy) { return _copy(img, x, y, w, h, dx, dy, w, h); }
  viewportClass copy(float x, float y, float w, float h, float dx, float dy, float dw, float dh) {
    pg.copy(px(x), py(y), _px(w), _py(h), px(dx), py(dy), _px(dw), _py(dh));
    return this;
  }
  viewportClass copy(float x, float y, float w, float h, float dx, float dy) { return copy(x, y, w, h, dx, dy, w, h); }
  viewportClass copy(PImage img, float x, float y, float w, float h, float dx, float dy, float dw, float dh) {
    pg.copy(img, px(x), py(y), _px(w), _py(h), px(dx), py(dy), _px(dw), _py(dh));
    return this;
  }
  viewportClass copy(PImage img, float x, float y, float w, float h, float dx, float dy) { return copy(img, x, y, w, h, dx, dy, w, h); }

  // ビューポートのキャプチャ（動作未確認）：もしかしたらプロセスコントロールが必要かも知れない
  viewportClass save(String fname) {
    pg.save(fname);
    return this;
  }
/*
  void saveFrame() {    // saveFrame()はPGraphicsに存在しないようだ
    pg.saveFrame();
  }
  void saveFrame(String format) {  // saveFrame()はPGraphicsに存在しないようだ
    pg.saveFrame(format);
  }
*/

  // --------------------------------------------------------
  // マウスによるビューポート内クリックに対するレスポンス関係
  // --------------------------------------------------------
  // (a) マウスポインターがこのビューポート上にあるか？（上に別のビューポートが重なっているかどうかは気にしない．クリックされたかどうかも関係ない）
  //     クリックされたかどうかはuserClicked()でイベント駆動にする．
  boolean mouseover() {
    if (visible != true) return false;
    if ((mouseX >= vx) && (mouseX <= vx + this.width) && (mouseY >= vy) && (mouseY <= vy + this.height)) return true;
    return false;
  }
  // (b) マウスポインターの座標を返します．領域外であっても返します（負の値を返す場合もあります）．
  // (b-1) ピクセル座標
  int _viewMouseX() {
    int [] tmp = new int [2];
    tmp = reverseBaseConv(tmp, mouseX - vx, mouseY -vy);
    return tmp[0];
  }
  int _viewMouseY() {
    int [] tmp = new int [2];
    tmp = reverseBaseConv(tmp, mouseX - vx, mouseY - vy);
    return tmp[1];
  }
  // (b-2) ワールド座標
  float viewMouseX() { return rpx(_viewMouseX()); }
  float viewMouseY() { return rpy(_viewMouseY()); }
}

// Tomahawk本体クラス（Crowbarに継承されます）
class tomahawkClass {
  protected int               viewMax;       // 宣言済みのビューポートの数
  protected int               cvNumber;      // 現在注目しているビューポートの番号
  protected viewportClass     cv;            // 現在注目しているビューポートクラス
  protected viewportClass []  viewStack;     // 宣言済みビューポート
  protected viewportClass []  cloneStack;    // Options(), Setup()で宣言した状態をバックアップ＆リストアするための領域
  protected int []            layerStack;    // ビューポートの表示順のスタック(viewLayerPush(), viewLayerPop()専用）
  protected int               tomahawkMode;  // mode == 0: noTomahawk, 1: Basic, 2: Advanced
  tomahawkClass(int mode) {
//  background(_BGCOLOR);  // Crowbarで指定する背景色で初期化
    viewStack    = (viewportClass [])new viewportClass[1];  // 配列を一つだけ確保する（あとで宣言される度に拡張する）
    viewStack[0] = null;
    cloneStack    = (viewportClass [])new viewportClass[1];  // 配列を一つだけ確保する（あとで宣言される度に拡張する）
    cloneStack[0] = null;
    cvNumber     = 0;
    cv           = null;
    viewMax      = 0;
    layerStack   = null;
    tomahawkMode = mode;
  }
  // 現在のビューポートの状態をバックアップ
  void backupViewports() {
    int i;
    if (cloneStack == null) return;
    for (i = 0; i < viewMax; i++) cloneStack[i].clone(viewStack[i]);
  }
  // バックアップされたビューポートの状態をリストアする
  void restoreViewports() {
    int i;
    if (cloneStack == null) return;
    for (i = 0; i < viewMax; i++) {
      viewStack[i].clone(cloneStack[i]);
      viewStack[i].restore();
    }
    layerStack = null;  // 初期化（つまりPushされていない状態）
    crowbar.clrViewAll();
  }
  // デバッグ用（全ビューの情報を表示する）
  void displayAllViewData() {
    int  i;
    viewportClass vp;
    for (i = 0; i < viewMax; i++) {
      vp = viewStack[i];
      print(str(i) + ":[" + str(vp.serialNumber) + "]" + vp.label + "/(" + str(vp.vx) + "," + str(vp.vy) + "," + str(vp.width) + "," + str(vp.height) + ")");
      print("-(" + str(vp.wx_min) + "," + str(vp.wy_min) + "," + str(vp.wx_max) + "," + str(vp.wy_max) + ")");
      print(" Color=#" + hex(vp.viewBgColor()) + ":Ratio =[" + str(vp.pixelRatioX) + "," + str(vp.pixelRatioY) + "]:Scale=" + str(vp.viewScale()));
      print(" Offset=" + str(vp.viewOffsetX()) + "," + str(vp.viewOffsetY()) + "Visible:" + str(vp.visible));
      println();
    }
  }
  
  // ----------------
  // ビューポート関連
  // ----------------
  // ビューポートの視点（変換マトリックス）の保存（ゼロの状態）と再設定
  void resetView(boolean reset) {
    int i;
    if (viewMax < 1) return;  // ビューポートが存在しない
    if (!reset) for (i = 0; i < viewMax; i++) getView(i).pg.pushMatrix();
      else      for (i = 0; i < viewMax; i++) getView(i).resetView();
  }

  // ビューポートの新規作成（成功した場合はビュー番号を返す．失敗した場合は -1 を返す）
  // (a) ピクセル単位で指定
  viewportClass createView(String label, int x, int y, int w, int h) {
    // エラーチェック
    if (crowbar.status != 0) {
      println("エラー：ビューポートの作成はOptions()かSetup()内で行なって下さい．");
      crowbar.halt();
      return null;
    }
    if (checkView(label) != null) {
      println("エラー：指定されたラベルと同じビューポートが既に作成済みです（重複）／" + label);
      crowbar.halt();
      return null;
    }
    // 領域確保
    if (viewMax > 0) {
      viewStack  = (viewportClass [])expand(viewStack,  viewMax + 1);  // 最初だけは配列数１で確保されているため
      cloneStack = (viewportClass [])expand(cloneStack, viewMax + 1);  // 最初だけは配列数１で確保されているため
    }
    if (viewStack.length != viewMax + 1) {
      println("エラー：メモリの確保に失敗しました．");
      crowbar.halt();
      return null;  // 正しく配列の拡大ができなかった場合
    }
    // 追加処理
    viewMax++;
    cvNumber = viewMax - 1;
    if ((cv = viewStack[cvNumber] = new viewportClass(label, x, y, w, h)) == null) {
      println("エラー：メモリの確保に失敗しました．");
      crowbar.halt();
      return null;  // 正しく配列の拡大ができなかった場合
    }
    cloneStack[cvNumber] = new viewportClass(label, x, y, w, h);                               // バックアップ領域を確保
    viewStack[cvNumber].serialNumber = cvNumber;  // 作成順のビューポート番号
    viewStack[cvNumber].layerLevel   = cvNumber;  // 表示順
    // 色情報などのデフォルト設定
    cv.fill(#ffffff,   cv.transparency); // 透明度に対応．初期色
    cv.stroke(#000000, cv.transparency); // 透明度に対応．初期色
    cv.world(0, 0, w, h);
    return cv;
  }
  // (b) 割合で指定する
  viewportClass createView(String label, int x, int y, float w, float h) {
    if ((w <= 0.0) || (h <= 0.0)) return null;
    // if (w >= 1.0) w = 1.0;  // 別に1.0よりも大きくてもいいか
    // if (h >= 1.0) h = 1.0;
    return createView(label, x, y, int(crowbar.screenSetting.gwx * w), int(crowbar.screenSetting.gwy * h));
  }
  // (c) 全画面指定のビューポートを作成する
  viewportClass createView(String label) {
    return createView(label, 0, 0, width, height);
  }

  // ビューポートを分割する．
  // (a) 水平方向
  // (a-1) ビューポート番号
  viewportClass splitViewH(int v, String label, float rate) {
    // エラーチェック
    if ((rate <= 0) || (rate >= 1.0)) {
      println("エラー：分割割合は 0 < rate < 1.0の範囲で指定して下さい．");
      crowbar.halt();
      return null;
    }
    if (isValidViewNumber(v) == false) {
      crowbar.halt(); // 不正なビュー番号
      return null;
    }
    return splitViewH(v, label, int(rate * getView(v).width));
  }
  viewportClass splitViewH(int v, String label, int w) {
    // エラーチェック
    if (isValidViewNumber(v) == false) {
      crowbar.halt(); // 不正なビュー番号
      return null;
    }
    // 分割
    viewportClass vp = getView(v);
    int xL, yL, wL, hL;  // 左（元）
    int xR, yR, wR, hR;  // 右（新規）
    // 変わらないもの
    xL = vp.vx;
    yL = yR = vp.vy;
    hL = hR = vp.height;
    // widthによって変わるもの
    if (w > 0) {
      // 左の幅指定
      wL = w;
      wR = vp.width - wL;      
    } else {
      // 右の幅指定
      wR = -w;
      wL = vp.width - wR;
    }
    xR = xL + wL;
    // チェック
    if ((wL <= 0) || (wR <= 0)) {
      println("エラー：分割されたビューポートの一方の幅がゼロ以下です／" + label);
      crowbar.halt();
      return null;
    }
    // 左の縮小
    vp.width = wL;
    cv.world(0, 0, vp.width, vp.height);
    // 終わったら右を新規作成
    if (createView(label, xR, yR, wR, hR) == null) {
      println("エラー：新しいビューポートの作成に失敗しました．");
      crowbar.halt();
      return null;
    }
    cv.cloneAttr(vp);  // 属性のコピー
    return cv;
  }
  // (a-2) ビューポートラベル
  viewportClass splitViewH(String target, String label, float rate) {
    viewportClass vp = getView(target);
    if (vp == null) {
      crowbar.halt();
      return null;
    }
    return splitViewH(vp.serialNumber, label, rate);
  }
  viewportClass splitViewH(String target, String label, int w) {
    viewportClass vp = getView(target);
    if (vp == null) {
      crowbar.halt();
      return null;
    }
    return splitViewH(vp.serialNumber, label, w);
  }
  // (a-3) 現在のビューポート
  viewportClass splitViewH(String label, float rate) { return splitViewH(cvNumber, label, rate); }
  viewportClass splitViewH(String label, int w)      { return splitViewH(cvNumber, label, w);    }

  // (b) 垂直方向
  // (b-1) 指定したビューポート番号
  viewportClass splitViewV(int v, String label, float rate) {
    // エラーチェック
    if ((rate <= 0) || (rate >= 1.0)) {
      println("エラー：分割割合（第二引数）は 0 < rate < 1.0の範囲でして下さい．");
      crowbar.halt();
      return null;
    }
    if (isValidViewNumber(v) == false) {
      crowbar.halt(); // 不正なビュー番号
      return null;
    }
    return splitViewV(v, label, int(rate * getView(v).height));
  }
  viewportClass splitViewV(int v, String label, int h) {
    // エラーチェック
    if (isValidViewNumber(v) == false) {
      crowbar.halt(); // 不正なビュー番号
      return null;
    }
    // 分割
    viewportClass vp = getView(v);
    int xU, yU, wU, hU;  // 上（元）
    int xL, yL, wL, hL;  // 下（新規）
    // 変わらないもの
    xU = xL = vp.vx;
    yU = vp.vy;
    wU = wL = vp.width;
    // hによって変わるもの
    if (h > 0) {
      // 上の高さ指定
      hU = h;
      hL = vp.height - hU;      
    } else {
      // 下の高さ指定
      hL = -h;
      hU = vp.height - hL;
    }
    yL = yU + hU;
    // チェック
    if ((hU <= 0) || (hL <= 0)) {
      println("エラー：分割されたビューポートの一方の高さがゼロ以下です／" + label);
      crowbar.halt();
      return null;
    }
    // 上の縮小
    vp.height = hU;
    cv.world(0, 0, vp.width, vp.height);
    // 終わったら下を新規作成
    if (createView(label, xL, yL, wL, hL) == null) {
      println("エラー：新しいビューポートの作成に失敗しました．");
      crowbar.halt();
      return null;
    }
    cv.cloneAttr(vp);  // 属性のコピー
    return cv;
  }
  // (b-2) ビューポートラベル
  viewportClass splitViewV(String target, String label, float rate) {
    viewportClass vp = getView(target);
    if (vp == null) {
      crowbar.halt();
      return null;
    }
    return splitViewV(vp.serialNumber, label, rate);
  }
  viewportClass splitViewV(String target, String label, int h) {
    viewportClass vp = getView(target);
    if (vp == null) {
      crowbar.halt();
      return null;
    }
    return splitViewV(vp.serialNumber, label, h);
  }
  // (b-3) 現在のビューポート
  viewportClass splitViewV(String label, float rate) { return splitViewV(cvNumber, label, rate); }
  viewportClass splitViewV(String label, int h)      { return splitViewV(cvNumber, label, h);    }

  // 全く同一のビューポートを複製する
  // (1) 指定したビューポート番号
  viewportClass cloneView(int v, String label) {
    if (isValidViewNumber(v) == false) {
      crowbar.halt(); // 不正なビュー番号
      return null;
    }
    viewportClass vp = getView(v);
    cv = createView(label, vp.vx, vp.vy, vp.width, vp.height);
    cv.cloneAttr(vp);  // 属性のコピー
    cv.cloneWorld(vp); // ワールド座標系のコピー
    return cv;
  }
  // (2) 指定したビューポートラベル
  viewportClass cloneView(String target, String label) {
    viewportClass vp = getView(target);
    if (vp == null) {
      crowbar.halt();
      return null;
    }
    return cloneView(vp.serialNumber, label);
  }
  // (3) 現在のビューポート
  viewportClass cloneView(String label) { return cloneView(cvNumber, label); }

  // 指定されたビュー番は正しいか？
  boolean isValidViewNumber(int v) {
    if ((v >= viewMax) || (v < 0) || (v > viewStack.length)) {
      println("エラー：指定されたビューポート番号は不正です／" + str(v));
      return false;
    }
    return true;
  }
  
  // ----------------------------------
  // 指定されたビューポートに切り替える
  // ----------------------------------
  // (a) 現在のビューポート番号を返す
  int view() { return cvNumber; }
  // (b) ビューポートを選択（成功した場合はtrue）
  // (b-1) 指定されたビューポート番号のビューポート
  viewportClass view(int n) {
    if (isValidViewNumber(n) == false) {
      crowbar.halt();
      return null;
    }
    cvNumber = n;
    cv = viewStack[cvNumber];
    return cv;
  }
  // (b-2) 指定されたラベルのビューポート
  viewportClass view(String label) {
    viewportClass vp = getView(label);
    if (vp == null) return null;
    cvNumber = vp.serialNumber;
    cv = vp;
    return cv;
  }
  // (c) 相対的/絶対的な移動
  // (c-1) 一つ次のビューポートへ（行き過ぎた場合は-1を返す）
  viewportClass nextView() {
    if (++cvNumber >= viewMax) return null;
    return view(cvNumber);
  }
  // (c-2) 一つ前のビューポートへ（戻り過ぎた場合は-1を返す）
  viewportClass prevView() {
    if (--cvNumber < 0) return null;
    return view(cvNumber);
  }
  // (c-3) 最初に確保されたビューポートへ
  viewportClass firstView() { return view(0); }
  // (c-4) 最後に確保されたビューポートへ
  viewportClass lastView()  { return view(viewMax - 1); }

  // ------------------------------------
  // 指定されたビューポートのクラスを返す
  // ------------------------------------
  // (a) 指定された番号のビューポートクラスを返す
  viewportClass _getView(int v, boolean error) {
    if (isValidViewNumber(v) == false) {
      if (error) println("エラー：指定されたビューポート番号は不正です／" + str(v));
      return null;
    }
    return viewStack[v];
  }
  viewportClass getView(int v)   { return _getView(v, true);  }  // エラー表示する
  viewportClass checkView(int v) { return _getView(v, false); }  // エラー表示しない（無いことを確認する場合用）
  // (b) 指定されたラベルのビューポートクラスを返す（大文字小文字は区別しない）
  viewportClass _getView(String label, boolean error) {
    for (int v = 0; v < viewMax; v++) if (viewStack[v].label.toUpperCase().equals(label.toUpperCase()) == true) return viewStack[v];
    if (error) println("エラー：指定されたビューポートラベルは不正です／" + label);
    return null;
  }
  viewportClass getView(String label)   { return _getView(label, true);  }  // エラー表示する
  viewportClass checkView(String label) { return _getView(label, false); }  // エラー表示しない（無いことを確認する場合用）
  // (c) 現在のビューポートクラスを返す（不要だけれど一応）
  viewportClass getView() {
    if (cv == null) println("エラー：まだビューポートは一つも作成されていないのに参照しようとしました（getView()）");
    return cv;
  }
  viewportClass checkView() { return cv; }
  // ----------------------------------------------
  // 指定されたビューポートのビューポート番号を返す
  // ----------------------------------------------
  // (a) 指定されたビューポートラベルのビューポート番号を返す（見付からない場合は -1）
  int  _getViewNumber(String label, boolean error) {
    viewportClass vp = _getView(label, error);
    if (vp == null) return -1;
    return vp.serialNumber;
  }
  int getViewNumber(String label)   { return _getViewNumber(label, true);  }  // エラー表示する
  int checkViewNumber(String label) { return _getViewNumber(label, false); }  // エラー表示しない（無いことを確認する場合用）
  // (b) 現在のビューポート番号を返す（不要だけれど一応）
  int getViewNumber(boolean error) {
    if (cv == null) println("エラー：まだビューポートは一つも作成されていないのに参照しようとしました（getView()）");
    return cvNumber;
  }
  int checkViewNumber() { return cvNumber; }
 
  // ----------------------------
  // ビューポートの上下関係の移動
  // ----------------------------
  // ビューポートの表示順を表わす行列を返す
  int [] getViewOrderList() {
    int [] order = new int[viewMax];
    int i;
    for (i = 0; i < viewMax; i++) order[viewStack[i].layerLevel] = i;
    return order;
  }
  // ビューポートの表示順リストを実際のビューポートのデータに書き戻す
  void updateViewOrder(int [] order) {
    int i;
    for (i = 0; i < viewMax; i++) viewStack[order[i]].layerLevel = i;
  }
  // --------------
  // 指定したビューポートの表示順を一番上に移す
  viewportClass viewMoveToTop(int v) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    viewportClass vp = view(v);
    if (viewMax <= 1) return vp;  // 不要なので
    int i;
    int [] order = getViewOrderList();
    for (i = viewStack[v].layerLevel; i < viewMax - 1; i++) order[i] = order[i + 1];
    order[i] = v;
    updateViewOrder(order);
    return vp;
  }
  viewportClass viewMoveToTop(String label) { return viewMoveToTop(getViewNumber(label)); }
  viewportClass viewMoveToTop()             { return viewMoveToTop(cvNumber); }
  // 指定したビューポートの表示順を一番下に移す
  viewportClass viewMoveToBottom(int v) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    viewportClass vp = view(v);
    if (viewMax <= 1) return vp;  // 不要なので
    int i;
    int [] order = getViewOrderList();
    for (i = viewStack[v].layerLevel; i > 0; i--) order[i] = order[i - 1];
    order[0] = v;
    updateViewOrder(order);
    return vp;
  }
  viewportClass viewMoveToBottom(String label) { return viewMoveToBottom(getViewNumber(label)); }
  viewportClass viewMoveToBottom()             { return viewMoveToBottom(cvNumber);            }
  // 指定したビューポートを指定したビューポートの上に移す．
  // 二つのビューポートを指定
  viewportClass viewMoveToAbove(int vt, int vr) {
    if ((isValidViewNumber(vt) == false) || (isValidViewNumber(vr) == false)) return null;  // 不正なビュー番号
    viewportClass vp = view(vt);
    if (viewMax <= 1) return vp;  // ビューポートが一つの場合は無効
    int it, ir;
    it = viewStack[vt].layerLevel;
    ir = viewStack[vr].layerLevel;
    if (it == viewMax - 1) return vp; // 既に一番上
    return viewMoveToRel(vt, ir - it + 1);
  }
  viewportClass viewMoveToAbove(String label1, String label2) { return viewMoveToAbove(getViewNumber(label1), getViewNumber(label2)); }
  // 一つのビューポートを指定（移動先のみ）
  viewportClass viewMoveToAbove(int vr)       { return viewMoveToAbove(cvNumber, vr); }
  viewportClass viewMoveToAbove(String label) { return viewMoveToAbove(cvNumber, getViewNumber(label)); }
  // 指定したビューポートを指定したビューポートの下に移す．
  // 二つのビューポートを指定
  viewportClass viewMoveToBelow(int vt, int vr) {
    if ((isValidViewNumber(vt) == false) || (isValidViewNumber(vr) == false)) return null;  // 不正なビュー番号
    viewportClass vp = view(vt);
    if (viewMax <= 1) return vp; // ビューポートが一つの場合は無効
    int it, ir;
    it = viewStack[vt].layerLevel;
    ir = viewStack[vr].layerLevel;
    if (it == 0) return vp;   // 既に一番下
    return viewMoveToRel(vt, ir - it - 1);
  }
  viewportClass viewMoveToBelow(String label1, String label2) { return viewMoveToBelow(getViewNumber(label1), getViewNumber(label2)); }
  // 一つのビューポートを指定（移動先のみ）
  viewportClass viewMoveToBelow(int vr)       { return viewMoveToBelow(cvNumber, vr); }
  viewportClass viewMoveToBelow(String label) { return viewMoveToBelow(cvNumber, getViewNumber(label)); }
  // 指定されたビューポートを指定された数だけ上下に移す
  viewportClass viewMoveToRel(int v, int num) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    viewportClass vp = view(v);
    if (viewMax <= 1) return vp;   // ビューポートが一つの場合は無効
    if (num == 0) return vp;  // 移動する必要なし
    if ((v + num) <= 0) {
      viewMoveToTop(v);      // 一番上に
    } else if ((v + num) >= viewMax - 1) {
      viewMoveToBottom(v);   // 一番下に
    } else {
      int i, it;
      int [] order = getViewOrderList();
      it = viewStack[v].layerLevel;
      if (num < 0) {
        // 下に
        for (i = it; i > it + num; i--) order[i] = order[i - 1];
      } else {
        // 上に
        for (i = it; i < it + num; i++) order[i] = order[i + 1];
      }
      order[i] = v;
      updateViewOrder(order);
    }
    return vp;
  }
  viewportClass viewMoveToRel(String label, int num) { return viewMoveToRel(getViewNumber(label), num); }
  viewportClass viewMoveToRel(int num)               { return viewMoveToRel(cvNumber, num);  }
  // 指定されたビューポートを指定された数だけ上に移す
  viewportClass viewMoveToUpper(int v, int num)        { return viewMoveToRel(v, num);     }
  viewportClass viewMoveToUpper(String label, int num) { return viewMoveToRel(label, num); }
  viewportClass viewMoveToUpper(int num)               { return viewMoveToRel(num);        }
  // 指定されたビューポートを指定された数だけ下に移す
  viewportClass viewMoveToLower(int v, int num)        { return viewMoveToRel(v, -num);     }
  viewportClass viewMoveToLower(String label, int num) { return viewMoveToRel(label, -num); }
  viewportClass viewMoveToLower(int num)               { return viewMoveToRel(-num);        }
  // 指定されたビューポートを一つ上に移す
  viewportClass viewMoveToUp(int v)         { return viewMoveToUpper(v, 1);     }
  viewportClass viewMoveToUp(String label)  { return viewMoveToUpper(label, 1); }
  viewportClass viewMoveToUp()              { return viewMoveToUpper(1);        }
  // 指定されたビューポートを一つ下に移す
  viewportClass viewMoveToDown(int v)         { return viewMoveToLower(v, 1);     }
  viewportClass viewMoveToDown(String label)  { return viewMoveToLower(label, 1); }
  viewportClass viewMoveToDown()              { return viewMoveToLower(1);        }
  // ビューポートの並び順を初期状態（作成順）に戻す
  void viewLayerReset() {
    int i;
    if (viewMax <= 1) return;                   // ビューポートが一つの場合は無効
    int [] order = getViewOrderList();
    for (i = 0; i < viewMax; i++) order[i] = i;
    updateViewOrder(order);
  }
  // ビューポートの並び順を記録する
  void viewLayerPush() {
    if (viewMax <= 1) return;                   // ビューポートが一つの場合は無効
    layerStack = getViewOrderList();
  }
  // ビューポートの並び順を復元する
  boolean viewLayerPop() {
    if (viewMax <= 1) return true;              // ビューポートが一つの場合は無効
    if (layerStack == null) {
      println("エラー：ビューポートの並び順の記録前にviewLayerPop()が呼ばれました．");
      return false;
    }
    int [] order = getViewOrderList();
    if (layerStack.length != viewMax) {
      println("エラー：ビューポートの並び順を復元しようとしましたがスタックのサイズが異なります．");
      return false;
    }
    for (int i = 0; i < viewMax; i++) order[i] = layerStack[i];
    updateViewOrder(order);
    return true;
  }

  // ----------------------------------
  // ビューポートの枠線や不透明度の設定
  // ----------------------------------
  // (a) フレームカラー
  // (a-1) フレームカラーの取得
  int getFrameColor(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).frameColor();
  }
  int getFrameColor(String label) { return getFrameColor(getViewNumber(label)); }
  int getFrameColor()             { return getFrameColor(cvNumber);       }
  // (a-2) フレームカラーの設定（＋変更前の値）
  int frameColor(int v, int c) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    int oldcolor = getFrameColor(v);
    getView(v).frameColor(c);
    return oldcolor;
  }
  int frameColor(String label, int c) { return frameColor(getViewNumber(label), c); }
  int frameColor(int c)               { return frameColor(cvNumber, c);             }
  viewportClass setFrameColor(int v, int c) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setFrameColor(c);
  }
  viewportClass setFrameColor(String label, int c) { return setFrameColor(getViewNumber(label), c); }
  viewportClass setFrameColor(int c)               { return setFrameColor(cvNumber, c);             }
  // (b) フレームの表示／非表示
  // (b-1) 状態取得
  boolean isDrawFrame(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).drawFrame();
  }
  boolean isDrawFrame(String label) { return isDrawFrame(getViewNumber(label)); }
  boolean isDrawFrame()             { return isDrawFrame(cvNumber);             }
  // (b-2) 設定変更（＋変更前の値）
  boolean drawFrame(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).drawFrame(flag);
  }
  boolean drawFrame(String label, boolean flag) { return drawFrame(getViewNumber(label), flag); }
  boolean drawFrame(boolean flag)               { return drawFrame(cvNumber, flag);             }
  viewportClass setDrawFrame(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setDrawFrame(flag);
  }
  viewportClass setDrawFrame(String label, boolean flag) { return setDrawFrame(getViewNumber(label), flag); }
  viewportClass setDrawFrame(boolean flag)               { return setDrawFrame(cvNumber, flag);             }
  // (c) フレーム幅
  // (c-1) フレーム幅の取得
  int getFrameWidth(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).frameWidth();
  }
  int getFrameWidth(String label) { return getFrameWidth(getViewNumber(label)); }
  int getFrameWidth()             { return getFrameWidth(cvNumber);             }
  // (c-2) 設定変更（＋変更前の値）
  int frameWidth(int v, int w) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).frameWidth(w);
  }
  int frameWidth(String label, int w) { return frameWidth(getViewNumber(label), w); }
  int frameWidth(int w)               { return frameWidth(cvNumber, w);             }
  viewportClass setFrameWidth(int v, int w) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setFrameWidth(w);
  }
  viewportClass setFrameWidth(String label, int w) { return setFrameWidth(getViewNumber(label), w); }
  viewportClass setFrameWidth(int w)               { return setFrameWidth(cvNumber, w);             }
  // (d) ビューポートの背景色関係
  // (d-1) 背景色の取得
  int getViewBgColor(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).viewBgColor();
  }
  int getViewBgColor(String label) { return getViewBgColor(getViewNumber(label)); }
  int getViewBgColor()             { return getViewBgColor(cvNumber);             }
  // (d-2) 背景色の変更（＋変更前の値）
  int viewBgColor(int v, int c) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).viewBgColor(c);
  }
  int viewBgColor(String label, int c) { return viewBgColor(getViewNumber(label), c); }
  int viewBgColor(int c)               { return viewBgColor(cvNumber, c);             }
  viewportClass setViewBgColor(int v, int c) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewBgColor(c);
  }
  viewportClass setViewBgColor(String label, int c) { return setViewBgColor(getViewNumber(label), c); }
  viewportClass setViewBgColor(int c)               { return setViewBgColor(cvNumber, c);             }
  // (e) ビューポートの背景を透明にするか？（trueで透明）
  // (e-1) 状態を取得
  boolean isTransparentView(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).transparentView;
  }
  boolean isTransparentView(String label) { return isTransparentView(getViewNumber(label)); }
  boolean isTransparentView()             { return isTransparentView(cvNumber);            }
  // (e-2) 設定変更（＋変更前の値）
  boolean transparentView(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).transparentView(flag);    
  }
  boolean transparentView(String label, boolean flag) { return transparentView(getViewNumber(label), flag); }
  boolean transparentView(boolean flag)               { return transparentView(cvNumber, flag);             }
  viewportClass setTransparentView(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setTransparentView(flag);    
  }
  viewportClass setTransparentView(String label, boolean flag) { return setTransparentView(getViewNumber(label), flag); }
  viewportClass setTransparentView(boolean flag)               { return setTransparentView(cvNumber, flag);             }
  // (f) ビューポート自体の不透明度の変更
  // (f-1) 状態を取得
  int getTransparency(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).transparency();
  }
  int getTransparency(String label) { return getTransparency(getViewNumber(label)); }
  int getTransparency()             { return getTransparency(cvNumber);             }
  // (f-2) 不透明度の設定（＋変更前の値）
  int transparency(int v, int trans) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v).transparency(trans);
  }
  int transparency(String label, int trans) { return transparency(getViewNumber(label), trans); }
  int transparency(int trans)               { return transparency(cvNumber, trans);             }
  viewportClass setTransparency(int v, int trans) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setTransparency(trans);
  }
  viewportClass setTransparency(String label, int trans) { return setTransparency(getViewNumber(label), trans); }
  viewportClass setTransparency(int trans)               { return setTransparency(cvNumber, trans);             }

  // ------------------------
  // ビューポートの位置の変更
  // ------------------------
  // (a) 指定した位置に移動
  viewportClass moveView(int v, int x, int y) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).moveView(x, y);
  }
  viewportClass moveView(String label, int x, int y) { return moveView(getViewNumber(label), x, y); }
  viewportClass moveView(int x, int y)               { return moveView(cvNumber, x, y); }
  // (b) 相対位置指令
  viewportClass moveViewRel(int v, int dx, int dy) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).moveViewRel(dx, dy);
  }
  viewportClass moveViewRel(String label, int dx, int dy) { return moveViewRel(getViewNumber(label), dx, dy); }
  viewportClass moveViewRel(int dx, int dy)               { return moveViewRel(cvNumber, dx, dy); }

  // 座標変換（現在のビュー）
  // 座標変換（ワールド→ビュー：大きさのみ）
  private int _px(float x) { return cv._px(x); }
  private int _py(float y) { return cv._py(y); }
  // 座標変換（ワールド→ビュー）
  private int px(float x) { return cv.px(x); }
  private int py(float y) { return cv.py(y); }

  // -----------------------------------
  // 描画コマンド関係（ユーザによる利用）
  // -----------------------------------
  // 全てのビューポートを画面消去する（現在のビューは変更しない）
  // flagがtrueの場合は背景が透明（transparentView == true)なビューポートは消去しない
  void clrViewAll(boolean flag) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) clrView(i, flag);
    if (i > 0) view(cv);
  }
  void clrViewAll() { clrViewAll(false); }
  // 指定されたビューポートを画面消去する（現在のビューポートをそのビューポートに移す）
  viewportClass clrView(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    if (view(v) == null) return null;
    cv.clrView(flag);
    return cv;
  }
  viewportClass clrView(String label, boolean flag) { return getView(label).clrView(flag); }
  viewportClass clrView(boolean flag)               { return cv.clrView(flag); }
  viewportClass clrView(int v)        { return clrView(v, false); }
  viewportClass clrView(String label) { return clrView(label, false); }
  viewportClass clrView()             { return clrView(false); }

  // ----------------------------------------------
  // ピボット（拡大縮小および回転の中心座標）の設定
  // ----------------------------------------------
  // (a) 現在のピボット座標を返す
  PVector viewPivot(int v) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).viewPivot();
  }
  PVector viewPivot(String label) { return viewPivot(getViewNumber(label)); }
  PVector viewPivot()             { return viewPivot(cvNumber); }
  // (b) ピボット座標の設定（＋変更前の値）
  PVector viewPivot(int v, float x, float y) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).viewPivot(x, y);
  }
  PVector viewPivot(String label, float x, float y) { return viewPivot(getViewNumber(label), x, y); }
  PVector viewPivot(float x, float y)               { return viewPivot(cvNumber, x, y); }
  viewportClass setViewPivot(int v, float x, float y) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewPivot(x, y);
  }
  viewportClass setViewPivot(String label, float x, float y) { return setViewPivot(getViewNumber(label), x, y); }
  viewportClass setViewPivot(float x, float y)               { return setViewPivot(cvNumber, x, y); }
  // 以下八方位（0:中央, 1:左上, 2:左下, 3:右上, 4: 右下, 5:中央上, 6:中央下, 7:左中央, 8:右中央
  PVector _viewPivot(int v, int pos) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    switch (pos) {
      case 0: return getView(v).viewPivotCenter();
      case 1: return getView(v).viewPivotLT();
      case 2: return getView(v).viewPivotLB();
      case 3: return getView(v).viewPivotRT();
      case 4: return getView(v).viewPivotRB();
      case 5: return getView(v).viewPivotCT();
      case 6: return getView(v).viewPivotCB();
      case 7: return getView(v).viewPivotLC();
      case 8: return getView(v).viewPivotRC();
    }
    return null;  // 不正なポジション
  }
  viewportClass _setViewPivot(int v, int pos) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    switch (pos) {
      case 0: return getView(v).setViewPivotCenter();
      case 1: return getView(v).setViewPivotLT();
      case 2: return getView(v).setViewPivotLB();
      case 3: return getView(v).setViewPivotRT();
      case 4: return getView(v).setViewPivotRB();
      case 5: return getView(v).setViewPivotCT();
      case 6: return getView(v).setViewPivotCB();
      case 7: return getView(v).setViewPivotLC();
      case 8: return getView(v).setViewPivotRC();
    }
    return null;  // 不正なポジション
  }
  // (b-2) ビューポート中心
  PVector viewPivotCenter(int v)          { return _viewPivot(v, 0); }
  PVector viewPivotCenter(String label)   { return viewPivotCenter(getViewNumber(label)); }
  PVector viewPivotCenter()               { return viewPivotCenter(cvNumber); }
  viewportClass setViewPivotCenter(int v)        { return _setViewPivot(v, 0); }
  viewportClass setViewPivotCenter(String label) { return setViewPivotCenter(getViewNumber(label)); }
  viewportClass setViewPivotCenter()             { return setViewPivotCenter(cvNumber); }
   // (b-3) 左上，左下，右上，右下
  PVector viewPivotLT(int v)        { return _viewPivot(v, 1); }
  PVector viewPivotLT(String label) { return viewPivotLT(getViewNumber(label)); }
  PVector viewPivotLT()             { return viewPivotLT(cvNumber); }
  PVector viewPivotLB(int v)        { return _viewPivot(v, 2); }
  PVector viewPivotLB(String label) { return viewPivotLB(getViewNumber(label)); }
  PVector viewPivotLB()             { return viewPivotLB(cvNumber); }
  PVector viewPivotRT(int v)        { return _viewPivot(v, 3); }
  PVector viewPivotRT(String label) { return viewPivotRT(getViewNumber(label)); }
  PVector viewPivotRT()             { return viewPivotRT(cvNumber); }
  PVector viewPivotRB(int v)        { return _viewPivot(v, 4); }
  PVector viewPivotRB(String label) { return viewPivotRB(getViewNumber(label)); }
  PVector viewPivotRB()             { return viewPivotRB(cvNumber); }
  PVector viewPivotCT(int v)        { return _viewPivot(v, 5); }
  PVector viewPivotCT(String label) { return viewPivotCT(getViewNumber(label)); }
  PVector viewPivotCT()             { return viewPivotCT(cvNumber); }
  PVector viewPivotCB(int v)        { return _viewPivot(v, 6); }
  PVector viewPivotCB(String label) { return viewPivotCB(getViewNumber(label)); }
  PVector viewPivotCB()             { return viewPivotCB(cvNumber); }
  PVector viewPivotLC(int v)        { return _viewPivot(v, 7); }
  PVector viewPivotLC(String label) { return viewPivotLC(getViewNumber(label)); }
  PVector viewPivotLC()             { return viewPivotLC(cvNumber); }
  PVector viewPivotRC(int v)        { return _viewPivot(v, 8); }
  PVector viewPivotRC(String label) { return viewPivotRC(getViewNumber(label)); }
  PVector viewPivotRC()             { return viewPivotRC(cvNumber); }

  viewportClass setViewPivotLT(int v)        { return _setViewPivot(v, 1); }
  viewportClass setViewPivotLT(String label) { return setViewPivotLT(getViewNumber(label)); }
  viewportClass setViewPivotLT()             { return setViewPivotLT(cvNumber); }
  viewportClass setViewPivotLB(int v)        { return _setViewPivot(v, 2); }
  viewportClass setViewPivotLB(String label) { return setViewPivotLB(getViewNumber(label)); }
  viewportClass setViewPivotLB()             { return setViewPivotLB(cvNumber); }
  viewportClass setViewPivotRT(int v)        { return _setViewPivot(v, 3); }
  viewportClass setViewPivotRT(String label) { return setViewPivotRT(getViewNumber(label)); }
  viewportClass setViewPivotRT()             { return setViewPivotRT(cvNumber); }
  viewportClass setViewPivotRB(int v)        { return _setViewPivot(v, 4); }
  viewportClass setViewPivotRB(String label) { return setViewPivotRB(getViewNumber(label)); }
  viewportClass setViewPivotRB()             { return setViewPivotRB(cvNumber); }
  viewportClass setViewPivotCT(int v)        { return _setViewPivot(v, 5); }
  viewportClass setViewPivotCT(String label) { return setViewPivotCT(getViewNumber(label)); }
  viewportClass setViewPivotCT()             { return setViewPivotCT(cvNumber); }
  viewportClass setViewPivotCB(int v)        { return _setViewPivot(v, 6); }
  viewportClass setViewPivotCB(String label) { return setViewPivotCB(getViewNumber(label)); }
  viewportClass setViewPivotCB()             { return setViewPivotCB(cvNumber); }
  viewportClass setViewPivotLC(int v)        { return _setViewPivot(v, 7); }
  viewportClass setViewPivotLC(String label) { return setViewPivotLC(getViewNumber(label)); }
  viewportClass setViewPivotLC()             { return setViewPivotLC(cvNumber); }
  viewportClass setViewPivotRC(int v)        { return _setViewPivot(v, 8); }
  viewportClass setViewPivotRC(String label) { return setViewPivotRC(getViewNumber(label)); }
  viewportClass setViewPivotRC()             { return setViewPivotRC(cvNumber); }

  // ----------------------------
  // ビューポートの視点の回転角度
  // ----------------------------
  // (a) 現在の角度を返す
  float viewRotate(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).viewRotate();
  }
  float viewRotate(String label) { return viewRotate(getViewNumber(label)); }
  float viewRotate()             { return viewRotate(cvNumber); }
  // (b) 視点の角度を設定する（＋変更前の値を返す）
  float viewRotate(int v, float a) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).viewRotate(a);
  }
  float viewRotate(String label, float a) { return viewRotate(getViewNumber(label), a); }
  float viewRotate(float a)               { return viewRotate(cvNumber, a); }
  viewportClass setViewRotate(int v, float a) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewRotate(a);
  }
  viewportClass setViewRotate(String label, float a) { return setViewRotate(getViewNumber(label), a); }
  viewportClass setViewRotate(float a)               { return setViewRotate(cvNumber, a); }
  // (c) 視点の角度を増減する（＋変更前の値を返す）
  float viewRotateRel(int v, float a) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).viewRotateRel(a);
  }
  float viewRotateRel(String label, float a) { return viewRotateRel(getViewNumber(label), a); }
  float viewRotateRel(float a)               { return viewRotateRel(cvNumber, a);  }
  viewportClass setViewRotateRel(int v, float a) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewRotateRel(a);
  }
  viewportClass setViewRotateRel(String label, float a) { return setViewRotateRel(getViewNumber(label), a); }
  viewportClass setViewRotateRel(float a)               { return setViewRotateRel(cvNumber, a);  }
  // (d) 全てのビューポートの回転角度を変更する
  void viewRotateAll(float a)    { for (int i = 0; i < viewMax; i++) viewRotate(i, a); }
  void viewRotateRelAll(float a) { for (int i = 0; i < viewMax; i++) viewRotateRel(i, a); }

  // --------
  // 倍率関係
  // --------
  // (a) 現在の倍率を返す：不正なビュー番号の場合は1.0を返す
  float viewScale(int v) {
    if (isValidViewNumber(v) == false) return 1.0;  // 不正なビュー番号
    return getView(v).viewScale();
  }
  float viewScale(String label) { return viewScale(getViewNumber(label)); }
  float viewScale()             { return viewScale(cvNumber); }
  // (b) 倍率の値を変更する（＋変更前の値を返す）
  float viewScale(int v, float a) {
    if (isValidViewNumber(v) == false) return 1.0;  // 不正なビュー番号
    return getView(v).viewScale(a);
  }
  float viewScale(String label, float a) { return viewScale(getViewNumber(label), a); }
  float viewScale(float a)               { return viewScale(cvNumber, a); }
  viewportClass setViewScale(int v, float a) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewScale(a);
  }
  viewportClass setViewScale(String label, float a) { return setViewScale(getViewNumber(label), a); }
  viewportClass setViewScale(float a)               { return setViewScale(cvNumber, a); }
  // (c) 全てのビューポートの倍率を指定した倍率に変更する
  void viewScaleAll(float a) { for (int i = 0; i < viewMax; i++) viewScale(i, a); }

  // ----------------
  // オフセット量関係
  // ----------------
  // (a) 指定したビューポートのオフセット量を返す
  // (a-1) ｘ方向
  float viewOffsetX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).viewOffsetX();
  }
  float viewOffsetX(String label) { return viewOffsetX(getViewNumber(label)); }
  float viewOffsetX()             { return viewOffsetX(cvNumber); }
  // (a-2) ｙ方向（上を正としている点に注意）
  float viewOffsetY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).viewOffsetY();
  }
  float viewOffsetY(String label) { return viewOffsetY(getViewNumber(label)); }
  float viewOffsetY()             { return viewOffsetY(cvNumber); }
  // (b) オフセット量を変更する（＋変更前の値を返す）：指定したビューポート
  // (b-1) ｘ方向
  float viewOffsetX(int v, float offset) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    float oldoffset = getView(v).viewOffsetX();
    getView(v).viewOffsetX(offset);
    return oldoffset;
  }
  float viewOffsetX(String label, float offset) { return viewOffsetX(getViewNumber(label), offset); }
  float viewOffsetX(float offset)               { return viewOffsetX(cvNumber, offset); }
  viewportClass setViewOffsetX(int v, float offset) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewOffsetX(offset);
  }
  viewportClass setViewOffsetX(String label, float offset) { return setViewOffsetX(getViewNumber(label), offset); }
  viewportClass setViewOffsetX(float offset)               { return setViewOffsetX(cvNumber, offset); }
  // (b-2) ｙ方向（上を正としている点に注意）
  float viewOffsetY(int v, float offset) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    float oldoffset = getView(v).viewOffsetY();
    getView(v).viewOffsetY(offset);
    return oldoffset;
  }
  float viewOffsetY(String label, float offset) { return viewOffsetY(getViewNumber(label), offset); }
  float viewOffsetY(float offset)               { return viewOffsetY(cvNumber, offset); }
  viewportClass setViewOffsetY(int v, float offset) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).setViewOffsetY(offset);
  }
  viewportClass setViewOffsetY(String label, float offset) { return setViewOffsetY(getViewNumber(label), offset); }
  viewportClass setViewOffsetY(float offset)               { return setViewOffsetY(cvNumber, offset); }
  // (b-3) ｘ方向／ｙ方向同時：成功した場合はtrueを返す
  boolean viewOffset(int v, float offx, float offy) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    float oldoffset;
    getView(v).viewOffsetX(offx);
    getView(v).viewOffsetY(offy);
    return true;
  }
  boolean viewOffset(String label, float offx, float offy) { return viewOffset(getViewNumber(label), offx, offy); }
  boolean viewOffset(float offx, float offy)               { return viewOffset(cvNumber, offx, offy); }
  viewportClass setViewOffset(int v, float offx, float offy) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    getView(v).viewOffsetX(offx);
    return getView(v).setViewOffsetY(offy);
  }
  viewportClass setViewOffset(String label, float offx, float offy) { return setViewOffset(getViewNumber(label), offx, offy); }
  viewportClass setViewOffset(float offx, float offy)               { return setViewOffset(cvNumber, offx, offy); }
  // 全てのビューポートの設定を変更する
  void viewOffsetXAll(float offset) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) viewOffsetX(i, offset);
    view(cv);
  }
  void viewOffsetYAll(float offset) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) viewOffsetY(i, offset);
    view(cv);
  }
  void viewOffsetAll(float offx, float offy) {
    int i, cv;
    cv = cvNumber;
    for (i = 0; i < viewMax; i++) viewOffset(i, offx, offy);
    view(cv);
  }

  // --------------------
  // ワールド座標系の設定
  // --------------------
  // 【参照】
  // (a) 指定したビューポートのワールド座標系の設定値の参照：不正なビューポートが指定された場合は0.0を返す
  // (a-1) ビューポート番号を指定
  float world_minX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wx_min;
  }
  float world_maxX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wx_max;
  }
  float world_minY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wy_min;
  }
  float world_maxY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).wy_max;
  }
  // (a-2) ビューポートラベルを指定
  float world_minX(String label) { return world_minX(getViewNumber(label)); }
  float world_maxX(String label) { return world_maxX(getViewNumber(label)); }
  float world_minY(String label) { return world_minY(getViewNumber(label)); }
  float world_maxY(String label) { return world_maxY(getViewNumber(label)); }
  // (a-3) 現在のビューポートを指定
  float world_minX() { return cv.wx_min; }
  float world_maxX() { return cv.wx_max; }
  float world_minY() { return cv.wy_min; }
  float world_maxY() { return cv.wy_max; }
  // 【変更】
  // (a) 指定したビューポートのワールド座標系の変更：成功した場合はtrueを返す
  // (a-1) ビューポート番号を指定
  viewportClass world(int v, float x1, float y1, float x2, float y2) {
    if (isValidViewNumber(v) == false) {  // 不正なビューポートラベルを指定された場合など，不正なビュー番号
      crowbar.halt();
      return null;
    }
    return getView(v).world(x1, y1, x2, y2);
  }
  // (a-2) ビューポートラベルを指定
  viewportClass world(String label, float x1, float y1, float x2, float y2) { return world(getViewNumber(label), x1, y1, x2, y2); }
  // (a-3) 現在のビューポートを指定
  viewportClass world(float x1, float y1, float x2, float y2) {
    if (cv == null) {
      println("エラー：ビューポートが存在しません．");
      crowbar.halt();
      return null;
    }
    return cv.world(x1, y1, x2, y2);
  }
  // (b) 指定したビューポートのｘ，ｙそれぞれの最小値／最大値を個別に変更：成功した場合はtrueを返す
  // (b-1) ビューポート番号を指定
  viewportClass world_minX(int v, float x) { return world(v, x, world_minY(v), world_maxX(v), world_maxY(v)); }
  viewportClass world_maxX(int v, float x) { return world(v, world_minX(v), world_minY(v), x, world_maxY(v)); }
  viewportClass world_minY(int v, float y) { return world(v, world_minX(v), y, world_maxX(v), world_maxY(v)); }
  viewportClass world_maxY(int v, float y) { return world(v, world_minX(v), world_minY(v), world_maxX(v), y); }
  viewportClass worldX(int v, float xmin, float xmax) { return world(v, xmin, world_minY(v), xmax, world_maxY(v)); }
  viewportClass worldY(int v, float ymin, float ymax) { return world(v, world_minX(v), ymin, world_maxX(v), ymax); }
  // (b-2) ビューポートラベルを指定
  viewportClass world_minX(String label, float x) { return world(label, x, world_minY(label), world_maxX(label), world_maxY(label)); }
  viewportClass world_maxX(String label, float x) { return world(label, world_minX(label), world_minY(label), x, world_maxY(label)); }
  viewportClass world_minY(String label, float y) { return world(label, world_minX(label), y, world_maxX(label), world_maxY(label)); }
  viewportClass world_maxY(String label, float y) { return world(label, world_minX(label), world_minY(label), world_maxX(label), y); }
  viewportClass worldX(String label, float xmin, float xmax) { return world(label, xmin, world_minY(label), xmax, world_maxY(label)); }
  viewportClass worldY(String label, float ymin, float ymax) { return world(label, world_minX(label), ymin, world_maxX(label), ymax); }
  // (b-3) 現在のビューポートを指定
  viewportClass world_minX(float x) { return world(x, world_minY(), world_maxX(), world_maxY()); }
  viewportClass world_maxX(float x) { return world(world_minX(), world_minY(), x, world_maxY()); }
  viewportClass world_minY(float y) { return world(world_minX(), y, world_maxX(), world_maxY()); }
  viewportClass world_maxY(float y) { return world(world_minX(), world_minY(), world_maxX(), y); }
  viewportClass worldX(float xmin, float xmax) { return world(xmin, world_minY(), xmax, world_maxY()); }
  viewportClass worldY(float ymin, float ymax) { return world(world_minX(), ymin, world_maxX(), ymax); }
  
  // (c) ピクセル比のみを指定（引数なし＝ピクセル比は変えない）
  // 以下八方位（0:中央, 1:左上, 2:左下, 3:右上, 4: 右下, 5:中央上, 6:中央下, 7:左中央, 8:右中央
  viewportClass _world(int v, int pos) {
    if (isValidViewNumber(v) == false) {
      crowbar.halt();
      return null;
    }
    switch (pos) {
      case 0: return getView(v).worldOrigin();
      case 1: return getView(v).worldLT();
      case 2: return getView(v).worldLB();
      case 3: return getView(v).worldRT();
      case 4: return getView(v).worldRB();
      case 5: return getView(v).worldCT();
      case 6: return getView(v).worldCB();
      case 7: return getView(v).worldLC();
      case 8: return getView(v).worldRC();
    }
    println("システムエラー：_world()のポジション指示が異常です．");
    return null;
  }
  viewportClass _world(int v, float ratio, int pos) {
    if (isValidViewNumber(v) == false) {
      crowbar.halt();
      return null;
    }
    switch (pos) {
      case 0: return getView(v).worldOrigin(ratio);
      case 1: return getView(v).worldLT(ratio);
      case 2: return getView(v).worldLB(ratio);
      case 3: return getView(v).worldRT(ratio);
      case 4: return getView(v).worldRB(ratio);
      case 5: return getView(v).worldCT(ratio);
      case 6: return getView(v).worldCB(ratio);
      case 7: return getView(v).worldLC(ratio);
      case 8: return getView(v).worldRC(ratio);
    }
    println("システムエラー：_world()のポジション指示が異常です．");
    return null;
  }
  // (c-1) ビューポート中央を原点
  viewportClass worldOrigin(int v)                     { return _world(v, 0); }
  viewportClass worldOrigin(int v, float ratio)        { return _world(v, ratio, 0); }
  viewportClass worldOrigin(String label)              { return worldOrigin(getViewNumber(label)); }
  viewportClass worldOrigin(String label, float ratio) { return worldOrigin(getViewNumber(label), ratio); }
  viewportClass worldOrigin()                          { return worldOrigin(cvNumber); }
  viewportClass worldOrigin(float ratio)               { return worldOrigin(cvNumber, ratio); }
  // (c-2) ビューポート左上を原点
  viewportClass worldLT(int v)                     { return _world(v, 1); }
  viewportClass worldLT(int v, float ratio)        { return _world(v, ratio, 1); }
  viewportClass worldLT(String label)              { return worldLT(getViewNumber(label)); }
  viewportClass worldLT(String label, float ratio) { return worldLT(getViewNumber(label), ratio); }
  viewportClass worldLT()                          { return worldLT(cvNumber); }
  viewportClass worldLT(float ratio)               { return worldLT(cvNumber, ratio); }
  // (c-3) ビューポート左下を原点
  viewportClass worldLB(int v)                     { return _world(v, 2); }
  viewportClass worldLB(int v, float ratio)        { return _world(v, ratio, 2); }
  viewportClass worldLB(String label)              { return worldLB(getViewNumber(label)); }
  viewportClass worldLB(String label, float ratio) { return worldLB(getViewNumber(label), ratio); }
  viewportClass worldLB()                          { return worldLB(cvNumber); }
  viewportClass worldLB(float ratio)               { return worldLB(cvNumber, ratio); }
  // (c-4) ビューポート右上を原点
  viewportClass worldRT(int v)                     { return _world(v, 3); }
  viewportClass worldRT(int v, float ratio)        { return _world(v, ratio, 3); }
  viewportClass worldRT(String label)              { return worldRT(getViewNumber(label)); }
  viewportClass worldRT(String label, float ratio) { return worldRT(getViewNumber(label), ratio); }
  viewportClass worldRT()                          { return worldRT(cvNumber); }
  viewportClass worldRT(float ratio)               { return worldRT(cvNumber, ratio); }
  // (c-5) ビューポート右下を原点
  viewportClass worldRB(int v)                     { return _world(v, 4); }
  viewportClass worldRB(int v, float ratio)        { return _world(v, ratio, 4); }
  viewportClass worldRB(String label)              { return worldRB(getViewNumber(label)); }
  viewportClass worldRB(String label, float ratio) { return worldRB(getViewNumber(label), ratio); }
  viewportClass worldRB()                          { return worldRB(cvNumber); }
  viewportClass worldRB(float ratio)               { return worldRB(cvNumber, ratio); }
  // (c-6) ビューポート中央上を原点
  viewportClass worldCT(int v)                     { return _world(v, 5); }
  viewportClass worldCT(int v, float ratio)        { return _world(v, ratio, 5); }
  viewportClass worldCT(String label)              { return worldCT(getViewNumber(label)); }
  viewportClass worldCT(String label, float ratio) { return worldCT(getViewNumber(label), ratio); }
  viewportClass worldCT()                          { return worldCT(cvNumber); }
  viewportClass worldCT(float ratio)               { return worldCT(cvNumber, ratio); }
  // (c-7) ビューポート中央下を原点
  viewportClass worldCB(int v)                     { return _world(v, 6); }
  viewportClass worldCB(int v, float ratio)        { return _world(v, ratio, 6); }
  viewportClass worldCB(String label)              { return worldCB(getViewNumber(label)); }
  viewportClass worldCB(String label, float ratio) { return worldCB(getViewNumber(label), ratio); }
  viewportClass worldCB()                          { return worldCB(cvNumber); }
  viewportClass worldCB(float ratio)               { return worldCB(cvNumber, ratio); }
  // (c-8) ビューポート左中央を原点
  // ビューポート番号を指定
  viewportClass worldLC(int v)                     { return _world(v, 7); }
  viewportClass worldLC(int v, float ratio)        { return _world(v, ratio, 7); }
  viewportClass worldLC(String label)              { return worldLC(getViewNumber(label)); }
  viewportClass worldLC(String label, float ratio) { return worldLC(getViewNumber(label), ratio); }
  viewportClass worldLC()                          { return worldLC(cvNumber); }
  viewportClass worldLC(float ratio)               { return worldLC(cvNumber, ratio); }
  // (c-9) ビューポート右中央を原点
  viewportClass worldRC(int v)                     { return _world(v, 8); }
  viewportClass worldRC(int v, float ratio)        { return _world(v, ratio, 8); }
  viewportClass worldRC(String label)              { return worldRC(getViewNumber(label)); }
  viewportClass worldRC(String label, float ratio) { return worldRC(getViewNumber(label), ratio); }
  viewportClass worldRC()                          { return worldRC(cvNumber); }
  viewportClass worldRC(float ratio)               { return worldRC(cvNumber, ratio); }
  
  // (d) ワールド座標のオフセット量とピクセル比を指定
  // つまりこの指定した座標がビューポートの中央に表示されるようにマッピングされる
  /// ビューポート番号を指定
  viewportClass worldCenter(int v, float x, float y) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).worldCenter(x, y);
  }
  viewportClass worldCenter(int v, float x, float y, float ratio) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).worldCenter(x, y, ratio);
  }
  // ビューポートラベルを指定
  viewportClass worldCenter(String label, float x, float y)              { return worldCenter(getViewNumber(label), x, y); }
  viewportClass worldCenter(String label, float x, float y, float ratio) { return worldCenter(getViewNumber(label), x, y, ratio); }
  // 現在のビューポートを指定
  viewportClass worldCenter(float x, float y)              { return worldCenter(cvNumber, x, y); }
  viewportClass worldCenter(float x, float y, float ratio) { return worldCenter(cvNumber, x, y, ratio); }

  // (e) ワールド座標をｘ軸，ｙ軸方向に相対的に動かす（ピクセル比は変更しない）
  // (e-1) ｘｙ軸同時に動かす
  // ビューポート番号を指定
  viewportClass worldMove(int v, float dx, float dy) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).worldMove(dx, dy);
  }
  // ビューポートラベルを指定
  viewportClass worldMove(String label, float dx, float dy) { return worldMove(getViewNumber(label), dx, dy); }
  // 現在のビューポートを指定
  viewportClass worldMove(float dx, float dy) { return worldMove(cvNumber, dx, dy); }
  // (e-2) ｘ軸方向に動かす
  // ビューポート番号を指定
  viewportClass worldMoveX(int v, float dx) { return worldMoveX(v, dx); }
  // ビューポートラベルを指定
  viewportClass worldMoveX(String label, float dx) { return worldMoveX(getViewNumber(label), dx); }
  // 現在のビューポートを指定
  viewportClass worldMoveX(float dx) { return worldMoveX(cvNumber, dx); }
  // (e-3) ｙ軸方向に動かす
  // ビューポート番号を指定
  viewportClass worldMoveY(int v, float dy) { return worldMoveY(v, dy); }
  // ビューポートラベルを指定
  viewportClass worldMoveY(String label, float dy) { return worldMoveY(getViewNumber(label), dy); }
  // 現在のビューポートを指定
  viewportClass worldMoveY(float dy) { return worldMoveY(cvNumber, dy); }

  // --------------------------------------------------
  // ビューポートの表示／非表示のダイナミックな切り替え
  // --------------------------------------------------
  // 状態
  boolean isViewVisible(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).isViewVisible();
  }
  boolean isViewVisible(String label) { return isViewVisible(getViewNumber(label)); }
  boolean isViewVisible()             { return isViewVisible(cvNumber); }
  // (a) 表示
  viewportClass viewVisible(int v) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).viewVisible();
  }
  viewportClass viewVisible(String label) { return viewVisible(getViewNumber(label)); }
  viewportClass viewVisible()             { return viewVisible(cvNumber); }
  // (b) 非表示
  viewportClass viewUnvisible(int v) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    return getView(v).viewUnvisible();
  }
  viewportClass viewUnvisible(String label) { return viewUnvisible(getViewNumber(label)); }
  viewportClass viewUnvisible()             { return viewUnvisible(cvNumber);             }
  // (c) 引数による指定
  viewportClass viewVisible(int v, boolean flag) {
    if (isValidViewNumber(v) == false) return null;  // 不正なビュー番号
    if (flag) return viewVisible(v); else return viewUnvisible(v);
  }
  viewportClass viewVisible(String label, boolean flag) { return viewVisible(getViewNumber(label), flag); }
  viewportClass viewVisible(boolean flag)               { return viewVisible(cvNumber, flag); }
  // (d) トグル動作（変更後の状態を返す）
  boolean toggleViewVisible(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).toggleViewVisible();
  }
  boolean toggleViewVisible(String label) { return toggleViewVisible(getViewNumber(label)); }
  boolean toggleViewVisible()             { return toggleViewVisible(cvNumber); }

  // --------------------------------------------------------
  // マウスによるビューポート内クリックに対するレスポンス関係
  // --------------------------------------------------------
  // (a) マウスポインターがこのビューポート上にあるか？（上に別のビューポートが重なっているかどうかは気にしない．クリックされたかどうかも関係ない）
  //     クリックされたかどうかはuserClicked()でイベント駆動にする．
  boolean mouseover(int v) {
    if (isValidViewNumber(v) == false) return false;  // 不正なビュー番号
    return getView(v).mouseover();
  }
  boolean mouseover(String label) { return mouseover(getViewNumber(label)); }
  boolean mouseover()             { return mouseover(cvNumber);            }
  // (b) マウスポインターの座標を返します．領域外であっても返します（負の値を返す場合もあります）．
  // (b-1) ピクセル座標
  int _viewMouseX(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v)._viewMouseX();
  }
  int _viewMouseX(String label) { return _viewMouseX(getViewNumber(label)); }
  int _viewMouseX()             { return _viewMouseX(cvNumber);             }
  int _viewMouseY(int v) {
    if (isValidViewNumber(v) == false) return -1;  // 不正なビュー番号
    return getView(v)._viewMouseY();
  }
  int _viewMouseY(String label) { return _viewMouseY(getViewNumber(label)); }
  int _viewMouseY()             { return _viewMouseY(cvNumber);             }
  // (b-2) ワールド座標
  float viewMouseX(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).rpx(_viewMouseX(v));
  }
  float viewMouseX(String label) { return viewMouseX(getViewNumber(label)); }
  float viewMouseX()             { return viewMouseX(cvNumber);             }
  float viewMouseY(int v) {
    if (isValidViewNumber(v) == false) return 0.0;  // 不正なビュー番号
    return getView(v).rpy(_viewMouseY(v)); 
  }
  float viewMouseY(String label) { return viewMouseY(getViewNumber(label)); }
  float viewMouseY()             { return viewMouseY(cvNumber);             }
  // 与えられた座標（マウスクリックなど）はどのビューポートを指しているのか？
  // （どのビューポートも含まれないならば-1）
  // trans == true の場合は背景が透明なビューポートを無視する
  int detectActiveView()             { return detectActiveView(mouseX, mouseY, true); }
  int detectActiveView(int x, int y) { return detectActiveView(x, y, true); }
  int detectActiveView(int x, int y, boolean trans) {
    int [] order = getViewOrderList();
    int i, x1, y1, x2, y2;
    viewportClass vp;
    for (i = viewMax - 1; i >= 0; i--) {
      vp = viewStack[order[i]];
      if (vp.isViewVisible() == false) continue;  // 非表示は無視
      if (vp.transparency == 0)        continue;  // ビューポートの不透明度がゼロも無視
      // ビューポートの背景が透明
      if (vp.transparentView == true) {
        if (trans == true) continue;
        int col = vp.pg.get(x - vp.vx, y - vp.vy);
        if ((col & 0x0ff000000) == 0) continue;  // もし描画されていないならば無視
      }
      x1 = vp.vx;
      y1 = vp.vy;
      x2 = x1 + vp.width;
      y2 = y1 + vp.height;
      if ((x1 <= x) && (x <= x2) && (y1 <= y) && (y <= y2)) return order[i];
    }
    return -1;
  }

// =========================================================================================================

  // -----------------------------------------------
  // ビューの合成（crowbarClass.pdeのdraw()にて使用）
  // -----------------------------------------------
  void refreshView() {
    if (tomahawkMode == 0) return;
//  if (tomahawkMode == 2) background(_BGCOLOR);
    if (crowbar.status == 10) {
      // ビューの合成
      int i;
      int [] order = getViewOrderList();
      // 表示順のリスト作成（きちんと欠番，重複なく並んでいることは上下関係の操作時に確認済みという前提）
      viewportClass vp;
      for (i = 0; i < viewMax; i++) {
        vp = viewStack[order[i]];
        if (vp.isViewVisible() == false) continue;
        if (vp.drawFrame) {
          vp.pg.endDraw();
          vp.pg.beginDraw();
          vp.pg.noFill();
          vp.pg.stroke(vp.frameColor, 255);
          vp.pg.strokeWeight(vp.frameWidth);
          vp.pg.rectMode(CORNER);
          vp.pg.rect(vp.frameWidth / 2, vp.frameWidth / 2, vp.width - vp.frameWidth, vp.height - vp.frameWidth);
//        vp.pg.rect(0, 0, vp.width - 1, vp.height - 1);
          vp.pg.stroke(vp.cdSet.strokeColor, vp.cdSet.strokeTrans);
          vp.pg.strokeWeight(vp.cdSet.strokeWeight);
        }
        vp.pg.endDraw();
        blend(vp.pg, 0, 0, vp.width, vp.height, vp.vx, vp.vy, vp.width, vp.height, BLEND);
      }
    }
    if (tomahawkMode == 2) blend(crowbar.screenSetting.pgText, 0, 0, crowbar.screenSetting.gwx, crowbar.screenSetting.gwy, 0, 0, crowbar.screenSetting.gwx, crowbar.screenSetting.gwy, BLEND);
  }
}

