/*
 * Cueplot: a GUI front end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "Process.h"
#include "ConsoleDialog.h"

#include <QCoreApplication>
#include <QFileDialog>
#include <QMessageBox>
#include <QProcess>
#include <QSettings>
#include <QTextCodec>

Process::Process(QObject *parent)
: QObject(parent), process(new QProcess)
{
	if (!startProcess()) {
		QMessageBox::critical(0, QCoreApplication::applicationName(),
				tr("Could not start gnuplot\n") +
				tr("Application will now exit"));
		exit(1);
	}
	connect(process, SIGNAL(readyReadStandardError()),
			this, SLOT(readStandardError()));

	consoleDialog = new ConsoleDialog;
}

bool Process::startProcess()
{
	QSettings settings;
	settings.beginGroup(metaObject()->className());
	QString path = settings.value("path").toString();
	settings.endGroup();

	// 1st attempt
	if (!path.isEmpty() && canStart(path)) {
		return true;
	}

	path = 
#ifdef Q_OS_WIN32
		"pgnuplot.exe"
#else
		"gnuplot"
#endif
		;

	// 2nd attempt
	if (canStart(path)) {
		return true;
	}

	path = QFileDialog::getOpenFileName(0, 
			tr("Open gnuplot") + " - "
			+ QCoreApplication::applicationName(), QString(),
			"gnuplot (" + path + ")");

	// 3rd attempt
	if (path.isNull()) {
		return false;
	}
	return canStart(path);
}

bool Process::canStart(const QString &path)
{
	process->start(path, QStringList());
	if (process->waitForStarted()) {
		QSettings settings;
		settings.beginGroup(metaObject()->className());
		settings.setValue("path", path);
		settings.endGroup();
		return true;
	} 
	return false;
}

void Process::execute(const QString &command)
{
	process->write(QTextCodec::codecForLocale()
			->fromUnicode(command).append('\n'));
	consoleDialog->execute(command);
}

void Process::readStandardError()
{
	consoleDialog->show();
	consoleDialog->readStandardError(process->readAllStandardError());
}

void Process::closeDialog()
{
	consoleDialog->close();
}

void Process::setDialogVisible(bool visible)
{
	consoleDialog->setVisible(visible);
}

bool Process::isDialogVisible()
{
	return consoleDialog->isVisible();
}
