/*
 * Cueplot: a GUI front end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "KeyOptionWidget.h"
#include "Key.h"

#include <QCheckBox>
#include <QDoubleSpinBox>
#include <QGridLayout>
#include <QGroupBox>
#include <QLabel>
#include <QLineEdit>
#include <QRadioButton>
#include <QSpinBox>
#include <QTabWidget>
#include <QVBoxLayout>

#include <cmath> // pow()

KeyOptionWidget::KeyOptionWidget(QWidget *parent)
: OptionWidget(parent)
{
	setWindowTitle(tr("Legend"));
}

void KeyOptionWidget::construct()
{
	keyCheckBox = new QCheckBox;
	keyCheckBox->setText(tr("Ac&tivation"));

	cornerRadioButton = new QRadioButton;
	cornerRadioButton->setText(tr("Co&rner"));

	const QString POSITION[] 
		= { tr("&left"), tr("ri&ght"), tr("&top"),
			tr("botto&m"), tr("o&utside"), tr("b&elow") };
	for (int i = 0; i < POSITION_NUM; i++) {
		positionButton[i] = new QCheckBox;
		positionButton[i]->setText(POSITION[i]);
		connect(cornerRadioButton, SIGNAL(toggled(bool)),
				positionButton[i], SLOT(setEnabled(bool)));
		connect(positionButton[i], SIGNAL(toggled(bool)),
				this, SLOT(validatePosition()));
	}

	coordinateRadioButton = new QRadioButton;
	coordinateRadioButton->setText(tr("Coord&inate"));

	const int LIMIT = 32767;
	const QString AXIS[] = { "&X", "&Y", "&Z" };
	for (int i = 0; i < AXIS_NUM; i++) {
		positionDoubleSpinBox[i] = new QDoubleSpinBox;
		positionDoubleSpinBox[i]->setMaximum(LIMIT);
		positionDoubleSpinBox[i]->setMinimum(-LIMIT);
		connect(coordinateRadioButton, SIGNAL(toggled(bool)),
				positionDoubleSpinBox[i], SLOT(setEnabled(bool)));

		positionLabel[i] = new QLabel;
		positionLabel[i]->setText(AXIS[i]);
		positionLabel[i]->setBuddy(positionDoubleSpinBox[i]);
		connect(coordinateRadioButton, SIGNAL(toggled(bool)),
				positionLabel[i], SLOT(setEnabled(bool)));
	}

	textJustifyCheckBox = new QCheckBox;
	textJustifyCheckBox->setText(tr("Left Justify"));

	reverseCheckBox = new QCheckBox;
	reverseCheckBox->setText(tr("Reverse"));

	autoTitlesCheckBox = new QCheckBox;
	autoTitlesCheckBox->setText(tr("Auto Titles"));

	enhancedCheckBox = new QCheckBox;
	enhancedCheckBox->setText(tr("Enhanced"));

	QLabel *sampleLengthLabel = new QLabel;
	sampleLengthLabel->setText(tr("Sample Length:"));

	sampleLengthDoubleSpinBox = new QDoubleSpinBox;

	QLabel *spacingLabel = new QLabel;
	spacingLabel->setText(tr("Spacing:"));

	spacingDoubleSpinBox = new QDoubleSpinBox;

	QLabel *widthIncrementLabel = new QLabel;
	widthIncrementLabel->setText(tr("Width Increment:"));

	widthIncrementDoubleSpinBox = new QDoubleSpinBox;
	widthIncrementDoubleSpinBox->setMinimum(-99);

	QLabel *heightIncrementLabel = new QLabel;
	heightIncrementLabel->setText(tr("Height Increment:"));

	heightIncrementDoubleSpinBox = new QDoubleSpinBox;
	heightIncrementDoubleSpinBox->setMinimum(-99);

	QLabel *legendTitleLabel = new QLabel;
	legendTitleLabel->setText(tr("Legend Title:"));

	legendTitleLineEdit = new QLineEdit;

	lineTypeSpinBox = new QSpinBox;

	lineTypeLabel = new QLabel;
	lineTypeLabel->setText(tr("&Line Type:"));
	lineTypeLabel->setBuddy(lineTypeSpinBox);

	lineWidthDoubleSpinBox = new QDoubleSpinBox;

	lineWidthLabel = new QLabel;
	lineWidthLabel->setText(tr("L&ine Width:"));
	lineWidthLabel->setBuddy(lineWidthDoubleSpinBox);

	QGridLayout *cornerLayout = new QGridLayout;
	for (int i = 0; i < POSITION_NUM; i++) {
		cornerLayout->addWidget(positionButton[i], i / 3, i % 3);
	}

	QGridLayout *coordinateLayout = new QGridLayout;
	for (int i = 0; i < AXIS_NUM; i++) {
		coordinateLayout->addWidget(positionLabel[i], i, 0);
		coordinateLayout->addWidget(positionDoubleSpinBox[i], i, 1);
	}
	coordinateLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);

	QGridLayout *positionLayout = new QGridLayout;
	positionLayout->addWidget(cornerRadioButton, 0, 0, 1, 4);
	positionLayout->addItem(new QSpacerItem(20, 20,
				QSizePolicy::Fixed, QSizePolicy::Minimum), 1, 0);
	positionLayout->addLayout(cornerLayout, 1, 1);
	positionLayout->addWidget(coordinateRadioButton, 2, 0, 1, 4);
	positionLayout->addItem(new QSpacerItem(20, 20,
				QSizePolicy::Fixed, QSizePolicy::Minimum), 3, 0);
	positionLayout->addLayout(coordinateLayout, 3, 1);
	positionLayout->addItem(new QSpacerItem(20, 40, 
				QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0);

	QWidget *positionWidget = new QWidget;
	positionWidget->setLayout(positionLayout);

	QGridLayout *boxLayout = new QGridLayout;
	boxLayout->addWidget(lineTypeLabel, 0, 0);
	boxLayout->addWidget(lineTypeSpinBox, 0, 1);
	boxLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	boxLayout->addWidget(lineWidthLabel, 1, 0);
	boxLayout->addWidget(lineWidthDoubleSpinBox, 1, 1);

	boxGroupBox = new QGroupBox(tr("Bo&x"));
	boxGroupBox->setCheckable(true);
	boxGroupBox->setLayout(boxLayout);
	connect(boxGroupBox, SIGNAL(toggled(bool)),
			lineTypeLabel, SLOT(setEnabled(bool)));
	connect(boxGroupBox, SIGNAL(toggled(bool)),
			lineTypeSpinBox, SLOT(setEnabled(bool)));
	connect(boxGroupBox, SIGNAL(toggled(bool)),
			lineWidthLabel, SLOT(setEnabled(bool)));
	connect(boxGroupBox, SIGNAL(toggled(bool)),
			lineWidthDoubleSpinBox, SLOT(setEnabled(bool)));

	QVBoxLayout *boxTabLayout = new QVBoxLayout;
	boxTabLayout->addWidget(boxGroupBox);
	boxTabLayout->addStretch();

	QWidget *boxWidget = new QWidget;
	boxWidget->setLayout(boxTabLayout);

	QGridLayout *othersLayout = new QGridLayout;
	othersLayout->addWidget(textJustifyCheckBox, 0, 0);
	othersLayout->addWidget(reverseCheckBox, 0, 1);
	othersLayout->addWidget(autoTitlesCheckBox, 1, 0);
	othersLayout->addWidget(enhancedCheckBox, 1, 1);
	othersLayout->addWidget(sampleLengthLabel, 2, 0);
	othersLayout->addWidget(sampleLengthDoubleSpinBox, 2, 1);
	othersLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 2, 2);
	othersLayout->addWidget(spacingLabel, 3, 0);
	othersLayout->addWidget(spacingDoubleSpinBox, 3, 1);
	othersLayout->addWidget(widthIncrementLabel, 4, 0);
	othersLayout->addWidget(widthIncrementDoubleSpinBox, 4, 1);
	othersLayout->addWidget(heightIncrementLabel, 5, 0);
	othersLayout->addWidget(heightIncrementDoubleSpinBox, 5, 1);
	othersLayout->addWidget(legendTitleLabel, 6, 0);
	othersLayout->addWidget(legendTitleLineEdit, 6, 1, 1, 2);
	othersLayout->addItem(new QSpacerItem(20, 40,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 7, 0);

	QWidget *othersWidget = new QWidget;
	othersWidget->setLayout(othersLayout);

	QTabWidget *tabWidget = new QTabWidget;
	tabWidget->addTab(positionWidget, tr("&Position"));
	tabWidget->addTab(boxWidget, tr("&Box"));
	tabWidget->addTab(othersWidget, tr("Ot&hers"));
	connect(keyCheckBox, SIGNAL(toggled(bool)), 
			tabWidget, SLOT(setEnabled(bool)));

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(keyCheckBox);
	mainLayout->addWidget(tabWidget);

	setLayout(mainLayout);
}

void KeyOptionWidget::init()
{
	Key key;
	keyCheckBox->setChecked(key.isOn());

	int pos = key.position();
	cornerRadioButton->setChecked(pos);
	for (int i = 0; i < POSITION_NUM; i++) {
		positionButton[i]->setChecked(pos & static_cast<int>(pow(2, i)));
		if (!pos) {
			positionButton[i]->setEnabled(false);
		}
	}
	coordinateRadioButton->setChecked(!pos);
	for (int i = 0; i < AXIS_NUM; i++) {
		positionDoubleSpinBox[i]->setValue(key.position(i));
		if (pos) {
			positionLabel[i]->setEnabled(false);
			positionDoubleSpinBox[i]->setEnabled(false);
		}
	}

	textJustifyCheckBox->setChecked(key.leftJustification());
	reverseCheckBox->setChecked(key.isReverse());
	sampleLengthDoubleSpinBox->setValue(key.sampleLength());
	spacingDoubleSpinBox->setValue(key.verticalSpacing());
	widthIncrementDoubleSpinBox->setValue(key.widthIncrement());
	heightIncrementDoubleSpinBox->setValue(key.heightIncrement());
	autoTitlesCheckBox->setChecked(key.isAutoTitles());
	legendTitleLineEdit->setText(key.title());
	enhancedCheckBox->setChecked(key.enhances());

	boxGroupBox->setChecked(key.isBox());
	if (key.isBox()) {
		lineTypeSpinBox->setValue(key.lineType());
		lineWidthDoubleSpinBox->setValue(key.lineWidth());
	} else {
		lineTypeSpinBox->setEnabled(false);
		lineTypeLabel->setEnabled(false);
		lineWidthDoubleSpinBox->setEnabled(false);
		lineWidthLabel->setEnabled(false);
	}
}

void KeyOptionWidget::apply()
{
	Key key;

	key.setOn(keyCheckBox->isChecked());

	if (cornerRadioButton->isChecked()) {
		int pos = 0;
		for (int i = 0; i < POSITION_NUM; i++) {
			if (positionButton[i]->isChecked()) {
				pos |= static_cast<int>(pow(2, i));
			}
		}
		key.setPosition(pos);
	} else {
		for (int i = 0; i < AXIS_NUM; i++) {
			key.setPosition(i, positionDoubleSpinBox[i]->value());
		}
	}

	key.setLeftJustification(textJustifyCheckBox->isChecked());
	key.setReverse(reverseCheckBox->isChecked());
	key.setSampleLength(sampleLengthDoubleSpinBox->value());
	key.setVerticalSpacing(spacingDoubleSpinBox->value());
	key.setWidthIncrement(widthIncrementDoubleSpinBox->value());
	key.setHeightIncrement(heightIncrementDoubleSpinBox->value());
	key.setAutoTitles(autoTitlesCheckBox->isChecked());
	key.setTitle(legendTitleLineEdit->text());
	key.setEnhanced(enhancedCheckBox->isChecked());
	key.setBox(boxGroupBox->isChecked());
	if (boxGroupBox->isChecked()) {
		key.setLineType(lineTypeSpinBox->value());
		key.setLineWidth(lineWidthDoubleSpinBox->value());
	}

	emit executed(key.command());
}

void KeyOptionWidget::validatePosition()
{
	for (int i = 0; i < POSITION_NUM; i++) {
		if (positionButton[i]->isChecked()) {
			return;
		}
	}
	coordinateRadioButton->setChecked(true);
}
