/*
 * Cueplot: a GUI front end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "TerminalOptionWidget.h"
#include "Terminal.h"
#include "LatexEmtexOptionWidget.h"
#include "PbmOptionWidget.h"
#include "PostScriptOptionWidget.h"

#include <QComboBox>
#include <QLabel>
#include <QHBoxLayout>
#include <QPushButton>
#include <QStackedWidget>
#include <QVBoxLayout>

const QString TerminalOptionWidget::TERMINAL[] = {
	"emtex", "latex", "pbm", "postscript", ""
};

TerminalOptionWidget::TerminalOptionWidget(QWidget *parent)
: OptionWidget(parent)
{
	setWindowTitle(tr("Terminal"));
}

void TerminalOptionWidget::construct()
{
	typeComboBox = new QComboBox;
#ifdef Q_OS_MACX
	typeComboBox->addItem("aqua");
#endif
	typeComboBox->addItem("corel");
	typeComboBox->addItem("eepic");
	typeComboBox->addItem("emtex");
	typeComboBox->addItem("fig");
	typeComboBox->addItem("latex");
	typeComboBox->addItem("pbm");
	typeComboBox->addItem("postscript");
	typeComboBox->addItem("pslatex");
	typeComboBox->addItem("pstricks");
	typeComboBox->addItem("table");
	typeComboBox->addItem("texdraw");
	typeComboBox->addItem("tgif");
	typeComboBox->addItem("tpic");
	typeComboBox->addItem("unknown");
#if defined(Q_OS_WIN32)
	typeComboBox->addItem("windows");
#elif defined(Q_OS_UNIX)
	typeComboBox->addItem("x11");
#endif

	QLabel *typeLabel = new QLabel;
	typeLabel->setText(tr("&Terminal Type:"));
	typeLabel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	typeLabel->setBuddy(typeComboBox);

	typeStackedWidget = new QStackedWidget;
	typeStackedWidget->addWidget(new LatexEmtexOptionWidget);
	typeStackedWidget->addWidget(new LatexEmtexOptionWidget);
	typeStackedWidget->addWidget(new PbmOptionWidget);
	typeStackedWidget->addWidget(new PostScriptOptionWidget);
	connect(typeComboBox, SIGNAL(activated(int)),
			this, SLOT(changeTerminal(int)));

	initBitArray.resize(typeStackedWidget->count());

	initPushButton = new QPushButton;
	initPushButton->setText(tr("&Initialize Terminal Options"));
	connect(initPushButton, SIGNAL(clicked()),
			this, SLOT(initOption()));

	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(initPushButton);

	QHBoxLayout *listLayout = new QHBoxLayout;
	listLayout->addWidget(typeLabel);
	listLayout->addWidget(typeComboBox);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout(listLayout);
	mainLayout->addWidget(typeStackedWidget);
	mainLayout->addStretch();
	mainLayout->addLayout(buttonLayout);

	setLayout(mainLayout);
}

void TerminalOptionWidget::init()
{
	Terminal terminal;
	QString type = terminal.terminalType();
	int index = typeComboBox->findText(type);
	bool flag = false;
	if (index >= 0) {
		typeComboBox->setCurrentIndex(index);
		changeTerminal(index);
		flag = hasOption(type);
	}
	typeStackedWidget->setEnabled(flag);
	initPushButton->setEnabled(flag);
}

void TerminalOptionWidget::initOption()
{
	qobject_cast<OptionWidget *>(typeStackedWidget->currentWidget())
		->init();
}

void TerminalOptionWidget::apply()
{
	Terminal terminal;
	terminal.setTerminalType(typeComboBox->currentText());

	if (hasOption(terminal.terminalType())) {
		qobject_cast<OptionWidget *>(typeStackedWidget->currentWidget())
			->apply();
	} else {
		emit executed(terminal.command());
	}
}

bool TerminalOptionWidget::hasOption(const QString &type)
{
	bool flag = false;
	for (int i = 0; !TERMINAL[i].isEmpty(); i++) {
		if (type == TERMINAL[i]) {
			flag = true;
			break;
		}
	}
	return flag;
}

void TerminalOptionWidget::changeTerminal(int index)
{
	QString type = typeComboBox->itemText(index);
	for (int i = 0; !TERMINAL[i].isEmpty(); i++) {
		if (type == TERMINAL[i]) {
			typeStackedWidget->setEnabled(true);
			initPushButton->setEnabled(true);
			if (initBitArray.at(i)) {
				return;
			}
			OptionWidget *optionWidget =
				qobject_cast<OptionWidget *>(typeStackedWidget->widget(i));
			optionWidget->construct();
			optionWidget->init();
			connect(optionWidget, SIGNAL(executed(const QString &)),
					this, SIGNAL(executed(const QString &)));
			initBitArray.setBit(i);
			typeStackedWidget->setCurrentIndex(i);
			return;
		}
	}
	typeStackedWidget->setEnabled(false);
	initPushButton->setEnabled(false);
}
