/*
 * Cueplot: a GUI front end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "TicsOptionWidget.h"
#include "Tics.h"

#include <QCheckBox>
#include <QComboBox>
#include <QLabel>
#include <QLineEdit>
#include <QRadioButton>
#include <QStackedWidget>
#include <QVBoxLayout>

TicsOptionWidget::TicsOptionWidget(QWidget *parent)
: OptionWidget(parent)
{
	setWindowTitle(tr("Tics"));
}

void TicsOptionWidget::construct()
{
	axisComboBox = new QComboBox;
	for (int i = 0; i < TICS_NUM; i++) {
		axisComboBox->addItem(Tics::AXIS[i]);
	}

	QLabel *axisLabel = new QLabel;
	axisLabel->setText(tr("A&xis:"));
	axisLabel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	axisLabel->setBuddy(axisComboBox);

	ticsStackedWidget = new QStackedWidget;

	QWidget *widget[TICS_NUM];
	for (int i = 0; i < TICS_NUM; i++) {
		widget[i] = new QWidget;
		setWidget(*widget[i], i);
		ticsStackedWidget->addWidget(widget[i]);
	}
	connect(axisComboBox, SIGNAL(activated(int)),
			ticsStackedWidget, SLOT(setCurrentIndex(int)));

	QHBoxLayout *selectLayout = new QHBoxLayout;
	selectLayout->addWidget(axisLabel);
	selectLayout->addWidget(axisComboBox);

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addLayout(selectLayout);
	mainLayout->addWidget(ticsStackedWidget);
	setLayout(mainLayout);
}

void TicsOptionWidget::init()
{
	Tics tics;
	for (int i = 0; i < TICS_NUM; i++) {
		onCheckBox[i]->setChecked(tics.flag(i));

		ticsTabWidget[i]->setEnabled(tics.flag(i));

		mirrorCheckBox[i]->setChecked(tics.mirrors(i));

		locationComboBox[i]->
			setCurrentIndex(!tics.isBorder(i));

		rotationCheckBox[i]->setChecked(tics.rotates(i));

		if (tics.isAutofreq(i)) {
			autofreqRadioButton[i]->setChecked(true);

			startPosLineEdit[i]->setEnabled(false);
			incPosLineEdit[i]->setEnabled(false);
			endPosLineEdit[i]->setEnabled(false);

			labelsPosLineEdit[i]->setEnabled(false);
		} else if (tics.isIncrement(i)) {
			sieRadioButton[i]->setChecked(true);

			startPosLineEdit[i]->setText(tics.start(i));
			incPosLineEdit[i]->setText(tics.increment(i));
			endPosLineEdit[i]->setText(tics.end(i));

			labelsPosLineEdit[i]->setEnabled(false);
		} else if (tics.isPosition(i)) {
			labelPosRadioButton[i]->setChecked(true);

			startPosLineEdit[i]->setEnabled(false);
			incPosLineEdit[i]->setEnabled(false);
			endPosLineEdit[i]->setEnabled(false);

			labelsPosLineEdit[i]->setText(tics.position(i).join(","));
		}
	}
}

void TicsOptionWidget::apply()
{
	Tics tics;
	// get and set options
	for (int i = 0; i < TICS_NUM; i++) {
		tics.setFlag(i, onCheckBox[i]->isChecked());
		tics.setMirror(i, mirrorCheckBox[i]->isChecked());
		tics.setBorder(i, !locationComboBox[i]->currentIndex());
		tics.setRotate(i, rotationCheckBox[i]->isChecked());

		if (autofreqRadioButton[i]->isChecked()) {
			tics.setAutofreq(i);
		} else if (sieRadioButton[i]->isChecked()) {
			tics.setIncrement(i, incPosLineEdit[i]->text(),
					startPosLineEdit[i]->text(),
					endPosLineEdit[i]->text());
		} else if (labelPosRadioButton[i]->isChecked()) {
			tics.setPosition(i, labelsPosLineEdit[i]->text().split(","));
		}
	}
	emit executed(tics.command());
}

void TicsOptionWidget::setWidget(QWidget &widget, int index)
{
	onCheckBox[index] = new QCheckBox;
	onCheckBox[index]->setText(tr("Ac&tivation"));

	locationComboBox[index] = new QComboBox;
	locationComboBox[index]->addItem(tr("border"));
	locationComboBox[index]->addItem(tr("axis"));

	mirrorCheckBox[index] = new QCheckBox;
	mirrorCheckBox[index]->setText(tr("&Mirroring"));

	rotationCheckBox[index] = new QCheckBox;
	rotationCheckBox[index]->setText(tr("&Rotation"));

	autofreqRadioButton[index] = new QRadioButton;
	autofreqRadioButton[index]->setText(tr("A&uto"));

	sieRadioButton[index] = new QRadioButton;
	sieRadioButton[index]->setText(tr("&Start/Inc/End"));

	startPosLineEdit[index] = new QLineEdit;
	connect(sieRadioButton[index], SIGNAL(toggled(bool)),
			startPosLineEdit[index], SLOT(setEnabled(bool)));

	QLabel *startPosLabel = new QLabel;
	startPosLabel->setText(tr("S&tart:"));
	startPosLabel->setBuddy(startPosLineEdit[index]);

	incPosLineEdit[index] = new QLineEdit;
	connect(sieRadioButton[index], SIGNAL(toggled(bool)),
			incPosLineEdit[index], SLOT(setEnabled(bool)));

	QLabel *incPosLabel = new QLabel;
	incPosLabel->setText(tr("&Increment:"));
	incPosLabel->setBuddy(incPosLineEdit[index]);

	endPosLineEdit[index] = new QLineEdit;
	connect(sieRadioButton[index], SIGNAL(toggled(bool)),
			endPosLineEdit[index], SLOT(setEnabled(bool)));

	QLabel *endPosLabel = new QLabel;
	endPosLabel->setText(tr("&End:"));
	endPosLabel->setBuddy(endPosLineEdit[index]);

	labelPosRadioButton[index] = new QRadioButton;
	labelPosRadioButton[index]->setText(tr("&Labels/Positions"));

	labelsPosLineEdit[index] = new QLineEdit;
	connect(labelPosRadioButton[index], SIGNAL(toggled(bool)),
			labelsPosLineEdit[index], SLOT(setEnabled(bool)));

	// create and assemble layouts

	QGridLayout *sieLayout = new QGridLayout;
	sieLayout->addWidget(startPosLabel, 0, 0);
	sieLayout->addWidget(startPosLineEdit[index], 0, 1);
	sieLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);
	sieLayout->addWidget(incPosLabel, 1, 0);
	sieLayout->addWidget(incPosLineEdit[index], 1, 1);
	sieLayout->addWidget(endPosLabel, 2, 0);
	sieLayout->addWidget(endPosLineEdit[index], 2, 1);

	QGridLayout *posLayout = new QGridLayout;
	posLayout->addWidget(autofreqRadioButton[index], 0, 0, 1, 2);
	posLayout->addWidget(sieRadioButton[index], 1, 0, 1, 2);
	posLayout->addItem(new QSpacerItem(20, 20, 
				QSizePolicy::Fixed, QSizePolicy::Minimum), 2, 0);
	posLayout->addLayout(sieLayout, 2, 1);
	posLayout->addWidget(labelPosRadioButton[index], 3, 0, 1, 2);
	posLayout->addItem(new QSpacerItem(20, 20,
				QSizePolicy::Fixed, QSizePolicy::Minimum), 4, 0);
	posLayout->addWidget(labelsPosLineEdit[index], 4, 1);

	QWidget *posWidget = new QWidget;
	posWidget->setLayout(posLayout);

	QGridLayout *othersLayout = new QGridLayout;
	othersLayout->addWidget(locationComboBox[index], 0, 0);
	othersLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 1);
	othersLayout->addWidget(mirrorCheckBox[index], 1, 0);
	othersLayout->addWidget(rotationCheckBox[index], 2, 0);
	othersLayout->addItem(new QSpacerItem(20, 40,
				QSizePolicy::Minimum, QSizePolicy::Expanding), 3, 0);

	QWidget *othersWidget = new QWidget;
	othersWidget->setLayout(othersLayout);

	ticsTabWidget[index] = new QTabWidget;
	ticsTabWidget[index]->addTab(posWidget, tr("&Position"));
	ticsTabWidget[index]->addTab(othersWidget, tr("O&thers"));
	connect(onCheckBox[index], SIGNAL(toggled(bool)),
			ticsTabWidget[index], SLOT(setEnabled(bool)));

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(onCheckBox[index]);
	mainLayout->addWidget(ticsTabWidget[index]);
	mainLayout->addStretch();

	widget.setLayout(mainLayout);
}
