/*
 * Cueplot: a GUI front end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "Key.h"

#include <cmath>

Key::Key() : isOn_(true), position_(0x02 | 0x04), 
	leftJustification_(false), isReverse_(false), 
	sampleLength_(4.0), verticalSpacing_(1.25),
	widthIncrement_(0.0), heightIncrement_(0.0), 
	isAutoTitles_(true), title_(""), enhances_(true), 
	lineType_(DEFAULT_LINE_TYPE), lineWidth_(1.0)
{
	for (int i = 0; i < AXIS_NUM; i++) {
		positionArray[i] = 0.0;
	}
}

void Key::setOn(bool on)
{
	isOn_ = on;
}

bool Key::isOn() const
{
	return isOn_;
}

void Key::setPosition(int pos)
{
	for (int i = 0; i < AXIS_NUM; i++) {
		positionArray[i] = 0.0;
	}
	position_ = pos;
}

int Key::position() const
{
	return position_;
}

void Key::setPosition(int axis, double pos)
{
	position_ = 0;
	positionArray[axis] = pos;
}

double Key::position(int axis) const
{
	return positionArray[axis];
}

void Key::setLeftJustification(bool leftJustify)
{
	leftJustification_ = leftJustify;
}

bool Key::leftJustification() const
{
	return leftJustification_;
}

void Key::setReverse(bool reverse)
{
	isReverse_ = reverse;
}

bool Key::isReverse() const
{
	return isReverse_;
}

void Key::setSampleLength(double length)
{
	sampleLength_ = length;
}

double Key::sampleLength() const
{
	return sampleLength_;
}

void Key::setVerticalSpacing(double spacing)
{
	verticalSpacing_ = spacing;
}

double Key::verticalSpacing() const
{
	return verticalSpacing_;
}

void Key::setWidthIncrement(double incr)
{
	widthIncrement_ = incr;
}

double Key::widthIncrement() const
{
	return widthIncrement_;
}

void Key::setHeightIncrement(double incr)
{
	heightIncrement_ = incr;
}

double Key::heightIncrement() const
{
	return heightIncrement_;
}

void Key::setAutoTitles(bool autoTitles)
{
	isAutoTitles_ = autoTitles;
}

bool Key::isAutoTitles() const
{
	return isAutoTitles_;
}

void Key::setTitle(const QString &text)
{
	title_ = text;
}

const QString &Key::title() const
{
	return title_;
}

void Key::setEnhanced(bool enhanced)
{
	enhances_ = enhanced;
}

bool Key::enhances() const
{
	return enhances_;
}

void Key::setLineType(int lt)
{
	lineType_ = lt;
}

void Key::setBox(bool box)
{
	lineType_ = box ? -1 : DEFAULT_LINE_TYPE;
}

bool Key::isBox() const
{
	return lineType_ != DEFAULT_LINE_TYPE; 
}

int Key::lineType() const
{
	return lineType_;
}

void Key::setLineWidth(double lw)
{
	lineWidth_ = lw;
}

double Key::lineWidth() const
{
	return lineWidth_;
}

QString Key::command() const
{
	QString str = "set key title";
	str += (title_.isEmpty()) ? "" : " \"" + title_ + "\"";
	str += "\n";
	if (!isOn_) {
		str += "unset key";
		return str;
	}
	str += "set key ";
	if (position_) {
		static const QString POSITION[] = 
		{ "left", "right", "top", "bottom", "outside", "below" };
		for (int i = 0; i < POSITION_NUM; i++) {
			if (position_ & static_cast<int>(pow(2, i))) {
				str += POSITION[i] + " ";
			}
		}
	} else {
		for (int i = 0; i < AXIS_NUM; i++) {
			str += QString::number(positionArray[i]);
			if (i != AXIS_NUM - 1) {
				str += ",";
			}
			str += " ";
		}
	}
	str += leftJustification_ ? "Left " : "Right ";

	str += isReverse_ ? "" : "no";
	str += "reverse ";

	str += enhances_ ? "" : "no";
	str += "enhanced ";

	str += "box linetype " + QString::number(lineType_)
		+ " linewidth " + QString::number(lineWidth_) + " ";

	str += "samplen " + QString::number(sampleLength_) + " "
		+ "spacing " + QString::number(verticalSpacing_) + " "
		+ "width " + QString::number(widthIncrement_) + " "
		+ "height " + QString::number(heightIncrement_) + " ";

	str += isAutoTitles_ ? "" : "no";
	str += "autotitles";

	return str;
}
