/*
 * Cueplot: a GUI front end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "Tics.h"

const QString Tics::AXIS[] = {
	"x", "y", "z", "x2", "y2"
};

Tics::Tics()
{
	for (int i = 0, j = AXIS_NUM; i < j; i++) {
		flagArray[i] = (i < 3);
		borderArray[i] = true;
		mirrorArray[i] = (i < 3);
		rotateArray[i] = false;
		angArray[i] = QString();
		modeArray[i] = AUTOFREQ;
		for (int j = 0; j < 3; j++) {
			incrArray[j][i] = "";
		}
		posList[i] = QStringList();
	}
}

void Tics::setFlag(int axis, bool flag)
{
	flagArray[axis] = flag;
}

bool Tics::flag(int axis) const
{
	return flagArray[axis];
}

void Tics::setMirror(int axis, bool mirror)
{
	mirrorArray[axis] = mirror;
}

bool Tics::mirrors(int axis) const
{
	return mirrorArray[axis];
}

void Tics::setBorder(int axis, bool border)
{
	borderArray[axis] = border;
}

bool Tics::isBorder(int axis) const
{
	return borderArray[axis];
}

void Tics::setRotate(int axis, bool rotate, const QString &ang)
{
	rotateArray[axis] = rotate;
	if (rotate) {
		angArray[axis] = ang;
	}
}

bool Tics::rotates(int axis) const
{
	return rotateArray[axis];
}

void Tics::setAutofreq(int axis)
{
	modeArray[axis] = AUTOFREQ;
}

bool Tics::isAutofreq(int axis) const
{
	return modeArray[axis] == AUTOFREQ;
}

void Tics::setIncrement(int axis, const QString &incr, 
		const QString &start, const QString &end)
{
	modeArray[axis] = INCR;
	incrArray[axis][0] = start;
	incrArray[axis][1] = incr;
	incrArray[axis][2] = end;
}

bool Tics::isIncrement(int axis) const
{
	return modeArray[axis] == INCR;
}

const QString &Tics::start(int axis) const
{
	return incrArray[axis][0];
}

const QString &Tics::increment(int axis) const
{
	return incrArray[axis][1];
}

const QString &Tics::end(int axis) const
{
	return incrArray[axis][2];
}

void Tics::setPosition(int axis, const QStringList &pos)
{
	modeArray[axis] = POS;
	posList[axis] = pos;
}

bool Tics::isPosition(int axis) const
{
	return modeArray[axis] == POS;
}

const QStringList &Tics::position(int axis) const
{
	return posList[axis];
}

QString Tics::command() const
{
	QStringList commandList;
	for (int i = 0, j = AXIS_NUM; i < j; i++) {
		QString str = "set ";
		// tics | notics
		if (!flagArray[i]) {
			str += "no" + AXIS[i] + "tics";
			commandList << str;
			continue;
		}
		str += AXIS[i] + "tics ";
		// axis | border
		str += borderArray[i] ? "border " : "axis "; 
		// mirror | nomirror
		str += mirrorArray[i] ? "mirror " : "nomirror "; 
		// rotate
		if (rotateArray[i]) {
			str += "rotate ";
			if (!angArray[i].isEmpty()) {
				// rotate by <ang>
				str += "by " + angArray[i] + " ";
			}
		} else {
			// norotate
			str += "norotate ";
		}
		if (modeArray[i] == AUTOFREQ) {
			// autofreq
			str += "autofreq ";
		} else if (modeArray[i] == INCR) {
			// <start>, <incr>, <end>
			if (!incrArray[i][0].isEmpty()) {
				str += incrArray[i][0] + ", ";
			}
			str += incrArray[i][1];
			if (!incrArray[i][2].isEmpty()) {
				str += ", " + incrArray[i][2];
			}
			str += " ";
		} else if (modeArray[i] == POS) {
			// "<label>" <pos> <level>...
			str += "(" + posList[i].join(",") + ")";
		}
		commandList << str;
	}
	return commandList.join("\n");
}
