/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "PlotDataWidget.h"
#include "Plot.h"
#include "PlotListWidget.h"
#include "PlotOptionDialog.h"

#include <QtGui/QListWidget>
#include <QtGui/QGridLayout>
#include <QtGui/QPushButton>

PlotDataWidget::PlotDataWidget(QWidget *parent)
: QWidget(parent)
{
	addPushButton = new QPushButton;
	addPushButton->setText(tr("&Add"));
	addPushButton->setEnabled(false);
	connect(addPushButton, SIGNAL(clicked()),
			this, SIGNAL(dataAdded()));

	optionPushButton = new QPushButton;
	optionPushButton->setText(tr("O&ption..."));
	optionPushButton->setEnabled(false);
	optionPushButton->setSizePolicy(
			QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	connect(optionPushButton, SIGNAL(clicked()),
			this, SLOT(setPlotOptions()));

	upPushButton = new QPushButton;
	upPushButton->setText(tr("&Up"));
	upPushButton->setEnabled(false);
	connect(upPushButton, SIGNAL(clicked()),
			this, SLOT(setDataUp()));

	downPushButton = new QPushButton;
	downPushButton->setText(tr("&Down"));
	downPushButton->setEnabled(false);
	connect(downPushButton, SIGNAL(clicked()),
			this, SLOT(setDataDown()));

	deletePushButton = new QPushButton;
	deletePushButton->setText(tr("D&elete"));
	deletePushButton->setEnabled(false);
	deletePushButton->setSizePolicy(
			QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	connect(deletePushButton, SIGNAL(clicked()),
			this, SLOT(deleteData()));

	plotListWidget = new PlotListWidget;
	connect(plotListWidget, SIGNAL(itemSelectionChanged()),
			this, SLOT(setPushButton()));
	connect(plotListWidget, SIGNAL(itemChanged(QListWidgetItem *)),
			this, SLOT(changeItem(QListWidgetItem *)));

	// ----------- assemble layouts and widgets
	// ------------------------ create layouts

	QGridLayout *mainLayout = new QGridLayout;
	mainLayout->setMargin(0);
	mainLayout->addWidget(addPushButton, 0, 0);
	mainLayout->addWidget(optionPushButton, 1, 0);
	mainLayout->addWidget(upPushButton, 2, 0);
	mainLayout->addWidget(downPushButton, 3, 0);
	// ウィンドウを最大化したときのことを考えて、QSpacerItem を入れる
	mainLayout->addItem(new QSpacerItem(20, 0, 
				QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0);
	mainLayout->addItem(new QSpacerItem(40, 20, 
				QSizePolicy::Expanding, QSizePolicy::Minimum), 5, 0, 1, 2);
	mainLayout->addWidget(deletePushButton, 5, 2);
	mainLayout->addWidget(plotListWidget, 0, 1, 5, 2);

	setLayout(mainLayout);
}

PlotDataWidget::~PlotDataWidget()
{
	while (!plotList.isEmpty()) {
		delete plotList.takeFirst();
	}
}

void PlotDataWidget::addData(Plot &plot)
{
	QString data = plot.isFunction() ? plot.function() : plot.dataFile();
	if (plotListWidget->currentRow() < 0) {
		// リスト中に項目がないとき
		plotListWidget->addItem(data);
		plotListWidget->setCurrentRow(plotListWidget->count() - 1);

		plotList.append(&plot);
		dialogList.append(0);
	} else {
		// 項目があるとき、選択されている項目の次に追加
		int row = plotListWidget->currentRow() + 1;
		plotListWidget->insertItem(row, data);
		plotListWidget->setCurrentRow(row);

		plotList.insert(row, &plot);
		dialogList.insert(row, 0);
	}
	QListWidgetItem *item = plotListWidget->currentItem();
	item->setFlags(item->flags() | Qt::ItemIsEditable 
			| Qt::ItemIsUserCheckable);
	item->setCheckState(Qt::Checked);
	emit setPlotEnabled(true);
	setPushButton();
}

void PlotDataWidget::setDataUp()
{
	int row = plotListWidget->currentRow();
	// 一番上にある項目はこれ以上、上に動かせない
	if (row > 0) {
		// 削除した項目を上に挿入
		QListWidgetItem *item = plotListWidget->takeItem(row);
		plotListWidget->insertItem(row - 1, item);
		plotListWidget->setCurrentRow(row - 1);

		Plot *plot = plotList.takeAt(row);
		plotList.insert(row - 1, plot);
		PlotOptionDialog *plotOptionDialog = dialogList.takeAt(row);
		dialogList.insert(row - 1, plotOptionDialog);
	}
}

void PlotDataWidget::setDataDown()
{
	int row = plotListWidget->currentRow();
	// 一番下にある項目はこれ以上、下に動かせない
	if (row < plotListWidget->count() - 1) {
		// 削除した項目を下に挿入
		QListWidgetItem *item = plotListWidget->takeItem(row);
		plotListWidget->insertItem(row + 1, item);
		plotListWidget->setCurrentRow(row + 1);

		Plot *plot = plotList.takeAt(row);
		plotList.insert(row + 1, plot);
		PlotOptionDialog *plotOptionDialog = dialogList.takeAt(row);
		dialogList.insert(row + 1, plotOptionDialog);
	}
}

void PlotDataWidget::deleteData()
{
	int row = plotListWidget->currentRow();
	delete plotListWidget->takeItem(row);
	delete plotList.takeAt(row);
	delete dialogList.takeAt(row);
	setAction();
	setPushButton();
}

void PlotDataWidget::setPushButton()
{
	// 項目が一つでも選択されていたら、
	// オプションボタンと削除ボタンを有効にする
	bool isSelected = (plotListWidget->currentRow() >= 0);
	optionPushButton->setEnabled(isSelected);
	deletePushButton->setEnabled(isSelected);

	// 選択された項目があり、複数の項目がリスト中にあれば、
	// 上ボタンと下ボタンを有効にする
	int count = plotListWidget->count();
	int selectedRow = -1;
	for (int i = 0; i < count; i++) {
#if QT_VERSION >= 0x040200
		if (plotListWidget->item(i)->isSelected()) {
#else
		// obsolete
		if (plotListWidget->isItemSelected(plotListWidget->item(i))) {
#endif
			selectedRow = i;
			break;
		}
	}
	bool isMulti = isSelected && (count > 1);
	// 選択された項目が一番上にあるときは上ボタンを無効にする
	upPushButton->setEnabled(isMulti && !(selectedRow == 0));
	// 選択された項目が一番下にあるときは下ボタンを無効にする
	downPushButton->setEnabled(isMulti && !(selectedRow == count - 1));
}

void PlotDataWidget::setAdditionEnabled(bool enable)
{
	addPushButton->setEnabled(enable);
}

void PlotDataWidget::setPlotOptions()
{
	int row = plotListWidget->currentRow();
	PlotOptionDialog *plotOptionDialog = dialogList.at(row);
	if (!plotOptionDialog) {
		plotOptionDialog = new PlotOptionDialog;
		plotOptionDialog->setPlot(*plotList[row]);
		dialogList[row] = plotOptionDialog;
	}
	plotOptionDialog->show();
	plotOptionDialog->raise();
	plotOptionDialog->activateWindow();
}

void PlotDataWidget::changeItem(QListWidgetItem *item)
{
	int row = plotListWidget->currentRow();
	plotListWidget->setCurrentItem(item);
	Plot *plot = plotList[row];
	QString text = item->text();
	if (plot->isFunction()) {
		plot->setFunction(text);
	} else {
		plot->setDataFile(text);
	}
	PlotOptionDialog *plotOptionDialog = dialogList.at(row);
	if (plotOptionDialog) {
		plotOptionDialog->setPlot(*plot);
	}
	setAction();
}

void PlotDataWidget::setAction()
{
	// チェックされている項目があるか調べる
	bool isChecked = false;
	for (int i = 0, j = plotListWidget->count(); i < j; i++) {
		if (plotListWidget->item(i)->checkState() == Qt::Checked) {
			isChecked = true;
			break;
		}
	}
	emit setPlotEnabled(isChecked);
}

bool PlotDataWidget::hasItem() const
{
	return plotListWidget->count() > 0;
}

QStringList PlotDataWidget::commandList() const
{
	QStringList commandList;
	for (int i = 0, j = plotList.size(); i < j; i++) {
		// チェックされている項目のみプロットする
		if (plotListWidget->item(i)->checkState() == Qt::Checked) {
			commandList << plotList.at(i)->command();
		}
	}
	return commandList;
}

void PlotDataWidget::closeDialog()
{
	while (!dialogList.isEmpty()) {
		delete dialogList.takeFirst();
	}
}
