/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "PlotOptionDialog.h"
#include "PlotModifierWidget.h"
#include "PlotOthersWidget.h"
#include "PlotStyleWidget.h"
#include "PlotTitleWidget.h"

#include <QtCore/QCoreApplication>
#include <QtCore/QDir>
#include <QtGui/QCloseEvent>
#if QT_VERSION >= 0x040200
# include <QtGui/QDialogButtonBox>
#endif
#include <QtGui/QListWidget>
#include <QtGui/QPushButton>
#include <QtGui/QSplitter>
#include <QtGui/QStackedWidget>
#include <QtGui/QVBoxLayout>

PlotOptionDialog::PlotOptionDialog(QWidget *parent)
: QDialog(parent)
{
	plotOthersWidget = new PlotOthersWidget;
	plotTitleWidget = new PlotTitleWidget;
	plotStyleWidget = new PlotStyleWidget;
	plotModifierWidget = new PlotModifierWidget;

	pagesStackedWidget = new QStackedWidget;
	pagesStackedWidget->addWidget(plotModifierWidget);
	pagesStackedWidget->addWidget(plotTitleWidget);
	pagesStackedWidget->addWidget(plotStyleWidget);
	pagesStackedWidget->addWidget(plotOthersWidget);

	contentsListWidget = new QListWidget;
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		QWidget *widget = pagesStackedWidget->widget(i);
		contentsListWidget->addItem(widget->windowTitle());
		contentsListWidget->item(i)->setWhatsThis(widget->whatsThis());
	}
	contentsListWidget->setCurrentRow(0);
	connect(contentsListWidget,
			SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
			this, SLOT(changePage(QListWidgetItem *, QListWidgetItem *)));

	QSplitter *centerSplitter = new QSplitter;
	centerSplitter->addWidget(contentsListWidget);
	centerSplitter->addWidget(pagesStackedWidget);

	QPushButton *acceptPushButton = new QPushButton;
	acceptPushButton->setText(tr("&OK"));
	acceptPushButton->setAutoDefault(true);
	acceptPushButton->setDefault(true);
	connect(acceptPushButton, SIGNAL(clicked()), this, SLOT(accept()));

	QPushButton *applyPushButton = new QPushButton;
	applyPushButton->setText(tr("&Apply"));
	connect(applyPushButton, SIGNAL(clicked()), this, SLOT(apply()));

	QPushButton *cancelPushButton = new QPushButton;
	cancelPushButton->setText(tr("&Cancel"));
	connect(cancelPushButton, SIGNAL(clicked()), this, SLOT(reject()));

	// ------------------------ create layouts
	// ----------- assemble layouts and widgets

#if QT_VERSION >= 0x040200
	QDialogButtonBox *dialogButtonBox = new QDialogButtonBox;
	dialogButtonBox->addButton(acceptPushButton, 
			QDialogButtonBox::AcceptRole);
	dialogButtonBox->addButton(applyPushButton,
			QDialogButtonBox::ApplyRole);
	dialogButtonBox->addButton(cancelPushButton,
			QDialogButtonBox::RejectRole);
#else
	QHBoxLayout *buttonLayout = new QHBoxLayout;
	buttonLayout->addStretch();
	buttonLayout->addWidget(acceptPushButton);
	buttonLayout->addWidget(applyPushButton);
	buttonLayout->addWidget(cancelPushButton);
#endif

	QVBoxLayout *mainLayout = new QVBoxLayout;
	mainLayout->addWidget(centerSplitter);
#if QT_VERSION >= 0x040200
	mainLayout->addWidget(dialogButtonBox);
#else
	mainLayout->addLayout(buttonLayout);
#endif

	setLayout(mainLayout);
}

void PlotOptionDialog::setPlot(const Plot &plot)
{
	plot_ = plot;
	bool isFunction = plot.isFunction();
	QString text;
	if (isFunction) {
		text = plot.function();
	} else {
		text = plot.dataFile();
		// ホームディレクトリを"~"で表示する
		text.replace(
			QRegExp("^" + QRegExp::escape(
#if QT_VERSION >= 0x040200
				QDir::toNativeSeparators(QDir::homePath())
#else
				QDir::convertSeparators(QDir::homePath())

#endif
			)), "~");
	}
	setWindowTitle(tr("Options of %1 %2")
			.arg(isFunction ? tr("Function") : tr("Data File"))
			.arg(text)
			+ " - " + QCoreApplication::applicationName());
	plotOthersWidget->setPlotAxis(plot.axis());
	plotTitleWidget->setPlotTitle(plot.titleSpecification());
	plotStyleWidget->setPlotStyle(plot.style());
	if (plot.isFunction()) {
		plotModifierWidget->setEnabled(false);
		QString title = plotModifierWidget->windowTitle();
		for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
			QListWidgetItem *listWidgetItem = contentsListWidget->item(i);
			if (listWidgetItem->text() == title) {
				listWidgetItem->setFlags(listWidgetItem->flags() 
						& ~Qt::ItemIsEnabled);
				break;
			}
		}
	} else {
		plotModifierWidget->setPlotModifier(plot.dataFileModifier());
	}
}

void PlotOptionDialog::accept()
{
	apply();
	QDialog::accept();
}

void PlotOptionDialog::apply()
{
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		qobject_cast<PlotOptionWidget *>(pagesStackedWidget->widget(i))
			->apply();
	}
	Plot plot = plot_;
	plot.setAxis(plotOthersWidget->plotAxis());
	plot.setTitleSpecification(plotTitleWidget->plotTitle());
	plot.setStyle(plotStyleWidget->plotStyle());
	if (!plot.isFunction()) {
		plot.setDataFileModifier(plotModifierWidget->plotModifier());
	}
	emit optionChanged(*this, plot);
}

void PlotOptionDialog::reject()
{
	discard();
	QDialog::reject();
}

void PlotOptionDialog::changePage(QListWidgetItem *current,
		QListWidgetItem *previous)
{
	if (!current) {
		current = previous;
	}
	pagesStackedWidget->setCurrentIndex(contentsListWidget->row(current));
}

void PlotOptionDialog::closeEvent(QCloseEvent *e)
{
	discard();
	e->accept();
}

void PlotOptionDialog::discard()
{
	for (int i = 0, j = pagesStackedWidget->count(); i < j; i++) {
		qobject_cast<PlotOptionWidget *>(pagesStackedWidget->widget(i))
			->discard();
	}
}
