/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "AxisTics.h"

const int AxisTics::AUTOFREQ = 0;
const int AxisTics::INCR = 1;
const int AxisTics::POS = 2;

const QString AxisTics::AXIS[] = {
	"x", "y", "z", "x2", "y2", "cb", ""
};

const char * const AxisTics::LOCATION[] = {
	QT_TRANSLATE_NOOP("AxisTics", "border"),
	QT_TRANSLATE_NOOP("AxisTics", "axis"),
	""
};

AxisTics::AxisTics()
{
	for (int i = 0; !AXIS[i].isEmpty(); i++) {
		flagList.append((i < 3) || (i == 5));
		borderList.append(true);
		mirrorList.append((i < 3) || (i == 5));
		rotateList.append(false);
		angList.append("");
		modeList.append(AUTOFREQ);
		for (int j = 0; j < 3; j++) {
			incrList[j].append("");
		}
		posList.append(QStringList());
	}
}

void AxisTics::setFlag(int axis, bool flag)
{
	flagList[axis] = flag;
}

bool AxisTics::flag(int axis) const
{
	return flagList.at(axis);
}

void AxisTics::setMirror(int axis, bool mirror)
{
	mirrorList[axis] = mirror;
}

bool AxisTics::mirrors(int axis) const
{
	return mirrorList.at(axis);
}

void AxisTics::setBorder(int axis, bool border)
{
	borderList[axis] = border;
}

bool AxisTics::isBorder(int axis) const
{
	return borderList.at(axis);
}

void AxisTics::setRotate(int axis, bool rotate, const QString &ang)
{
	rotateList[axis] = rotate;
	if (rotate) {
		angList[axis] = ang;
	}
}

bool AxisTics::rotates(int axis) const
{
	return rotateList[axis];
}

void AxisTics::setAutofreq(int axis)
{
	modeList[axis] = AUTOFREQ;
}

bool AxisTics::isAutofreq(int axis) const
{
	return modeList.at(axis) == AUTOFREQ;
}

void AxisTics::setIncrement(int axis, const QString &incr, 
		const QString &start, const QString &end)
{
	modeList[axis] = INCR;
	incrList[0][axis] = start;
	incrList[1][axis] = incr;
	incrList[2][axis] = end;
}

bool AxisTics::isIncrement(int axis) const
{
	return modeList.at(axis) == INCR;
}

const QString &AxisTics::start(int axis) const
{
	return incrList[0].at(axis);
}

const QString &AxisTics::increment(int axis) const
{
	return incrList[1].at(axis);
}

const QString &AxisTics::end(int axis) const
{
	return incrList[2].at(axis);
}

void AxisTics::setPosition(int axis, const QStringList &pos)
{
	modeList[axis] = POS;
	posList[axis] = pos;
}

bool AxisTics::isPosition(int axis) const
{
	return modeList.at(axis) == POS;
}

const QStringList &AxisTics::position(int axis) const
{
	return posList.at(axis);
}

QString AxisTics::command() const
{
	QStringList commandList;
	for (int i = 0; !AXIS[i].isEmpty(); i++) {
		QString str = "set ";
		// tics | notics
		if (!flagList.at(i)) {
			str += "no" + AXIS[i] + "tics";
			commandList << str;
			continue;
		}
		str += AXIS[i] + "tics ";
		// axis | border
		str += QString(LOCATION[!borderList.at(i)]) + " ";
		// mirror | nomirror
		str += mirrorList.at(i) ? "mirror " : "nomirror "; 
		// rotate
		if (rotateList.at(i)) {
			str += "rotate ";
			if (!angList.at(i).isEmpty()) {
				// rotate by <ang>
				str += "by " + angList.at(i) + " ";
			}
		} else {
			// norotate
			str += "norotate ";
		}
		if (modeList.at(i) == AUTOFREQ) {
			// autofreq
			str += "autofreq ";
		} else if (modeList.at(i) == INCR) {
			// <start>, <incr>, <end>
			if (!incrList[0].at(i).isEmpty()) {
				str += incrList[0].at(i) + ", ";
			}
			str += incrList[1].at(i);
			if (!incrList[2].at(i).isEmpty()) {
				str += ", " + incrList[2].at(i);
			}
			str += " ";
		} else if (modeList.at(i) == POS) {
			// "<label>" <pos> <level>...
			str += "(" + posList.at(i).join(",") + ")";
		}
		commandList << str;
	}
	return commandList.join("\n");
}
