/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2008 Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.cabinet;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cabinet.util.CabinetUtils;
import com.aimluck.eip.cayenne.om.portlet.EipTCabinetFile;
import com.aimluck.eip.cayenne.om.portlet.EipTCabinetFolder;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 共有フォルダのフォルダフォームデータを管理するクラス <BR>
 * 
 */
public class CabinetFolderFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(CabinetFolderFormData.class.getName());

  /** フォルダ名 */
  private ALStringField folder_name;

  /** 親フォルダ ID */
  private ALNumberField parent_id;

  /** メモ */
  private ALStringField note;

  /** フォルダ情報一覧 */
  private ArrayList folder_hierarchy_list;

  /** 選択されたフォルダ情報 */
  private FolderInfo selected_folderinfo = null;

  /** 親フォルダ情報 */
  private FolderInfo parent_folderinfo = null;

  private String folderid = null;

  private DataContext dataContext;

  private String org_id;

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    // 自ポートレットからのリクエストであれば、パラメータを展開しセッションに保存する。
    if (ALEipUtils.isMatch(rundata, context)) {
      // ENTITY ID
      if (rundata.getParameters().containsKey(CabinetUtils.KEY_FOLDER_ID)) {
        ALEipUtils.setTemp(rundata, context, CabinetUtils.KEY_FOLDER_ID,
            rundata.getParameters().getString(CabinetUtils.KEY_FOLDER_ID));
      }
    }

    org_id = DatabaseOrmService.getInstance().getOrgId(rundata);
    dataContext = DatabaseOrmService.getInstance().getDataContext();

    String tmpfid = ALEipUtils.getTemp(rundata, context,
        CabinetUtils.KEY_FOLDER_ID);
    int fid = CabinetUtils.ROOT_FODLER_ID;
    if (tmpfid != null && !"".equals(tmpfid)) {
      try {
        fid = Integer.parseInt(tmpfid);
      } catch (Exception e) {
        fid = CabinetUtils.ROOT_FODLER_ID;
      }
    } else {
      fid = CabinetUtils.ROOT_FODLER_ID;
    }
    FolderInfo rootFolderInfo = null;
    folder_hierarchy_list = CabinetUtils.getFolderList();
    if (folder_hierarchy_list != null && folder_hierarchy_list.size() > 0) {
      int size = folder_hierarchy_list.size();
      for (int i = 0; i < size; i++) {
        FolderInfo info = (FolderInfo) folder_hierarchy_list.get(i);
        info.setVisible(true);
        if (info.getFolderId() == 1) {
          rootFolderInfo = info;
        }
        if (info.getFolderId() == fid) {
          selected_folderinfo = info;
        }
      }
    }

    if (selected_folderinfo == null) {
      selected_folderinfo = rootFolderInfo;
    }

    super.init(action, rundata, context);
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean setFormData(RunData rundata, Context context,
      ArrayList msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      if (!rundata.getParameters().containsKey("parent_id")) {
        parent_id.setValue(selected_folderinfo.getFolderId());
      }

      if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        folderid = ALEipUtils.getTemp(rundata, context,
            CabinetUtils.KEY_FOLDER_ID);
      }
    }
    return res;
  }

  /**
   * 各フィールドを初期化します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  public void initField() {
    // フォルダ名
    folder_name = new ALStringField();
    folder_name.setFieldName("フォルダ名");
    folder_name.setTrim(true);
    // 親フォルダ
    parent_id = new ALNumberField();
    parent_id.setFieldName("追加位置");
    parent_id.setValue(0);
    // メモ
    note = new ALStringField();
    note.setFieldName("メモ");
    note.setTrim(true);
  }

  /**
   * フォルダの各フィールドに対する制約条件を設定します。 <BR>
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  protected void setValidator() {
    // フォルダ名必須項目
    folder_name.setNotNull(true);
    // フォルダ名の文字数制限
    folder_name.limitMaxLength(128);
    // 追加位置
    parent_id.limitMinValue(0);
    // メモの文字数制限
    note.limitMaxLength(1000);
  }

  /**
   * フォルダのフォームに入力されたデータの妥当性検証を行います。 <BR>
   * 
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  protected boolean validate(ArrayList msgList) {
    // フォルダ名
    folder_name.validate(msgList);
    // 追加位置
    parent_id.validate(msgList);
    // メモ
    note.validate(msgList);

    if (existsFolderName()) {
      msgList.add("このフォルダ名と同じフォルダがすでに存在するため、登録できません。フォルダ名を変更してください。");
    }

    return (msgList.size() == 0);
  }

  private boolean existsFolderName() {
    String fname = folder_name.getValue();
    if (fname == null || "".equals(fname))
      return false;

    try {
      SelectQuery query = new SelectQuery(EipTCabinetFolder.class);
      if (ALEipConstants.MODE_INSERT.equals(getMode())) {
        Expression exp = ExpressionFactory.matchExp(
            EipTCabinetFolder.PARENT_ID_PROPERTY, Integer
                .valueOf((int) parent_id.getValue()));
        query.setQualifier(exp);
      } else if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
        Expression exp1 = ExpressionFactory.matchExp(
            EipTCabinetFolder.PARENT_ID_PROPERTY, Integer
                .valueOf((int) parent_id.getValue()));
        query.setQualifier(exp1);
        Expression exp2 = ExpressionFactory.noMatchDbExp(
            EipTCabinetFolder.FOLDER_ID_PK_COLUMN, Integer.valueOf(folderid));
        query.andQualifier(exp2);
      }

      List list = dataContext.performQuery(query);
      if (list != null && list.size() > 0) {
        EipTCabinetFolder tmpfolder = null;
        int size = list.size();
        for (int i = 0; i < size; i++) {
          tmpfolder = (EipTCabinetFolder) list.get(i);
          if (fname.equals(tmpfolder.getFolderName())) {
            return true;
          }
        }
      }
    } catch (Exception e) {
      return true;
    }

    return false;
  }

  /**
   * フォルダをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  protected boolean loadFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      EipTCabinetFolder folder = CabinetUtils.getEipTCabinetFolder(rundata,
          context);
      if (folder == null)
        return false;

      // フォルダ名
      folder_name.setValue(folder.getFolderName());
      // メモ
      note.setValue(folder.getNote());

      // フォルダ階層の表示切替
      int folder_id = selected_folderinfo.getFolderId();
      ArrayList disable_list = new ArrayList();
      disable_list.add(Integer.valueOf(folder_id));
      int size = folder_hierarchy_list.size();
      for (int i = 0; i < size; i++) {
        FolderInfo info = (FolderInfo) folder_hierarchy_list.get(i);
        boolean flg = false;
        int len = disable_list.size();
        for (int j = 0; j < len; j++) {
          Integer id = (Integer) disable_list.get(j);
          if (info.getFolderId() == id.intValue()
              || info.getParentFolderId() == id.intValue()) {
            flg = true;
            break;
          }
        }
        if (flg) {
          info.setVisible(false);
          disable_list.add(Integer.valueOf(info.getFolderId()));
        } else {
          info.setVisible(true);
        }
      }

      for (int i = 0; i < size; i++) {
        FolderInfo info = (FolderInfo) folder_hierarchy_list.get(i);
        if (info.getFolderId() == selected_folderinfo.getParentFolderId()) {
          // 現在編集中のフォルダの親フォルダを選択させる．
          selected_folderinfo = info;
        }
      }
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * フォルダをデータベースとファイルシステムから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  protected boolean deleteFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      String folderid = ALEipUtils.getTemp(rundata, context,
          CabinetUtils.KEY_FOLDER_ID);
      int delete_id = Integer.parseInt(folderid);
      if (delete_id <= 1) {
        // フォルダ『ルートフォルダ』は削除不可にする．
        return false;
      }
      int delete_parent_id = 0;

      // フォルダ情報を取得する．
      SelectQuery query = new SelectQuery(EipTCabinetFolder.class);
      Expression exp1 = ExpressionFactory.matchDbExp(
          EipTCabinetFolder.FOLDER_ID_PK_COLUMN, Integer.valueOf(folderid));
      Expression exp2 = ExpressionFactory.matchExp(
          EipTCabinetFolder.PARENT_ID_PROPERTY, Integer.valueOf(folderid));
      query.setQualifier(exp1.orExp(exp2));
      List list = dataContext.performQuery(query);
      if (list == null || list.size() <= 0) {
        return false;
      }

      ArrayList folderids = new ArrayList();
      EipTCabinetFolder folder = null;
      int size = list.size();
      for (int i = 0; i < size; i++) {
        folder = (EipTCabinetFolder) list.get(i);
        folderids.add(folder.getFolderId());
        if (folder.getFolderId().intValue() == delete_id) {
          delete_parent_id = folder.getParentId().intValue();
        }
      }

      // データベースのフォルダ情報を削除する．
      SelectQuery delfolderquery = new SelectQuery(EipTCabinetFolder.class);
      Expression delfolderexp = ExpressionFactory.inDbExp(
          EipTCabinetFolder.FOLDER_ID_PK_COLUMN, folderids);
      delfolderquery.setQualifier(delfolderexp);
      List delFolderList = dataContext.performQuery(delfolderquery);

      int delFolderListsize = delFolderList.size();

      // フォルダ情報を削除
      for (int i = 0; i < delFolderListsize; i++) {
        EipTCabinetFolder delfolder = (EipTCabinetFolder) delFolderList.get(i);

        // entityIdを取得
        Integer entityId = delfolder.getFolderId();
        // フォルダ名を取得
        String folderName = delfolder.getFolderName();

        // List cabinetfiles = delfolder.getEipTCabinetFiles();
        List cabinetfiles = CabinetUtils.getEipTCabinetFileList(delfolder
            .getFolderId().intValue());
        ArrayList fpaths = new ArrayList();
        if ((cabinetfiles != null) && (cabinetfiles.size() > 0)) {
          int tsize = cabinetfiles.size();
          for (int k = 0; k < tsize; k++) {
            fpaths.add(((EipTCabinetFile) cabinetfiles.get(k)).getFilePath());
          }
        }

        // フォルダ情報を削除
        dataContext.deleteObject(delfolder);
        dataContext.commitChanges();

        // ログに保存
        ALEventlogFactoryService.getInstance().getEventlogHandler().log(
            entityId, ALEventlogConstants.PORTLET_TYPE_CABINET_FOLDER,
            folderName);

        if (fpaths.size() > 0) {
          // ローカルファイルに保存されているファイルを削除する．
          File file = null;
          int fsize = fpaths.size();
          for (int k = 0; k < fsize; k++) {
            file = new File(CabinetUtils.getSaveDirPath(org_id)
                + (String) fpaths.get(k));
            if (file.exists()) {
              file.delete();
            }
          }
        }
      }

      // 論理フォルダ階層をリロードする．
      folder_hierarchy_list = CabinetUtils.getFolderList();
      selected_folderinfo = CabinetUtils.getSelectedFolderInfo(
          folder_hierarchy_list, delete_parent_id);
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * フォルダをデータベースとファイルシステムに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean insertFormData(RunData rundata, Context context,
      ArrayList msgList) {
    boolean res = false;
    try {
      int uid = ALEipUtils.getUserId(rundata);

      // 新規オブジェクトモデル
      EipTCabinetFolder folder = (EipTCabinetFolder) dataContext
          .createAndRegisterNewObject(EipTCabinetFolder.class);
      // 親フォルダ ID
      folder.setParentId(Integer.valueOf((int) parent_id.getValue()));
      // フォルダ名
      folder.setFolderName(folder_name.getValue());
      // メモ
      folder.setNote(note.getValue());
      // 作成者
      folder.setCreateUserId(Integer.valueOf(uid));
      // 更新者
      folder.setUpdateUserId(Integer.valueOf(uid));
      // 作成日
      folder.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      folder.setUpdateDate(Calendar.getInstance().getTime());
      // フォルダを登録
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          folder.getFolderId(),
          ALEventlogConstants.PORTLET_TYPE_CABINET_FOLDER,
          folder_name.getValue());

      int size = folder_hierarchy_list.size();
      for (int i = 0; i < size; i++) {
        FolderInfo info = (FolderInfo) folder_hierarchy_list.get(i);
        if (info.getFolderId() == parent_id.getValue()) {
          parent_folderinfo = info;
          break;
        }
      }

      selected_folderinfo.setFolderId(folder.getFolderId().intValue());
      selected_folderinfo.setFolderName(folder.getFolderName());
      selected_folderinfo.setHierarchyIndex(parent_folderinfo
          .getHierarchyIndex() + 1);
      selected_folderinfo.setParentFolderId(parent_folderinfo.getFolderId());

      res = true;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return res;
  }

  /**
   * データベースとファイルシステムに格納されているフォルダを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗O
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context, java.util.ArrayList)
   */
  protected boolean updateFormData(RunData rundata, Context context,
      ArrayList msgList) {
    try {
      // オブジェクトモデルを取得
      EipTCabinetFolder folder = CabinetUtils.getEipTCabinetFolder(rundata,
          context);
      if (folder == null)
        return false;

      int uid = ALEipUtils.getUserId(rundata);

      // フォルダ名
      folder.setFolderName(folder_name.getValue());
      // 親フォルダ ID
      folder.setParentId(Integer.valueOf((int) parent_id.getValue()));
      // メモ
      folder.setNote(note.getValue());
      // 更新者
      folder.setUpdateUserId(Integer.valueOf(uid));
      // 更新日
      folder.setUpdateDate(Calendar.getInstance().getTime());
      // フォルダを更新
      dataContext.commitChanges();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
          folder.getFolderId(),
          ALEventlogConstants.PORTLET_TYPE_CABINET_FOLDER,
          folder_name.getValue());
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return false;
    }
    return true;
  }

  /**
   * フォルダ名を取得する． <BR>
   * 
   * @return
   */
  public ALStringField getFolderName() {
    return folder_name;
  }

  public ALNumberField getParentId() {
    return parent_id;
  }

  /**
   * メモを取得する． <BR>
   * 
   * @return
   */
  public ALStringField getNote() {
    return note;
  }

  public ArrayList getFolderHierarchyList() {
    return folder_hierarchy_list;
  }

  public FolderInfo getSelectedFolderInfo() {
    return selected_folderinfo;
  }

  /**
   * 
   * @param id
   * @return
   */
  public boolean isMatch(int id1, long id2) {
    return id1 == (int) id2;
  }

  /**
   * アクセス権限チェック用メソッド。<br />
   * アクセス権限の機能名を返します。
   * 
   * @return
   */
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_CABINET_FOLDER;
  }
}
