/*
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.enquete;

import java.util.List;
import java.util.jar.Attributes;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnquete;
import com.aimluck.eip.cayenne.om.portlet.AvzTEnqueteChoiceItem;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.enquete.util.EnqueteUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.util.ALCommonUtils;
import com.aimluck.eip.util.ALEipUtils;

/**
 * Enquete検索データを管理するクラスです。 <BR>
 * 
 */
public class EnqueteListSelectData extends ALAbstractSelectData implements ALData {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(EnqueteSelectData.class.getName());

  /** 現在選択されているタブ */
  private String currentTab;

  /** Enquete の総数 */
  private int enqueteSum;
  

  private int loginuserid;

  /** 回答状態（未回答） */
  public static final String RESPONSE_STATUS_BEFORE = "0";
  
  /** 回答状態（回答済） */
  public static final String RESPONSE_STATUS_AFTER = "1";
  
  /** 公開締切状態（未公開） */
  public static final String PUBLIC_STATUS_BEFORE = "0";

  /** 公開締切状態（実施中） */
  public static final String PUBLIC_STATUS_PRACTICE = "1";

  /** 公開締切状態（締切済） */
  public static final String PUBLIC_STATUS_AFTER = "2";
  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractSelectData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    super.init(action, rundata, context);
    String sort = ALEipUtils.getTemp(rundata, context, LIST_SORT_STR);
    if (sort == null || sort.equals("")) {
      ALEipUtils.setTemp(rundata, context, LIST_SORT_STR, ALEipUtils
          .getPortlet(rundata, context).getPortletConfig().getInitParameter(
              "p2a-sort"));
      logger.debug("[EnqueteSelectData] Init Parameter. : "
          + ALEipUtils.getPortlet(rundata, context).getPortletConfig()
              .getInitParameter("p2a-sort"));
    }
    String tabParam = rundata.getParameters().getString("tab");
    currentTab = ALEipUtils.getTemp(rundata, context, "tab");
    if (tabParam == null && currentTab == null) {
      ALEipUtils.setTemp(rundata, context, "tab", "list");
      currentTab = "list";
    } else if (tabParam != null) {
      ALEipUtils.setTemp(rundata, context, "tab", tabParam);
      currentTab = tabParam;
    }
    
    loginuserid = Integer.valueOf(ALEipUtils.getUserId(rundata));

  }


  /**
   * 一覧データを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   * @see com.aimluck.eip.common.ALAbstractListData#selectData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  public List selectList(RunData rundata, Context context) {
    try {
      
      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();

      SelectQuery query = getSelectQuery(rundata, context);
      buildSelectQueryForListView(query);
      buildSelectQueryForListViewSort(query, rundata, context);


      List list = dataContext.performQuery(query);      
      
      EnqueteUtils.setPublicStatus(dataContext, list);

      // Enquete の総数をセットする．
      enqueteSum = list.size();
      return buildPaginatedList(list);

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery getSelectQuery(RunData rundata, Context context) {

    SelectQuery query = new SelectQuery(AvzTEnquete.class);
  
    //回答ユーザーとログインユーザーが一致するデータを取得
    Expression exp = ExpressionFactory.matchExp(
        AvzTEnquete.AVZ_TENQUETE_CHOICE_ITEM_PROPERTY + "."
        + AvzTEnqueteChoiceItem.ANSWER_ID_PROPERTY, Integer.valueOf(ALEipUtils.getUserId(rundata)));
  
    //未回答のアンケートを取得
    Expression exp2 = ExpressionFactory.matchExp(
        AvzTEnquete.AVZ_TENQUETE_CHOICE_ITEM_PROPERTY + "."
        + AvzTEnqueteChoiceItem.RESPONSE_STATUS_PROPERTY, RESPONSE_STATUS_BEFORE);
  
    //回答済みのアンケートを取得
    Expression exp3 = ExpressionFactory.matchExp(
        AvzTEnquete.AVZ_TENQUETE_CHOICE_ITEM_PROPERTY + "."
        + AvzTEnqueteChoiceItem.RESPONSE_STATUS_PROPERTY, RESPONSE_STATUS_AFTER);
  
    //作成ユーザーとログインユーザーが一致するデータを取得
    Expression exp4 = ExpressionFactory.matchExp(AvzTEnquete.CREATER_ID_PROPERTY,
        Integer.valueOf(ALEipUtils.getUserId(rundata)));
    
    //「未公開」以外のアンケートを取得
    Expression exp5 = ExpressionFactory.noMatchExp(
        AvzTEnquete.PUBLIC_STATUS_PROPERTY , PUBLIC_STATUS_BEFORE);
    
    if("list".equals(currentTab)){
      query.setQualifier((exp.andExp(exp2).andExp(exp5)));
    }else if ("complete".equals(currentTab)){
      query.setQualifier((exp.andExp(exp3).andExp(exp5)));
    }else if ("create".equals(currentTab)){
      query.setQualifier(exp4);
    }
    query.setDistinct(true);

    return buildSelectQueryForFilter(query, rundata, context);
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  protected Object getResultData(Object obj) {
    try {

      AvzTEnquete record = (AvzTEnquete) obj;
      EnqueteResultData rd = new EnqueteResultData();
      
      rd.initField();
      rd.setEnqueteId(record.getEnqueteId().intValue());
      rd.setCreaterId(record.getCreaterId().intValue());
      rd.setName(ALCommonUtils.compressString(record.getName(),getStrLength()));
      rd.setPublicStatus(record.getPublicStatus());
      rd.setDeadlineDate(record.getDeadlineDate());
      rd.setDeadlineMethod(record.getDeadlineMethod());
      ALStringField creatername = new ALStringField();
      creatername.setValue(ALEipUtils.getUserFullName(record.getCreaterId()));
      rd.setCreaterName(creatername);

      return rd;
      
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 詳細データを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#selectDetail(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  public Object selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException {
    return null;
  }

  /**
   * ResultData に値を格納して返します。（詳細データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getResultDataDetail(java.lang.Object)
   */
  protected Object getResultDataDetail(Object obj) {
    return null;
  }


  /**
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getColumnMap()
   */
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    map.putValue("deadline_date", AvzTEnquete.DEADLINE_DATE_PROPERTY);
    return map;
  }

  public int getUserId() {
    return loginuserid;
  }

  public String getCurrentTab() {
    return currentTab;
  }
  
  /**
   * Enquete の総数を返す． <BR>
   * 
   * @return
   */
  public int getEnqueteSum() {
    return enqueteSum;
  }

}
