/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.util.template;

// Java imports

// Junit imports
import junit.awtui.TestRunner;
import junit.framework.Test;
import junit.framework.TestSuite;

// Cactus imports
import org.apache.cactus.ServletTestCase;
import org.apache.cactus.WebRequest;

// Jetspeed imports
import org.apache.jetspeed.om.profile.ProfileLocator;
import org.apache.jetspeed.services.Profiler;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.util.template.JetspeedLink;
import org.apache.jetspeed.util.template.JetspeedLinkFactory;
import org.apache.jetspeed.test.TurbineTestUtilities;

// Turbine imports
import org.apache.turbine.services.pull.TurbinePull;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.RunDataFactory;

// Velocity
import org.apache.velocity.context.Context;
/**
 * TestJespeedLink
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestJetspeedLink.java,v 1.1 2004/04/07 22:02:42 jford Exp $
 */
public class TestJetspeedLink extends ServletTestCase
{
    private static String TEST_CONTEXT = "/test";
    private static String TEST_HOST = "localhost";
    private static String TEST_SERVLET = "/portal";
    
    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestJetspeedLink(String name)
    {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[] { TestJetspeedLink.class.getName() } );
    }
    
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestJetspeedLink.class );
    }
    
    
    public void beginBaseLink(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET, "", null);
        System.out.println("post set URL = " + theRequest.getURL());
    }
    
    /**
     * Test the base link
     * @throws Exception
     */
    public void testBaseLink() throws Exception
    {
        ProfileLocator profileLocator = null;

        // Create the RunData object to be used during testing.
        RunData rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);
        
        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);
        
        // Verify tool are in the context
        JetspeedLink jsLink = (JetspeedLink) context.get("jslink");
        assertNotNull( "Got jslink from context", jsLink);
                
        // Generate the link
        String link = jsLink.toString();
        System.out.println("Base link = " + link);
        assertTrue( "Verifing base Link", 
          link.startsWith("http://" + TEST_HOST + TEST_CONTEXT + TEST_SERVLET));

        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);
        
        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    
    public void endBaseLink(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text length = " + theResponse.getText().length());
    }
    
    public void beginGroupLink(WebRequest theRequest)
    {
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/" + Profiler.PARAM_GROUP + "/apache" , null);
        System.out.println("post set URL = " + theRequest.getURL());
    }
    
    /**
     * Test the Group link
     * @throws Exception
     */
    public void testGroupLink() throws Exception
    {
        // Create the RunData object to be used during testing.
        RunData rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);
        
        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);
        
        // Verify tool are in the context
        JetspeedLink jsLink = (JetspeedLink) context.get("jslink");
        assertNotNull( "Got jslink from context", jsLink);
        
        // Generate the link
        String link = jsLink.toString();
        System.out.println("Group link = " + link);
        assertTrue( "Verifing base Link", 
          link.startsWith("http://" + TEST_HOST + TEST_CONTEXT + TEST_SERVLET));

        assertTrue( "Verify link contains /" + Profiler.PARAM_GROUP + "/apache", 
          (link.indexOf("/" + Profiler.PARAM_GROUP + "/apache") >0));

        assertEquals( "Verify link does NOT contain /" + Profiler.PARAM_USER, -1,
           link.indexOf("/" + Profiler.PARAM_USER ));

        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);
        
        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
        
    }
    
    public void endGroupLink(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text = " + theResponse.getText().length());
    }
    
    public void beginPageLink(WebRequest theRequest)
    {
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/" + Profiler.PARAM_PAGE + "/news" , null);
        System.out.println("post set URL = " + theRequest.getURL());
    }
    
    /**
     * Test the User link
     * @throws Exception
     */
    public void testPageLink() throws Exception
    {
        // Create the RunData object to be used during testing.
        RunData rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);
        
        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);
        
        // Verify tool are in the context
        JetspeedLink jsLink = (JetspeedLink) context.get("jslink");
        assertNotNull( "Got jslink from context", jsLink);
        
        // Generate the link
        String link = jsLink.toString();
        System.out.println("Page Link = " + link);
        assertTrue( "Verifing base Link", 
          link.startsWith("http://" + TEST_HOST + TEST_CONTEXT + TEST_SERVLET));

        assertTrue( "Verify link contains /" + Profiler.PARAM_USER + "/anon", 
          (link.indexOf("/" + Profiler.PARAM_USER + "/anon") >0));

        assertTrue( "Verify link contains /" + Profiler.PARAM_PAGE + "/news", 
          (link.indexOf("/" + Profiler.PARAM_PAGE + "/news") >0));

        assertEquals( "Verify link does NOT contain /" + Profiler.PARAM_GROUP, -1,
           link.indexOf("/" + Profiler.PARAM_GROUP ));

        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);
        
        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    public void endPageLink(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text = " + theResponse.getText().length());
    }
    
    public void beginGroupPage(WebRequest theRequest)
    {
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/" + Profiler.PARAM_GROUP + "/apache/" + Profiler.PARAM_PAGE + "/news" , null);
        System.out.println("post set URL = " + theRequest.getURL());
    }
    
    /**
     * Test the User link
     * @throws Exception
     */
    public void testGroupPage() throws Exception
    {
        // Create the RunData object to be used during testing.
        RunData rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);
        
        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);
        
        // Verify tool are in the context
        JetspeedLink jsLink = (JetspeedLink) context.get("jslink");
        assertNotNull( "Got jslink from context", jsLink);
        
        // Generate the link
        String link = jsLink.toString();
        System.out.println("Group/Page Link = " + link);
        assertTrue( "Verifing base Link", 
          link.startsWith("http://" + TEST_HOST + TEST_CONTEXT + TEST_SERVLET));

        assertTrue( "Verify link contains /" + Profiler.PARAM_GROUP + "/apache", 
          (link.indexOf("/" + Profiler.PARAM_GROUP + "/apache") >0));

        assertEquals( "Verify link does NOT contain /" + Profiler.PARAM_USER, -1,
           link.indexOf("/" + Profiler.PARAM_USER ));

        assertTrue( "Verify link contains /" + Profiler.PARAM_PAGE + "/news", 
          (link.indexOf("/" + Profiler.PARAM_PAGE + "/news") >0));
        
        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);
        
        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    public void endGroupPage(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text = " + theResponse.getText().length());
    }

    public void beginGroupPage2(WebRequest theRequest)
    {
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/" + Profiler.PARAM_GROUP + "/apache/" + Profiler.PARAM_PAGE + "/news" , null);
        System.out.println("post set URL = " + theRequest.getURL());
    }
    
    /**
     * Test the User link
     * @throws Exception
     */
    public void testGroupPage2() throws Exception
    {
        JetspeedLink jetspeedLink = null;
        
        // Create the RunData object to be used during testing.
        RunData rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);
        
        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);
        
        // Verify tool are in the context
        JetspeedLink jsLink = (JetspeedLink) context.get("jslink");
        assertNotNull( "Got jslink from context", jsLink);
        
        // Generate the link
        String link = jsLink.toString();
        System.out.println("Group/Page Link = " + link);
        
        // Generate the link from JetspeedLink
        jetspeedLink = JetspeedLinkFactory.getInstance(rundata);
        System.out.println("Group/Page jetspeedLink = " + jetspeedLink.toString());
        assertEquals( "jsLink = jetspeedLink", link, jetspeedLink.toString());
        JetspeedLinkFactory.putInstance(jsLink);
        
        // Make sure 2 calls to toString() return the same value
        jetspeedLink = JetspeedLinkFactory.getInstance(rundata);
        assertEquals( "jsLink = jetspeedLink", jetspeedLink.toString(), jetspeedLink.toString());
        JetspeedLinkFactory.putInstance(jsLink);

        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);
        
        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    public void endGroupPage2(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text = " + theResponse.getText().length());
    }
    
    
    public void beginGetLink(WebRequest theRequest)
    {
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
        , "/" + Profiler.PARAM_GROUP + "/apache/" + Profiler.PARAM_PAGE + "/news/" + JetspeedResources.PATH_PORTLETID_KEY + "/p_1234/" + JetspeedResources.PATH_TEMPLATE_KEY + "/Home/" + Profiler.PARAM_MEDIA_TYPE + "/html/" + Profiler.PARAM_LANGUAGE + "/en/" + Profiler.PARAM_COUNTRY + "/US" , null);
        System.out.println("post set URL = " + theRequest.getURL());
    }
    
    /**
     * Test the User link
     * @throws Exception
     */
    public void testGetLink() throws Exception
    {
        JetspeedLink jetspeedLink = null;
        
        // Create the RunData object to be used during testing.
        RunData rundata = RunDataFactory.getRunData( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);
        
        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);
        
        // Verify tool are in the context
        JetspeedLink jsLink = (JetspeedLink) context.get("jslink");
        assertNotNull( "Got jslink from context", jsLink);
        
        // Generate the link from JetspeedLink
        jetspeedLink = JetspeedLinkFactory.getInstance(rundata);
        String baseLink = new String("http://" + TEST_HOST + TEST_CONTEXT + TEST_SERVLET);
        String currentLink = new String( baseLink 
            + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType() 
            + "/" + Profiler.PARAM_GROUP + "/" + jetspeedLink.getGroupName() 
            + "/" + Profiler.PARAM_PAGE + "/" + jetspeedLink.getPageName());
 
        assertEquals( "Testing getGroupName()", "apache" , jetspeedLink.getGroupName() );
        assertEquals( "Testing getPageName()", "news.psml" , jetspeedLink.getPageName() );
        assertEquals( "Testing getPageName()", "html" , jetspeedLink.getMediaType() );
        assertNull( "Testing getRoleName()", jetspeedLink.getRoleName() );
        assertNull( "Testing getUserName()", jetspeedLink.getUserName() );

        // Test conveniance methods
        assertEquals( "Testing getAction(actionName)", currentLink + "?" + JetspeedResources.PATH_ACTION_KEY + "=myAction", jetspeedLink.getAction("myAction").toString());
        assertEquals( "Testing getAction(actionName, peid)", currentLink + "/" + JetspeedResources.PATH_PORTLETID_KEY + "/actionpeid?" + JetspeedResources.PATH_ACTION_KEY + "=myAction1", jetspeedLink.getAction("myAction1","actionpeid").toString() );
        assertEquals( "Testing getGroup(groupName)", baseLink + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType()+ "/" + Profiler.PARAM_GROUP + "/myGroup", jetspeedLink.getGroup("myGroup").toString());
        assertEquals( "Testing getGroup(groupName,pageName)", baseLink + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType() + "/" + Profiler.PARAM_GROUP + "/myGroup1/" + Profiler.PARAM_PAGE + "/myGroupPage", jetspeedLink.getGroup("myGroup1","myGroupPage").toString());
        assertEquals( "Testing getHomePage()", baseLink , jetspeedLink.getHomePage().toString() );
        assertEquals( "Testing getRole(roleName)", baseLink + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType() + "/" + Profiler.PARAM_ROLE + "/myRole", jetspeedLink.getRole("myRole").toString());
        assertEquals( "Testing getRole(roleName,pageName)", baseLink + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType()+ "/" + Profiler.PARAM_ROLE + "/myRole1/" + Profiler.PARAM_PAGE + "/myRolePage", jetspeedLink.getRole("myRole1","myRolePage").toString());
        assertEquals( "Testing getTemplate(templateName)", currentLink  + "/" + JetspeedResources.PATH_TEMPLATE_KEY + "/myTemplate", jetspeedLink.getTemplate("myTemplate").toString());
        assertEquals( "Testing getTemplate()", currentLink , jetspeedLink.getTemplate().toString() );
        assertEquals( "Testing getUser(userName)", baseLink + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType() + "/" + Profiler.PARAM_USER + "/myUser", jetspeedLink.getUser("myUser").toString());
        assertEquals( "Testing getUser(userName,PageName)", baseLink + "/" + Profiler.PARAM_MEDIA_TYPE + "/" + jetspeedLink.getMediaType() + "/" + Profiler.PARAM_USER + "/myUser1/" + Profiler.PARAM_PAGE + "/myUserPage", jetspeedLink.getUser("myUser1","myUserPage").toString());

        // Test getLink()
        assertEquals( "Testing getLink() with default & nulls",
            baseLink + 
            "/" + Profiler.PARAM_MEDIA_TYPE +
            "/" + jetspeedLink.getMediaType() + 
            "/" + Profiler.PARAM_PAGE + 
            "/" + jetspeedLink.getPageName(),
            jetspeedLink.getLink(JetspeedLink.DEFAULT, null, null, JetspeedLink.DEFAULT, null, null).toString() );

        assertEquals( "Testing getLink() with current & nulls", 
            baseLink + 
            "/" + Profiler.PARAM_MEDIA_TYPE + 
            "/" + jetspeedLink.getMediaType()  + 
            "/" + Profiler.PARAM_GROUP + 
            "/" + jetspeedLink.getGroupName()+ 
            "/" + Profiler.PARAM_PAGE + 
            "/" + jetspeedLink.getPageName(),
            jetspeedLink.getLink(JetspeedLink.CURRENT, null, null, JetspeedLink.CURRENT, null, null).toString() );

        assertEquals( "Testing getLink() with default & nulls + media/language/country", 
            baseLink + 
            "/" + Profiler.PARAM_PAGE + 
            "/" + jetspeedLink.getPageName() + 
            "/" + Profiler.PARAM_MEDIA_TYPE + 
            "/" + "test_html"+ 
            "/" + Profiler.PARAM_LANGUAGE + 
            "/" + "test_en" + 
            "/" + Profiler.PARAM_COUNTRY + 
            "/" + "test_US", 
            jetspeedLink.getLink(JetspeedLink.DEFAULT, null, null, JetspeedLink.DEFAULT, null, null, null, "test_html", "test_en", "test_US").toString() );

        JetspeedLinkFactory.putInstance(jsLink);
        
        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);
        
        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    public void endGetLink(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text = " + theResponse.getText().length());
    }
}
